/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.profile.context.navigate.InboundMessageContextLookup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.openid.connect.sdk.rp.OIDCClientMetadata;

import net.minidev.json.JSONObject;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCClientRegistrationMetadataPolicyContext;
import net.shibboleth.oidc.metadata.policy.MetadataPolicy;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Adds the remaining claims from the input metadata to the output {@link OIDCClientMetadata}, which haven't been
 * added there already, but which are recognized in the metadata policy.
 */
public class AddRemainingClaimsToClientMetadata extends AbstractOIDCClientMetadataPopulationAction {

    /** Class logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(AddRemainingClaimsToClientMetadata.class);

    /** Strategy that will return {@link OIDCClientRegistrationMetadataPolicyContext}. */
    @Nonnull private Function<ProfileRequestContext, OIDCClientRegistrationMetadataPolicyContext>
        registrationMetadataPolicyContextLookupStrategy;

    /** The metadata policy used for finding out remaining claims. */
    @Nullable private Map<String, MetadataPolicy> metadataPolicy;

    /**
     * Constructor.
     */
    public AddRemainingClaimsToClientMetadata() {
        registrationMetadataPolicyContextLookupStrategy = 
                new ChildContextLookup<>(OIDCClientRegistrationMetadataPolicyContext.class, false).compose(
                        new InboundMessageContextLookup());
    }
    
    /**
     * Set the strategy that will return {@link OIDCClientRegistrationMetadataPolicyContext}.
     * 
     * @param strategy Strategy that will return {@link OIDCClientRegistrationMetadataPolicyContext}.
     */
    public void setRegistrationMetadataPolicyContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, OIDCClientRegistrationMetadataPolicyContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        registrationMetadataPolicyContextLookupStrategy = Constraint.isNotNull(strategy,
                "Registration metadata policy context lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        final OIDCClientRegistrationMetadataPolicyContext metadataPolicyContext =
                registrationMetadataPolicyContextLookupStrategy.apply(profileRequestContext);
        if (metadataPolicyContext == null) {
            log.debug("{} No OIDCClientRegistrationMetadataPolicyContext found, nothing to do", getLogPrefix());
            return false;
        }
        
        metadataPolicy = metadataPolicyContext.getMetadataPolicy();
        if (metadataPolicy == null || metadataPolicy.isEmpty()) {
            log.debug("{} No metadata policy found, nothing to do", getLogPrefix());
            return false;
        }
        
        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        final JSONObject input = getInputMetadata().toJSONObject();
        final JSONObject initialOutput = getOutputMetadata().toJSONObject();
        for (final String claim : input.keySet()) {
            if (!initialOutput.containsKey(claim)) {
                if (metadataPolicy.containsKey(claim)) {
                    log.debug("{} added {} to the output metadata", getLogPrefix(), claim);
                    getOutputMetadata().setCustomField(claim, input.get(claim));
                } else {
                    log.warn("{} input contained claim {} that is not recognized and is thus ignored", getLogPrefix(),
                            claim);
                }
            }
        }
    }
}
