/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.encoding.impl;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletResponse;

import com.google.common.base.MoreObjects;
import com.nimbusds.oauth2.sdk.AccessTokenResponse;
import com.nimbusds.oauth2.sdk.ErrorObject;
import com.nimbusds.oauth2.sdk.ErrorResponse;
import com.nimbusds.oauth2.sdk.Response;
import com.nimbusds.oauth2.sdk.TokenErrorResponse;
import com.nimbusds.oauth2.sdk.TokenIntrospectionErrorResponse;
import com.nimbusds.oauth2.sdk.TokenIntrospectionResponse;
import com.nimbusds.oauth2.sdk.TokenIntrospectionSuccessResponse;
import com.nimbusds.oauth2.sdk.TokenResponse;
import com.nimbusds.oauth2.sdk.client.ClientInformation;
import com.nimbusds.oauth2.sdk.client.ClientInformationResponse;
import com.nimbusds.oauth2.sdk.client.ClientRegistrationErrorResponse;
import com.nimbusds.oauth2.sdk.http.HTTPResponse;
import com.nimbusds.oauth2.sdk.token.Tokens;
import com.nimbusds.openid.connect.sdk.AuthenticationErrorResponse;
import com.nimbusds.openid.connect.sdk.AuthenticationResponse;
import com.nimbusds.openid.connect.sdk.AuthenticationSuccessResponse;
import com.nimbusds.openid.connect.sdk.UserInfoErrorResponse;
import com.nimbusds.openid.connect.sdk.UserInfoResponse;
import com.nimbusds.openid.connect.sdk.UserInfoSuccessResponse;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientInformationResponse;
import com.nimbusds.openid.connect.sdk.token.OIDCTokens;

import net.shibboleth.idp.plugin.oidc.op.decoding.impl.RequestUtil;
import net.shibboleth.idp.plugin.oidc.op.messaging.JSONSuccessResponse;
import net.shibboleth.idp.plugin.oidc.op.oauth2.messaging.impl.OAuth2RevocationErrorResponse;
import net.shibboleth.idp.plugin.oidc.op.oauth2.messaging.impl.OAuth2RevocationSuccessResponse;

/** Response logging helper class. */
public final class ResponseUtil {
    
    /** Private constructor. */
    private ResponseUtil() {
        
    }

    /**
     * Helper method to print response to string for logging.
     * 
     * @param httpResponse response to be printed
     * @return response as formatted string.
     */
    protected static String toString(@Nullable final HTTPResponse httpResponse) {
        if (httpResponse == null) {
            return null;
        }
        final String nl = System.lineSeparator();
        String ret = nl;
        final Map<String, List<String>> headers = httpResponse.getHeaderMap();
        if (headers != null) {
            ret += "Headers:" + nl;
            for (final Entry<String, List<String>> entry : headers.entrySet()) {
                ret += "\t" + entry.getKey() + ":" + entry.getValue().get(0) + nl;
            }
        }
        if (httpResponse.getContent() != null) {
            ret += "Content:" + httpResponse.getContent();
        }
        return ret;
    }

    /**
     * Helper method to print response to string for logging.
     * 
     * @param httpServletResponse response to be printed
     * @param content message content
     * 
     * @return response as formatted string.
     */
    @Nullable protected static String toString(@Nullable final HttpServletResponse httpServletResponse,
            @Nullable final String content) {
        if (httpServletResponse == null) {
            return null;
        }
        final String nl = System.lineSeparator();
        String ret = nl;
        final Collection<String> headerNames = httpServletResponse.getHeaderNames();
        if (headerNames != null) {
            ret += "Headers:" + nl;
            for (final String headerName : headerNames) {
                ret += "\t" + headerName + ":" + httpServletResponse.getHeader(headerName) + nl;
            }
        }
        if (content != null) {
            ret += "Content:" + content;
        }
        return ret;
    }

 // Checkstyle: CyclomaticComplexity|ReturnCount OFF

    /**
     * Helper method for getting protocol message for a Nimbus response object. This method can currently
     * recognize success and error responses for OIDC authentication, token, userinfo, introspetion and
     * revocation.
     * 
     * @param response The response message
     * @return The response message specific log message
     */
    @Nullable public static String getProtocolMessage(@Nullable final Response response) {
        if (response == null) {
            return null;
        }
        if (response instanceof AuthenticationResponse) {
            return getProtocolMessageForAuthenticationResponse((AuthenticationResponse) response);
        } else if (response instanceof OIDCClientInformationResponse) {
            return getProtocolMessageForRegistrationResponse((OIDCClientInformationResponse) response);
        } else if (response instanceof TokenResponse) {
            return getProtocolMessageForTokenResponse((TokenResponse) response);
        } else if (response instanceof UserInfoResponse) {
            return getProtocolMessageForUserInfoResponse((UserInfoResponse) response);
        } else if (response instanceof TokenIntrospectionResponse) {
            return getProtocolMessageForIntrospectionResponse((TokenIntrospectionResponse) response);
        } else if (response instanceof OAuth2RevocationSuccessResponse) {
            return getProtocolMessageForRevocationResponse(response);
        } else if (response instanceof OAuth2RevocationErrorResponse) {
            return getProtocolMessageForRevocationResponse(response);
        } else if (response instanceof JSONSuccessResponse) {
            return getProtocolMessageForJSONSuccessResponse(response);
        } else if (response instanceof ErrorResponse) {
            final ErrorResponse genericError = (ErrorResponse) response;
            return MoreObjects.toStringHelper(genericError).omitNullValues()
                    .add("errorObject", genericError.getErrorObject())
                    .toString();
        }
        return MoreObjects.toStringHelper(response).toString();
    }
 // Checkstyle: CyclomaticComplexity|ReturnCount ON

    /**
     * Helper method for getting protocol message for OIDC authentication response.
     * 
     * @param response The response message
     * @return The response message specific log message
     */
    @Nullable public static String getProtocolMessageForAuthenticationResponse(
            @Nonnull final AuthenticationResponse response) {
        if (response.indicatesSuccess()) {
            final AuthenticationSuccessResponse successResponse = response.toSuccessResponse();
            return MoreObjects.toStringHelper(successResponse).omitNullValues()
                    .add("accessToken", RequestUtil.getAccessTokenLog(successResponse.getAccessToken()))
                    .add("authorizationCode", successResponse.getAuthorizationCode())
                    .add("idToken", successResponse.getIDToken() == null ?
                            null : successResponse.getIDToken().serialize())
                    .add("issuer", successResponse.getIssuer())
                    .add("jwtResponse", successResponse.getJWTResponse() == null ?
                            null : successResponse.getJWTResponse().serialize())
                    .add("redirectionURI", successResponse.getRedirectionURI())
                    .add("responseMode", successResponse.getResponseMode())
                    .add("sessionState", successResponse.getSessionState())
                    .add("state", successResponse.getState())
                    .toString();
        } else {
            final AuthenticationErrorResponse errorResponse = response.toErrorResponse();
            return MoreObjects.toStringHelper(errorResponse).omitNullValues()
                    .add("errorObject", getProtocolMessageForErrorObject(errorResponse.getErrorObject()))
                    .add("issuer", errorResponse.getIssuer())
                    .add("jwtResponse", errorResponse.getJWTResponse() == null ?
                            null : errorResponse.getJWTResponse().serialize())
                    .add("redirectionURI", errorResponse.getRedirectionURI())
                    .add("responseMode", errorResponse.getResponseMode())
                    .add("state", errorResponse.getState())
                    .toString();
        }
    }

    /**
     * Helper method for getting protocol message for OIDC registration response.
     * 
     * @param response The response message
     * @return The response message specific log message
     */
    @Nullable public static String getProtocolMessageForRegistrationResponse(
            @Nonnull final OIDCClientInformationResponse response) {
        if (response.indicatesSuccess()) {
            final ClientInformationResponse successResponse = response.toSuccessResponse();
            return MoreObjects.toStringHelper(successResponse).omitNullValues()
                    .add("clientInformation",
                            getProtocolMessageForClientInformation(successResponse.getClientInformation()))
                    .toString();
        } else {
            final ClientRegistrationErrorResponse errorResponse = response.toErrorResponse();
            return MoreObjects.toStringHelper(errorResponse).omitNullValues()
                    .add("errorObject", getProtocolMessageForErrorObject(errorResponse.getErrorObject()))
                    .toString();
        }
    }

    /**
     * Helper method for getting protocol message for client information object.
     * 
     * @param clientInformation The client information
     * @return The log message
     */
    @Nullable public static String getProtocolMessageForClientInformation(
            @Nullable final ClientInformation clientInformation) {
        return clientInformation == null ? null : MoreObjects.toStringHelper(clientInformation).omitNullValues()
                .add("clientId", clientInformation.getID())
                .add("idIssuedDate", clientInformation.getIDIssueDate())
                .add("metadadata", clientInformation.getMetadata())
                .add("registrationAccessToken",
                        RequestUtil.getAccessTokenLog(clientInformation.getRegistrationAccessToken()))
                .add("registrationURI", clientInformation.getRegistrationURI())
                .add("secret", clientInformation.getSecret() == null ? null : "<secret>")
                .toString();
    }
    
    /**
     * Helper method for getting protocol message for token response.
     * 
     * @param response The response message
     * @return  The response message specific log message
     */
    @Nullable public static String getProtocolMessageForTokenResponse(@Nonnull final TokenResponse response) {
        if (response.indicatesSuccess()) {
            final AccessTokenResponse successResponse = response.toSuccessResponse();
            return MoreObjects.toStringHelper(successResponse).omitNullValues()
                    .add("customParameters", successResponse.getCustomParameters())
                    .add("tokens", successResponse.getTokens())
                    .toString();
        } else {
            final TokenErrorResponse errorResponse = response.toErrorResponse();
            return MoreObjects.toStringHelper(errorResponse).omitNullValues()
                    .add("errorObject", getProtocolMessageForErrorObject(errorResponse.getErrorObject()))
                    .toString();
        }
    }

    /**
     * Helper method for getting protocol message for error object.
     * 
     * @param errorObject The error object
     * @return The log message
     */
    @Nullable public static String getProtocolMessageForErrorObject(@Nullable final ErrorObject errorObject) {
        return errorObject == null ? null : MoreObjects.toStringHelper(errorObject).omitNullValues()
                .add("httpStatusCode", errorObject.getHTTPStatusCode())
                .add("code", errorObject.getCode())
                .add("description", errorObject.getDescription())
                .add("uri", errorObject.getURI())
                .toString();
    }

    /**
     * Helper method for getting protocol message for tokens object.
     * 
     * @param tokens The tokens object
     * @return The log message
     */
    @Nullable public static String getProtocolMessageForTokens(@Nullable final Tokens tokens) {
        final String idToken = tokens instanceof OIDCTokens ? MoreObjects.toStringHelper(OIDCTokens.class)
                .omitNullValues()
                .add("idToken", tokens.toOIDCTokens().getIDToken() == null ? null :
                    tokens.toOIDCTokens().getIDToken().serialize())
                .toString() : null;
        return tokens == null ? null : MoreObjects.toStringHelper(tokens).omitNullValues()
                .add("accessToken", tokens.getAccessToken())
                .add("metadata", tokens.getMetadata())
                .add("parameterNames", tokens.getParameterNames())
                .add("refreshToken", tokens.getRefreshToken())
                .add("idToken", idToken)
                .toString();
    }

    /**
     * Helper method for getting protocol message for OIDC user info response.
     * 
     * @param response The response message
     * @return  The response message specific log message
     */
    @Nullable public static String getProtocolMessageForUserInfoResponse(@Nonnull final UserInfoResponse response) {
        if (response.indicatesSuccess()) {
            final UserInfoSuccessResponse successResponse = response.toSuccessResponse();
            return MoreObjects.toStringHelper(successResponse).omitNullValues()
                    .add("entityContentType", successResponse.getEntityContentType())
                    .add("userInfo", successResponse.getUserInfo() != null ? 
                            successResponse.getUserInfo().toJSONString() : null)
                    .add("userInfoJWT", successResponse.getUserInfoJWT() == null ? null :
                        successResponse.getUserInfoJWT().serialize())
                    .toString();
        } else {
            final UserInfoErrorResponse errorResponse = response.toErrorResponse();
            return MoreObjects.toStringHelper(errorResponse).omitNullValues()
                    .add("errorObject", errorResponse.getErrorObject())
                    .toString();
        }
    }

    /**
     * Helper method for getting protocol message for introspection response.
     * 
     * @param response The response message
     * @return  The response message specific log message
     */
    @Nullable public static String getProtocolMessageForIntrospectionResponse(
            @Nonnull final TokenIntrospectionResponse response) {
        if (response.indicatesSuccess()) {
            final TokenIntrospectionSuccessResponse successResponse = response.toSuccessResponse();
            return MoreObjects.toStringHelper(successResponse).omitNullValues()
                    .add("active", successResponse.isActive())
                    .add("audience", successResponse.getAudience())
                    .add("clientId", successResponse.getClientID())
                    .add("expirationTime", successResponse.getExpirationTime())
                    .add("issuer", successResponse.getIssuer())
                    .add("issueTime", successResponse.getIssueTime())
                    .add("jwtId", successResponse.getJWTID())
                    .add("notBeforeTime", successResponse.getNotBeforeTime())
                    .add("parameters", successResponse.getParameters())
                    .add("scope", successResponse.getScope())
                    .add("jwkThumbprintConfirmation", successResponse.getJWKThumbprintConfirmation())
                    .add("subject", successResponse.getSubject())
                    .add("tokenType", successResponse.getTokenType())
                    .add("username", successResponse.getUsername())
                    .toString();
        } else {
            final TokenIntrospectionErrorResponse errorResponse = response.toErrorResponse();
            return MoreObjects.toStringHelper(errorResponse).omitNullValues()
                    .add("errorObject", errorResponse.getErrorObject())
                    .toString();
        }
    }

    /**
     * Helper method for getting protocol message for revocation response.
     * 
     * @param response The response message
     * @return  The response message specific log message
     */
    @Nullable public static String getProtocolMessageForRevocationResponse(final @Nonnull Response response) {
        if (response instanceof OAuth2RevocationSuccessResponse) {
            final OAuth2RevocationSuccessResponse successResponse = (OAuth2RevocationSuccessResponse) response;
            return MoreObjects.toStringHelper(successResponse).omitNullValues().toString();
        }
        if (response instanceof OAuth2RevocationErrorResponse) {
            final OAuth2RevocationErrorResponse errorResponse = (OAuth2RevocationErrorResponse) response;
            return MoreObjects.toStringHelper(errorResponse).omitNullValues()
                    .add("errorObject", errorResponse.getErrorObject())
                    .toString();
        } else {
            return null;
        }
    }

    /**
     * Helper method for getting protocol message for JSON success response.
     * 
     * @param response The response message
     * @return  The response message specific log message
     */
    @Nullable public static String getProtocolMessageForJSONSuccessResponse(final @Nonnull Response response) {
        if (response instanceof JSONSuccessResponse) {
            final JSONSuccessResponse successResponse = (JSONSuccessResponse) response;
            return successResponse.toString();
        }
        return null;
    }

}
