/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.audit.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;

import org.opensaml.profile.context.ProfileRequestContext;

import com.nimbusds.openid.connect.sdk.claims.ClaimsSet;

import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultResponseClaimsSetLookupFunction;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * A function that resolves a claim value from the id_token claims set.
 */
public class IdTokenClaimsAuditExtractor implements Function<ProfileRequestContext,String> {

    /** Lookup strategy for id token claims to read from. */
    @Nonnull private Function<ProfileRequestContext, ClaimsSet> idTokenClaimsLookupStrategy;

    /** The claim whose value is to be extracted. */
    @Nonnull @NotEmpty private final String key;
    
    /**
     * Constructor.
     *
     * @param claim The claim whose value is to be resolved from the id_token claims set.
     */
    public IdTokenClaimsAuditExtractor(@Nonnull @NotEmpty final String claim) {
        idTokenClaimsLookupStrategy = new DefaultResponseClaimsSetLookupFunction();
        key = Constraint.isNotEmpty(claim, "The claim cannot be empty");
    }
    
    /**
     * Set the lookup strategy for id token claims to read from.
     * @param strategy What to set.
     */
    public void setIdTokenClaimsLookupStrategy(final Function<ProfileRequestContext, ClaimsSet> strategy) {
        idTokenClaimsLookupStrategy =
                Constraint.isNotNull(strategy, "IdTokenClaimsStrategy lookup strategy cannot be null");

    }

    /** {@inheritDoc} */
    @Override
    public String apply(@Nonnull final ProfileRequestContext input) {
        final ClaimsSet claims = idTokenClaimsLookupStrategy.apply(input);
        if (claims != null) {
            return claims.toJSONObject().getAsString(key);
        }
        return null;
    }
}
