/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.admin.impl;

import java.io.IOException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.shibboleth.idp.profile.context.SpringRequestContext;
import net.shibboleth.oidc.metadata.ClientInformationManager;
import net.shibboleth.oidc.metadata.ClientInformationManagerException;
import net.shibboleth.oidc.metadata.ClientInformationResolver;
import net.shibboleth.oidc.metadata.criterion.ClientIDCriterion;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.webflow.execution.RequestContext;

import com.google.common.base.Strings;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientInformation;

/**
 * Action that implements a JSON REST API for querying and deleting OIDC client information.
 * 
 * <p>The API supports GET and DELETE at the moment, using jsonapi.org conventions.</p>
 * 
 * @event {@link EventIds#PROCEED_EVENT_ID}
 * @event {@link EventIds#INVALID_PROFILE_CTX}
 * @event {@link EventIds#IO_ERROR}
 * 
 * @since 3.1.0
 */
public class DoClientManagementOperation extends AbstractAdminApiProfileAction {
    
    /** Flow variable indicating ID of storage key. */
    @Nonnull @NotEmpty public static final String CLIENT_ID = "clientId";

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(DoClientManagementOperation.class);
    
    /** {@link ClientInformationResolver} to operate on. */
    @NonnullAfterInit private ClientInformationResolver resolver;

    /** {@link ClientInformationManager} to operate on. */
    @NonnullAfterInit private ClientInformationManager manager;

    /** Client ID to operate on. */
    @Nullable @NotEmpty private String clientId;

   /**
    * Set the {@link ClientInformationResolver} to use for retrieval.
    * 
    * @param theResolver client info resolver
    */
   public void setClientInformationResolver(@Nonnull final ClientInformationResolver theResolver) {
       ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
       
       resolver = Constraint.isNotNull(theResolver, "ClientInformationResolver cannot be null");
   }

   /**
    * Set the {@link ClientInformationManager} to use for deletion.
    * 
    * @param theManager client info manager
    */
   public void setClientInformationManager(@Nonnull final ClientInformationManager theManager) {
       ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
       
       manager = Constraint.isNotNull(theManager, "ClientInformationManager cannot be null");
   }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (resolver == null) {
            throw new ComponentInitializationException("ClientInformationResolver cannot be null");
        }

        if (manager == null) {
            throw new ComponentInitializationException("ClientInformationManager cannot be null");
        }
   }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(final ProfileRequestContext profileRequestContext) {
        
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        try {
            final SpringRequestContext springRequestContext =
                    profileRequestContext.getSubcontext(SpringRequestContext.class);
            if (springRequestContext == null) {
                log.warn("{} Spring request context not found in profile request context", getLogPrefix());
                sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
                        "Internal Server Error", "System misconfiguration.");
                return false;
            }
    
            final RequestContext requestContext = springRequestContext.getRequestContext();
            if (requestContext == null) {
                log.warn("{} Web Flow request context not found in Spring request context", getLogPrefix());
                sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
                        "Internal Server Error", "System misconfiguration.");
                return false;
            }
            
            clientId = (String) requestContext.getFlowScope().get(CLIENT_ID);
            if (Strings.isNullOrEmpty(clientId)) {
                sendError(HttpServletResponse.SC_NOT_FOUND,
                        "Missing client_id", "No client identifier specified.");
                return false;
            }
            
        } catch (final IOException e) {
            log.error("{} I/O error issuing API response", getLogPrefix(), e);
            ActionSupport.buildEvent(profileRequestContext, EventIds.IO_ERROR);
            return false;
        }

        return true;
    }

    /** {@inheritDoc} */
    @Override protected void doExecute(final ProfileRequestContext profileRequestContext) {
        
        try {
            final HttpServletRequest request = getHttpServletRequest();
            final HttpServletResponse response = getHttpServletResponse();
            
            response.setContentType("application/json");
            response.setHeader("Cache-Control", "must-revalidate,no-cache,no-store");
            
            if ("GET".equals(request.getMethod())) {
                final OIDCClientInformation record;
                try {
                    record = resolver.resolveSingle(new CriteriaSet(new ClientIDCriterion(new ClientID(clientId))));
                    if (record != null) {
                        response.setContentType("application/json");
                        response.setStatus(HttpServletResponse.SC_OK);
                        response.getOutputStream().print(record.toJSONObject().toJSONString());
                    } else {
                        sendError(HttpServletResponse.SC_NOT_FOUND,
                                "Record Not Found", "The specified record was not present or has expired.");
                    }
                } catch (final ResolverException e) {
                    log.error("{} Resolver error looking up client ID {}", getLogPrefix(), clientId, e);
                    sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, "Internal Server Error",
                            "Client info resolution error.");
                }
            } else if ("DELETE".equals(request.getMethod())) {
                try {
                    manager.destroyClientInformation(new ClientID(clientId));
                    response.setStatus(HttpServletResponse.SC_NO_CONTENT);
                } catch (final ClientInformationManagerException e) {
                    log.error("{} Error deleting client ID {}", getLogPrefix(), clientId, e);
                    sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, "Internal Server Error",
                            "ClientInformationManager error.");
                }
                
            } else {
                log.warn("{} Invalid method: {}", getLogPrefix(), request.getMethod());
                sendError(HttpServletResponse.SC_METHOD_NOT_ALLOWED,
                        "Unknown Operation", "Only GET and DELETE are supported.");
            }
                        
        } catch (final IOException e) {
            log.error("{} I/O error responding to request", getLogPrefix(), e);
            ActionSupport.buildEvent(profileRequestContext, EventIds.IO_ERROR);
        }
    }

}