/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.sdk.impl;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.plugin.authn.duo.DuoClientException;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClient;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClientFactory;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCIntegration;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.Unmodifiable;
import net.shibboleth.utilities.java.support.component.AbstractInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.primitive.StringSupport;


/** Abstract factory implementation for the {@link DuoSDKClientAdaptor}. */
@ThreadSafe
public final class DuoSDKClientFactory extends AbstractInitializableComponent implements DuoOIDCClientFactory{
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(DuoSDKClientFactory.class);     
    
    /** List of CA Certificate pins. If null, the client's default set are used.*/
    @GuardedBy("this") @Nullable @NonnullElements @Unmodifiable private List<String> caCerts;
      
    /**
     * Sets the list of CA certificate pins used to verify the Duo client connection
     * to the API host. If {@code null}, the client will use the internal defaults.
     * 
     * @param certs the list of certificate pins.
     */
    public synchronized void setCaCerts(@Nullable final List<String> certs) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        //check if null, as the native duo client uses the internal defaults if null
        //but will respect an empty list.
        if (certs != null) {
            caCerts = List.copyOf(StringSupport.normalizeStringCollection(certs));
        } else {
            caCerts = null;
        }       
    }
    
    /**
     * Get the caCerts, guarded by this objects monitor.
     * 
     * @return the caCerts.
     */
    @Nullable @NonnullElements @Unmodifiable private synchronized List<String> getCaCerts(){
        return caCerts;
    }

    @Override
    @Nonnull public DuoOIDCClient createInstance(@Nonnull final DuoOIDCIntegration integration) 
            throws DuoClientException {
        //every integration shares the same list of caCert pins.
        return new DuoSDKClientAdaptor(integration, getCaCerts());
    }

}
