/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.impl;

import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.servlet.http.HttpServletRequest;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.authn.AbstractAuthenticationAction;
import net.shibboleth.idp.authn.AuthnEventIds;
import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.plugin.authn.duo.DuoException;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClient;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCClientRegistry;
import net.shibboleth.idp.plugin.authn.duo.DuoOIDCIntegration;
import net.shibboleth.idp.plugin.authn.duo.DynamicDuoOIDCIntegration;
import net.shibboleth.idp.plugin.authn.duo.context.DuoOIDCAuthenticationContext;
import net.shibboleth.idp.session.context.navigate.CanonicalUsernameLookupStrategy;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;

/**
 * An action to create (or lookup) and populate the {@link DuoOIDCAuthenticationContext} 
 * with the username, chosen {@link DuoOIDCIntegration}, and {@link DuoOIDCClient} appropriate for this request. 
 * 
 * <p>Also determines the usable redirect_uri, either from one registered, or computed from the
 * HTTP request. Is set once, before the client is constructed, for every client. If however, the client supports 
 * dynamic use of the redirect_uri, it is also set into the context for use downstream.</p>
 * 
 * @event {@link org.opensaml.profile.action.EventIds#PROCEED_EVENT_ID}
 * @event {@link org.opensaml.profile.action.EventIds#INVALID_PROFILE_CTX}
 * @event {@link net.shibboleth.idp.authn.AuthnEventIds#NO_CREDENTIALS}
 * @event {@link net.shibboleth.idp.authn.AuthnEventIds#AUTHN_EXCEPTION}
 * @post See above.
 */
public class PopulateDuoAuthenticationContext extends AbstractAuthenticationAction {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(PopulateDuoAuthenticationContext.class);
    
    /** Strategy used to locate or create the {@link DuoOIDCAuthenticationContext} to populate. */
    @Nonnull private Function<ProfileRequestContext,DuoOIDCAuthenticationContext> duoAuthContextCreationStrategy;

    /** Lookup strategy for username to match against Duo identity. */
    @Nonnull private Function<ProfileRequestContext, String> usernameLookupStrategy;

    /** Lookup strategy for Duo integration. */
    @Nonnull private Function<ProfileRequestContext, DuoOIDCIntegration> duoIntegrationLookupStrategy;
    
    /** Strategy used to compute the redirectURI from the given Duo integration if supported.*/
    @Nullable
    private BiFunction<HttpServletRequest, DynamicDuoOIDCIntegration, String> redirectURICreationStrategy;
    
    /** The registry for locating the DuoClient for the established integration.*/
    @NonnullAfterInit private DuoOIDCClientRegistry clientRegistry;

    /** Constructor.*/
    public PopulateDuoAuthenticationContext() {
        //default creates duo authentication context under authentication context.
        duoAuthContextCreationStrategy =
                new ChildContextLookup<>(DuoOIDCAuthenticationContext.class, true).
                compose(new ChildContextLookup<>(AuthenticationContext.class));
        
        usernameLookupStrategy = new CanonicalUsernameLookupStrategy();
        duoIntegrationLookupStrategy = FunctionSupport.constant(null);
    }
    
    /**
     * Set the Duo client registry.
     * 
     * @param duoRegistry the registry
     */
    public void setClientRegistry(@Nonnull final DuoOIDCClientRegistry duoRegistry) { 
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        clientRegistry = Constraint.isNotNull(duoRegistry,"DuoClient registry can not be null");
    }

    /**
     * Set the lookup strategy to use for the username to match against Duo identity.
     * 
     * @param strategy lookup strategy
     */
    public void setUsernameLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        usernameLookupStrategy = Constraint.isNotNull(strategy, "Username lookup strategy cannot be null");
    }
    
    /**
     * Set the redirect URI creation strategy. The strategy is free to use or create a redirectURI based
     * either on runtime parameters, or static information in the {@link DuoOIDCIntegration}.
     * 
     * @param strategy the creation strategy.
     */
    public void setRedirectURICreationStrategy(
            @Nonnull final BiFunction<HttpServletRequest, DynamicDuoOIDCIntegration, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        redirectURICreationStrategy =  Constraint.isNotNull(strategy, "RedirectURI"
                + " creation strategy cannot be null");
    }
    
    /**
     * Set the strategy used to locate the {@link DuoOIDCAuthenticationContext} to operate on.
     * 
     * @param strategy lookup strategy
     */
    public void setDuoContextCreationStrategy(
            @Nonnull final Function<ProfileRequestContext,DuoOIDCAuthenticationContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        duoAuthContextCreationStrategy = Constraint.isNotNull(strategy, "DuoAuthenticationContext"
                + " creation strategy cannot be null");
    }

    /**
     * Set DuoIntegration lookup strategy to use.
     * 
     * @param strategy lookup strategy
     */
    public void setDuoIntegrationLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, DuoOIDCIntegration> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        duoIntegrationLookupStrategy = Constraint.isNotNull(strategy, "DuoIntegration lookup strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (clientRegistry ==  null) {
            throw new ComponentInitializationException("Duo Client Registry cannot be null");
        }

    }

    /** {@inheritDoc} */
    @Override protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext) {

        
        final DuoOIDCAuthenticationContext context = duoAuthContextCreationStrategy.apply(profileRequestContext);
        if (context == null) {
            log.error("{} Error creating DuoAuthenticationContext", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return;
        }
        
        final DuoOIDCIntegration duoIntegration = duoIntegrationLookupStrategy.apply(profileRequestContext);
        if (duoIntegration == null) {
            log.warn("{} No DuoIntegration returned by lookup strategy", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return;
        }
        context.setIntegration(duoIntegration);
        
        final HttpServletRequest request = getHttpServletRequest();
        if (request == null) {
            log.warn("{} Profile action does not contain an HttpServletRequest", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return;
        }   
        
        final String username = usernameLookupStrategy.apply(profileRequestContext);
        if (username == null) {
            log.warn("{} No principal name available to initiate a Duo 2FA request", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.NO_CREDENTIALS);
            return;
        }
        context.setUsername(username);
        
        try {
            computeAndStoreRedirectURIIfSupported(duoIntegration, request, context);     
            
            //Configure the Duo client for the established integration           
            final DuoOIDCClient client = clientRegistry.getClientOrCreate(duoIntegration);
            context.setClient(client);
            
        } catch (final DuoException e) {
            log.warn("{} Unable to establish a Duo Client for the given integration", getLogPrefix(),e);
            ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.AUTHN_EXCEPTION);
            return;
        }
        
        log.debug("Created Duo authentication context for '{}'",username);
    }
    
    /**
     * <p>For {@link DynamicDuoOIDCIntegration DynamicDuoOIDCIntegrations}, apply the redirect_uri creation
     * strategy to compute a redirect_uri to use.</p>
     * 
     *  <p>The redirect_uri is computed for each request, but is only set once as the usable redirect_uri
     *  on the integration itself i.e. for the client to read using {@link DuoOIDCIntegration#getRedirectURI()}.
     *  This allows all clients to see a computed (by the {@code redirectURICreationStrategy}) redirect_uri 
     *  from the first request onward.</p>
     *  
     *  <p>The computed redirect_uri is also added to the context as an override redirect_uri which - if 
     *  supported by the client - can be used dynamically when creating authorization or token exchange 
     *  requests.</p>
     *  
     * @param duoIntegration the Duo integration pertaining to this request.
     * @param request the http servlet request.
     * @param context the Duo authentication context to store the computed override redirect_uri.
     * 
     * @throws DuoException if the redirect_uri could not be created by the strategy.
     */
    private void computeAndStoreRedirectURIIfSupported(@Nonnull final DuoOIDCIntegration duoIntegration,
            @Nonnull final HttpServletRequest request,
            @Nonnull final DuoOIDCAuthenticationContext context) throws DuoException {
        
        if (duoIntegration instanceof DynamicDuoOIDCIntegration) {
            
            if (redirectURICreationStrategy == null) {
                throw new DuoException("A dynamic DuoOIDC integration was supplied, but the redirect URI"
                        + " creation strategy was null. Please set a redirect URI creation strategy.");
            }
            
            final String redirectURI = redirectURICreationStrategy.apply(request,
                    (DynamicDuoOIDCIntegration)duoIntegration);
            if (redirectURI == null) {
                throw new DuoException("A redirect_uri was not registered or could not be computed");
            }
           
            // set the redirectURI iff not already set. This should happen at least once on the first
            // request (whichever thread gets here first), but not thereafter. This allows it to be 
            // computed from the request, and is compatible with clients that do not support per-request 
            // redirect_uri overrides (if required).
            ((DynamicDuoOIDCIntegration)duoIntegration).setRedirectURIIfAbsent(redirectURI); 
            
            //always add to the context for the client to use if it supports dynamic redirect_uris
            log.trace("{} Adding a dynamic redirect_uri '{}' to the context for the DuoClient to use if "
                    + "supported",getLogPrefix(),redirectURI);
            context.setRedirectURIOverride(redirectURI);
        }
    }
    
}
