/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.authn.duo.impl;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.function.BiFunction;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.ThreadSafe;
import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.plugin.authn.duo.DynamicDuoOIDCIntegration;
import net.shibboleth.idp.plugin.authn.duo.URISupport;
import net.shibboleth.utilities.java.support.annotation.ParameterName;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 *  Constructive, pure, function that returns a redirect_uri from one of (ordered):
 *  <ol>
 *      <li>A pre-registered redirect_uri on the Duo integration, 
 *          {@link DynamicDuoOIDCIntegration#getRegisteredRedirectURI()}. Or, if none are pre-registered;
 *      <li>Derived from the HTTP Servlet request server parameters, checking the origin
 *          against an allowed set of origins from the Duo integration - to prevent Host header injection.
 *   </ol>
 *  
 *  <p>Returns null if one can not be constructed.</p>
 *  
 *  <p>Is thread-safe and immutable</p> 
 */
@ThreadSafe
@Immutable
public final class DefaultRedirectURICreationStrategy 
            implements BiFunction<HttpServletRequest, DynamicDuoOIDCIntegration, String>{
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(DefaultRedirectURICreationStrategy.class);
    
    /** The path, excluding the context and servlet paths, to the Duo callback handler.*/
    @Nonnull @NotEmpty private final String callbackServletPath;
    
    /**
     * Constructor.
     *
     * @param callbackPath the path segment relative to the servlet path of the callback endpoint.
     */
    public DefaultRedirectURICreationStrategy(
            @Nonnull @NotEmpty @ParameterName(name="callbackPath") final String callbackPath) {
        
        callbackServletPath = Constraint.isNotNull(callbackPath,"Duo Call back path can not be null");
    }

    @Override
    @Nullable public String apply(@Nonnull final HttpServletRequest request, 
            @Nonnull final DynamicDuoOIDCIntegration integration) {
        
        final String redirectFromIntegration = integration.getRegisteredRedirectURI();
        if (redirectFromIntegration != null) {
            log.trace("Using redirect_uri '{}' from the Duo integration settings", redirectFromIntegration);
            return redirectFromIntegration;
        }
        try {
            final URI uri = URISupport.buildURIIgnoreDefaultPorts(request.getScheme(),
                    request.getServerName(),
                    request.getServerPort(),
                    request.getContextPath()+request.getServletPath()+callbackServletPath);
    
            final String origin = URISupport.buildOrigin(uri);
            if (!integration.getAllowedOrigins().contains(origin)) {
                log.warn("The 'origin' of the computed redirect_uri ('{}') is not allowed. If permissible, add it "
                        + "to the allowed origins property.",origin);
                return null;
            }
            return uri.toString();
        } catch (final URISyntaxException e) {
            log.warn("Unable to generate redirect_uri, {}",e.getMessage());            
        }
        return null;
        
    }
    
}
