/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.saml.attribute.resolver.impl;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.idp.attribute.EmptyAttributeValue;
import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.IdPAttributeValue;
import net.shibboleth.idp.attribute.StringAttributeValue;
import net.shibboleth.idp.attribute.resolver.AbstractDataConnector;
import net.shibboleth.idp.attribute.resolver.PluginDependencySupport;
import net.shibboleth.idp.attribute.resolver.ResolverAttributeDefinitionDependency;
import net.shibboleth.idp.attribute.resolver.ResolverDataConnectorDependency;
import net.shibboleth.idp.attribute.resolver.ResolverPluginDependency;
import net.shibboleth.idp.attribute.resolver.context.AttributeResolverWorkContext;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The basis of a {@link net.shibboleth.idp.attribute.resolver.DataConnector} that handles persistent IDs that depend on
 * a source {@link IdPAttribute}.
 */
public abstract class AbstractPersistentIdDataConnector extends AbstractDataConnector {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(AbstractPersistentIdDataConnector.class);

    /** ID of the attribute generated by this data connector. */
    @NonnullAfterInit private String generatedAttribute;

    /** ID of the attribute whose first value is used when generating the computed ID. */
    @NonnullAfterInit private String sourceAttribute;

    /** Information about the dependency. */
    @NonnullAfterInit private String sourceInformation;

    /**
     * Get the ID of the attribute whose first value is used when generating the computed ID.
     * 
     * @return ID of the attribute whose first value is used when generating the computed ID
     */
    @Nullable @Deprecated public String getSourceAttributeId() {
        return sourceAttribute;
    }
    
    /**
     * Get Information about the attribute whose first value is used when generating the computed ID.
     * This is derived from the sourceID (if present) and/or the dependencies.  
     * Public purely as an aid to testing.
     *
     * @return log-friend information.
     */
    @Nullable @NonnullAfterInit public String getSourceAttributeInformation() {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        return sourceInformation;
    }

    /**
     * Set the ID of the attribute whose first value is used when generating the computed ID.
     * 
     * @param newAttributeId what to set.
     */
    public void setSourceAttributeId(@Nullable final String newAttributeId) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        sourceAttribute = StringSupport.trimOrNull(newAttributeId);
    }

    /**
     * Get the ID of the attribute generated by this connector.
     * 
     * @return ID of the attribute generated by this connector
     */
    @NonnullAfterInit public String getGeneratedAttributeId() {
        return generatedAttribute;
    }

    /**
     * Set the ID of the attribute generated by this connector.
     * 
     * @param newAttributeId what to set.
     */
    public void setGeneratedAttributeId(@Nullable final String newAttributeId) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        generatedAttribute = newAttributeId;
    }
    
    /**
     * Do the dance with dependencies.
     * 
     * Old style ones get the sourceId added (failing if it isn't there).
     * New style ones get their names added to the information string.
     *
     * @throws ComponentInitializationException if the dependencies are not aligned correctly
     */
 // Checkstyle: CyclomaticComplexity|MethodLength OFF
    private void doDependencyInformation() throws ComponentInitializationException {
        final StringBuilder dependencyInformation = new StringBuilder();
        boolean seenAttribute = false;
        for (final ResolverPluginDependency depends : getDependencies()) {
            if (seenAttribute) {
                dependencyInformation.append(", ");
            }
            if (depends instanceof ResolverAttributeDefinitionDependency) {
                dependencyInformation.append(depends.getDependencyPluginId());
                // No other work needed.  The name is the reference
            } else if (depends instanceof ResolverDataConnectorDependency) {
                final ResolverDataConnectorDependency dataConnectorDependency =
                        (ResolverDataConnectorDependency) depends;
                if (dataConnectorDependency.isAllAttributes()) {
                    dependencyInformation.append(depends.getDependencyPluginId()).append("/*");
                } else if (dataConnectorDependency.getAttributeNames().isEmpty()) {
                    throw new ComponentInitializationException(getLogPrefix() + " No source attribute present.");
                } else if (dataConnectorDependency.getAttributeNames().size() == 1) {
                    dependencyInformation.append(dataConnectorDependency.getDependencyPluginId()).
                                          append('/').
                                          append(dataConnectorDependency.getAttributeNames().iterator().next());
                } else {
                    dependencyInformation.append(dataConnectorDependency.getDependencyPluginId()).
                                          append('/').
                                          append(dataConnectorDependency.getAttributeNames().toString());
                }
                // No work needed.  The names are stored elsewhere
            } else {
                if (null == getSourceAttributeId()) {
                    throw new ComponentInitializationException(getLogPrefix() + " No source attribute present.");
                }
                dependencyInformation.append(depends.getDependencyPluginId()).
                                      append('/').
                                      append(getSourceAttributeId());
                depends.setDependencyAttributeId(getSourceAttributeId());
            }
            seenAttribute = true;
        }
        if (!seenAttribute) {
            if (null == getSourceAttributeId()) {
                throw new ComponentInitializationException(getLogPrefix() + " No source attribute present.");
            } else {
                log.warn("{} source Attribute {} present, but not declared as a dependency", getLogPrefix(), 
                        getSourceAttributeId());
                dependencyInformation.append(getSourceAttributeId());
            }
        }
        sourceInformation = dependencyInformation.toString();
        log.debug("{} Source for definition: {}", getLogPrefix(), sourceInformation);
    }
 // Checkstyle: CyclomaticComplexity|MethodLength ON

    /** {@inheritDoc} */
    @Override protected void doInitialize() throws ComponentInitializationException { 

        // Set up the dependencies first. Then the initialize in the parent
        // will correctly rehash the dependencies.
        doDependencyInformation();
        super.doInitialize();

        if (null == generatedAttribute) {
            generatedAttribute = getId();
            log.info("{} No generated attribute ID supplied, using ID of connector: {}", getLogPrefix(),
                    generatedAttribute);
        }
    }

    /**
     * Helper function to locate the source Attribute in the dependencies.
     * 
     * @param workContext the context to look in
     * @return the value, or null in any of the failure cases.
     */
    @Nullable protected String resolveSourceAttribute(@Nonnull final AttributeResolverWorkContext workContext) {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        final List<IdPAttributeValue<?>> attributeValues =
                PluginDependencySupport.getMergedAttributeValues(workContext, getDependencies(), getId());
        if (attributeValues == null || attributeValues.isEmpty()) {
            log.debug("{} Source attribute {} for connector {} provide no values", getLogPrefix(),
                    getSourceAttributeInformation(), getId());
            return null;
        }

        if (attributeValues.size() > 1) {
            log.warn("{} Source attribute {} for connector {} has more than one value, only one value is used",
                    getLogPrefix(), getSourceAttributeInformation(), getId());
        }

        final IdPAttributeValue attributeValue = attributeValues.iterator().next();

        final String val;

        if (attributeValue instanceof StringAttributeValue) {
            if (StringSupport.trimOrNull((String) attributeValue.getValue()) == null) {
                log.warn("{} Source attribute {} for connector {} was all-whitespace", getLogPrefix(),
                        getSourceAttributeInformation(), getId());
                return null;
            }
            val = (String) attributeValue.getValue();
        } else if (attributeValue instanceof EmptyAttributeValue) {
            final EmptyAttributeValue emptyVal = (EmptyAttributeValue) attributeValue;
            log.warn("{} Source attribute {} value for connector {} was an empty value of type {}", getLogPrefix(),
                    getSourceAttributeInformation(), getId(), emptyVal.getDisplayValue());
            return null;
        } else {
            log.warn("{} Source attribute {} for connector {} was of an unsupported type: {}", getLogPrefix(),
                    getSourceAttributeInformation(), getId(), attributeValue.getClass().getName());
            return null;
        }

        if (val == null) {
            log.warn("{} Attribute value {} for connector {} resolved as empty or null", getLogPrefix(),
                    getSourceAttributeInformation(), getId());
        }
        return val;
    }

    /**
     * Encode the provided string.
     * 
     * @param value the value to encode or null if that failed
     * @return null or the attribute.
     */
    @Nullable protected Map<String, IdPAttribute> encodeAsAttribute(@Nullable final String value) {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        if (null == value) {
            // The message will have been logged above
            return null;
        }
        final IdPAttribute attribute = new IdPAttribute(getGeneratedAttributeId());
        attribute.setValues(Collections.singletonList(StringAttributeValue.valueOf(value)));
        return Collections.singletonMap(getGeneratedAttributeId(), attribute);
    }

}