/*
 * =============================================================================
 * Copyright (C) 2010-2021 by the Okapi Framework contributors
 * -----------------------------------------------------------------------------
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
package net.sf.okapi.filters.openxml;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventFactory;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Attribute;
import javax.xml.stream.events.EndElement;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

interface Cell {
    String NAME = "c";
    CellType type();
    String worksheetName();
    CellReferencesRange cellReferencesRange();
    boolean valuePresent();
    CellValue value();
    List<XMLEvent> inlineStringEvents();
    boolean excluded();
    MetadataContext metadataContext();
    void readWith(final XMLEventReader reader) throws XMLStreamException;
    Markup asMarkup();

    class Default implements Cell {
        private static final String INLINE_STRING = "is";
        private static final QName CELL_LOCATION_REFERENCE = new QName("r");
        private static final QName CELL_TYPE = new QName("t");
        private static final QName CELL_STYLE = new QName("s");
        private final ConditionalParameters conditionalParameters;
        private final XMLEventFactory eventFactory;
        private final boolean date1904;
        private final StyleDefinitions styleDefinitions;
        private final Set<Integer> excludedRows;
        private final Set<String> excludedColumns;
        private final Set<Integer> metadataRows;
        private final Set<String> metadataColumns;
        private final List<CellReferencesRange> cellReferencesRanges;
        private final String worksheetName;
        private final StartElement origStartElement;
        private StartElement startElement;
        private CellType type;
        private CellReferencesRange cellReferencesRange;
        private Formula formula;
        private CellValue value;
        private List<XMLEvent> inlineStringEvents;
        private boolean excluded;
        private MetadataContext metadataContext;
        private EndElement endElement;

        Default(
            final ConditionalParameters conditionalParameters,
            final XMLEventFactory eventFactory,
            final boolean date1904,
            final StyleDefinitions styleDefinitions,
            final Set<Integer> excludedRows,
            final Set<String> excludedColumns,
            final Set<Integer> metadataRows,
            final Set<String> metadataColumns,
            final List<CellReferencesRange> cellReferencesRanges,
            final String worksheetName,
            final StartElement startElement
        ) {
            this.conditionalParameters = conditionalParameters;
            this.eventFactory = eventFactory;
            this.date1904 = date1904;
            this.styleDefinitions = styleDefinitions;
            this.excludedRows = excludedRows;
            this.excludedColumns = excludedColumns;
            this.metadataRows = metadataRows;
            this.metadataColumns = metadataColumns;
            this.cellReferencesRanges = cellReferencesRanges;
            this.worksheetName = worksheetName;
            this.origStartElement = startElement;
        }

        @Override
        public CellType type() {
            return this.type;
        }

        @Override
        public String worksheetName() { return this.worksheetName; }

        @Override
        public CellReferencesRange cellReferencesRange() {
            return this.cellReferencesRange;
        }

        @Override
        public boolean valuePresent() {
            return null != this.value;
        }

        @Override
        public CellValue value() {
            return this.value;
        }

        @Override
        public List<XMLEvent> inlineStringEvents() {
            return this.inlineStringEvents;
        }

        @Override
        public boolean excluded() {
            return this.excluded;
        }

        @Override
        public MetadataContext metadataContext() {
            if (null == this.metadataContext) {
                final boolean metadataRow = this.cellReferencesRange.rows().stream()
                    .anyMatch(r -> this.metadataRows.contains(r));
                final boolean metadataColumn = this.cellReferencesRange.columns().stream()
                    .anyMatch(c -> this.metadataColumns.contains(c));
                if (metadataRow && metadataColumn) {
                    this.metadataContext = MetadataContext.ROW_AND_COLUMN;
                } else if (metadataRow) {
                    this.metadataContext = MetadataContext.ROW;
                } else if (metadataColumn) {
                    this.metadataContext = MetadataContext.COLUMN;
                } else {
                    this.metadataContext = MetadataContext.NONE;
                }
            }
            return this.metadataContext;
        }

        @Override
        public void readWith(final XMLEventReader reader) throws XMLStreamException {
            final Attribute typeAttr = this.origStartElement.getAttributeByName(CELL_TYPE);
            this.type = CellType.from(typeAttr);
            if (CellType.INLINE_STRING == this.type) {
                this.startElement = newStartElement(this.origStartElement, typeAttr);
            } else {
                this.startElement = this.origStartElement;
            }
            this.cellReferencesRange = cellReferencesRange(
                new CellReference(this.startElement.getAttributeByName(CELL_LOCATION_REFERENCE).getValue())
            );
            final DifferentialFormat.Combined format = combinedDifferentialFormatFor(this.startElement);
            this.excluded = !this.cellReferencesRange.partialMatch(this.excludedRows, this.excludedColumns)
                || excludedFor(format);

            while (reader.hasNext()) {
                final XMLEvent e = reader.nextEvent();
                if (e.isEndElement() && e.asEndElement().getName().equals(this.startElement.getName())) {
                    this.endElement = e.asEndElement();
                    break;
                }
                if (!e.isStartElement()) {
                    continue;
                }
                final StartElement se = e.asStartElement();
                if (Formula.F.equals(se.getName().getLocalPart())) {
                  this.formula = new Formula.Default(
                      se,
                      new MarkupComponent.Context.Default(this.worksheetName)
                  );
                  this.formula.readWith(reader);
                } else if (CellValue.NAME.equals(se.getName().getLocalPart())) {
                    this.value = new CellValue.Default(this.eventFactory, this.date1904, this.type, format, se);
                    this.value.readWith(reader);
                } else if (INLINE_STRING.equals(se.getName().getLocalPart())) {
                    this.inlineStringEvents = Default.inlineStringEvents(se, reader);
                    this.value = new CellValue.Default(
                        this.eventFactory,
                        this.date1904,
                        CellType.SHARED_STRING,
                        format,
                        this.eventFactory.createStartElement(
                            se.getName().getPrefix(),
                            se.getName().getNamespaceURI(),
                            CellValue.NAME
                        )
                    );
                }
            }
        }

        private StartElement newStartElement(final StartElement startElement, final Attribute attribute) {
            final List<Attribute> newAttributes = new ArrayList<>();
            final Iterator iterator = startElement.getAttributes();
            while (iterator.hasNext()) {
                final Attribute a = (Attribute) iterator.next();
                if (CELL_TYPE.equals(a.getName())) {
                    newAttributes.add(
                        this.eventFactory.createAttribute(
                            attribute.getName(),
                            CellType.SHARED_STRING.toString()
                        )
                    );
                    continue;
                }
                newAttributes.add(a);
            }
            return this.eventFactory.createStartElement(
                startElement.getName(),
                newAttributes.iterator(),
                startElement.getNamespaces()
            );
        }

        private CellReferencesRange cellReferencesRange(final CellReference cellReference) {
            return this.cellReferencesRanges.stream()
                .filter(r -> r.first().equals(cellReference))
                .findFirst()
                .orElse(new CellReferencesRange(cellReference));
        }

        private DifferentialFormat.Combined combinedDifferentialFormatFor(final StartElement startElement) {
            final Attribute styleAttr = startElement.getAttributeByName(CELL_STYLE);
            if (styleAttr == null) {
                return this.styleDefinitions.combinedDifferentialFormatFor(ExcelStyleDefinition.CellFormats.DEFAULT_INDEX);
            }
            final int styleIndex = Integer.parseUnsignedInt(styleAttr.getValue());
            return this.styleDefinitions.combinedDifferentialFormatFor(styleIndex);
        }

        private boolean excludedFor(final DifferentialFormat format) {
            return this.conditionalParameters.tsExcelExcludedColors.stream()
                .anyMatch(c -> format.fill().pattern().foregroundColor().argb().equals(c));
        }

        private static List<XMLEvent> inlineStringEvents(final StartElement startElement, final XMLEventReader reader) throws XMLStreamException {
            final List<XMLEvent> events = new ArrayList<>();
            while (reader.hasNext()) {
                final XMLEvent e = reader.nextEvent();
                if (e.isEndElement() && e.asEndElement().getName().equals(startElement.getName())) {
                    break;
                }
                events.add(e);
            }
            return events;
        }

        @Override
        public Markup asMarkup() {
            final MarkupBuilder mb = new MarkupBuilder(new Markup.General(new ArrayList<>()));
            mb.add(this.startElement);
            if (null != this.formula) {
                mb.add(this.formula);
            }
            if (valuePresent()) {
                mb.add(this.value.asMarkup());
            }
            mb.add(this.endElement);
            return mb.build();
        }
    }
}
