/*===========================================================================
  Copyright (C) 2016-2017 by the Okapi Framework contributors
-----------------------------------------------------------------------------
  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.
===========================================================================*/

package net.sf.okapi.filters.openxml;

import javax.xml.stream.XMLEventFactory;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.EndElement;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;

interface Worksheet {
	void readWith(final XMLEventReader reader) throws XMLStreamException;
	Markup asMarkup();

	final class Default implements Worksheet {
		private static final String SHEET_VIEW = "sheetView";
		private final ConditionalParameters conditionalParameters;
		private final XMLEventFactory eventFactory;
		private final boolean date1904;
		private final Cells cells;
		private final StyleDefinitions styleDefinitions;
		private final String name;
		private final WorksheetFragments fragments;
		private MarkupBuilder markupBuilder;

		Default(
			final ConditionalParameters conditionalParameters,
			final XMLEventFactory eventFactory,
			final boolean date1904,
			final Cells cells,
			final StyleDefinitions styleDefinitions,
			final String name,
			final WorksheetFragments fragments
		) {
			this.conditionalParameters = conditionalParameters;
			this.eventFactory = eventFactory;
			this.date1904 = date1904;
			this.cells = cells;
			this.styleDefinitions = styleDefinitions;
			this.name = name;
			this.fragments = fragments;
		}

		@Override
		public void readWith(final XMLEventReader reader) throws XMLStreamException {
			this.fragments.readWith(reader);
			final Set<Integer> excludedRows = new HashSet<>(
				this.conditionalParameters.worksheetConfigurations().excludedRowsFor(this.name)
			);
			excludedRows.addAll(this.fragments.hiddenRows());
			final Set<Integer> metadataRows = this.conditionalParameters.worksheetConfigurations().metadataRowsFor(this.name);
			excludedRows.addAll(metadataRows);
			final Set<String> excludedColumns = new HashSet<>(
				this.conditionalParameters.worksheetConfigurations().excludedColumnsFor(this.name)
			);
			excludedColumns.addAll(this.fragments.hiddenColumns());
			final Set<String> metadataColumns = this.conditionalParameters.worksheetConfigurations().metadataColumnsFor(this.name);
			excludedColumns.addAll(metadataColumns);
			this.markupBuilder = new MarkupBuilder(new Markup.General(new ArrayList<>()));
			final XMLEventReader eventsReader = new XMLEventsReader(this.fragments.events());

			while (eventsReader.hasNext()) {
				final XMLEvent e = eventsReader.nextEvent();
				if (e.isStartElement()) {
					final StartElement se = e.asStartElement();
					if (SHEET_VIEW.equals(se.getName().getLocalPart())) {
						this.markupBuilder.add(new MarkupComponent.Start(this.eventFactory, se));
						continue;
					} else if (Cell.NAME.equals(se.getName().getLocalPart())) {
						final Cell cell = new Cell.Default(
							this.conditionalParameters,
							this.eventFactory,
							this.date1904,
							this.styleDefinitions,
							excludedRows,
							excludedColumns,
							metadataRows,
							metadataColumns,
							this.fragments.cellReferencesRanges(),
							this.name,
							se
						);
						cell.readWith(eventsReader);
						this.cells.add(cell);
						this.markupBuilder.add(cell.asMarkup());
						continue;
					}
				} else if (e.isEndElement()) {
					final EndElement ee = e.asEndElement();
					if (SHEET_VIEW.equals(ee.getName().getLocalPart())) {
						this.markupBuilder.add(new MarkupComponent.End(ee));
						continue;
					}
				}
				this.markupBuilder.add(e);
			}
		}

		@Override
		public Markup asMarkup() {
			return this.markupBuilder.build();
		}
	}
}
