package net.sf.jagg;

/**
 * This class represents the "max" aggregator over <code>Comparable</code>
 * values.
 *
 * @author Randy Gettman
 * @since 0.1.0
 */
public class MaxAggregator extends Aggregator
{
   private Comparable myMax;

   /**
    * Constructs a <code>MaxAggregator</code> that operates on the specified
    * property.
    * @param property Determine the maximum of this property's values.
    */
   public MaxAggregator(String property)
   {
      setProperty(property);
   }

   /**
    * Returns an uninitialized copy of this <code>Aggregator</code> object,
    * with the same property(ies) to analyze.
    * @return An uninitialized copy of this <code>Aggregator</code> object.
    */
   public MaxAggregator replicate()
   {
      return new MaxAggregator(getProperty());
   }
   
   /**
    * Initialize the maximum to <code>null</code>.
    */
   public void init()
   {
      myMax = null;
   }

   /**
    * Store the property value if it's higher than the current maximum.
    *
    * @param value The value to aggregate.
    */
   @SuppressWarnings("unchecked")
   public void iterate(Object value)
   {
      if (value != null)
      {
         String property = getProperty();
         try
         {
            Comparable obj = (Comparable) getValueFromProperty(value, property);
            // Don't count nulls.
            if (obj != null)
            {
               if (myMax == null || obj.compareTo(myMax) > 0)
                  myMax = obj;
            }
         }
         catch (ClassCastException e)
         {
            throw new UnsupportedOperationException("Property \"" + property +
               "\" must be Comparable.", e);
         }
      }
   }

   /**
    * Merge the given <code>Aggregator</code> into this one by taking the
    * maximum of the two maximums.
    *
    * @param agg The <code>Aggregator</code> to merge into this one.
    */
   @SuppressWarnings("unchecked")
   public void merge(Aggregator agg)
   {
      if (agg != null && agg instanceof MaxAggregator)
      {
         MaxAggregator otherAgg = (MaxAggregator) agg;
         if (otherAgg.myMax.compareTo(myMax) > 0)
            myMax = otherAgg.myMax;
      }
   }

   /**
    * Return the maximum.
    *
    * @return The maximum as a <code>Comparable</code>, or <code>null</code> if
    *    no values were processed.
    */
   public Comparable terminate()
   {
      return myMax;
   }
}
