/**
 *  Copyright 2003-2010 Terracotta, Inc.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package net.sf.ehcache.terracotta;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;

import net.sf.ehcache.CacheException;
import net.sf.ehcache.cluster.CacheCluster;
import net.sf.ehcache.cluster.ClusterNode;
import net.sf.ehcache.cluster.ClusterScheme;
import net.sf.ehcache.cluster.ClusterTopologyListener;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * {@link CacheCluster} implementation that delegates to an underlying cache cluster. The underlying {@link CacheCluster} can be changed
 * dynamically
 *
 * @author Abhishek Sanoujam
 *
 */
public class TerracottaCacheCluster implements CacheCluster {

    private static final Logger LOGGER = LoggerFactory.getLogger(TerracottaCacheCluster.class);
    private final List<ClusterTopologyListener> listeners = new CopyOnWriteArrayList<ClusterTopologyListener>();
    private volatile CacheCluster realCacheCluster;

    /**
     * Set the underlying cache
     *
     * @param newCacheCluster
     */
    public void setUnderlyingCacheCluster(CacheCluster newCacheCluster) {
        if (newCacheCluster == null) {
            throw new IllegalArgumentException("CacheCluster can't be null");
        }
        final CacheCluster oldRealCacheCluster = this.realCacheCluster;
        this.realCacheCluster = newCacheCluster;
        for (ClusterTopologyListener listener : listeners) {
            this.realCacheCluster.addTopologyListener(listener);
        }
        if (oldRealCacheCluster != null) {
            for (ClusterTopologyListener listener : listeners) {
                oldRealCacheCluster.removeTopologyListener(listener);
            }
        }
    }

    /**
     * Fire Rejoin event to all listeners.
     * Package protected method
     *
     * @param clusterNode
     * @param oldNode
     */
    void fireNodeRejoinedEvent(ClusterNode oldNode, ClusterNode newNode) {
        Set<ClusterTopologyListener> firedToListeners = new HashSet<ClusterTopologyListener>();
        for (ClusterTopologyListener listener : listeners) {
            fireRejoinEvents(oldNode, newNode, listener);
            firedToListeners.add(listener);
        }
        for (ClusterTopologyListener listener : realCacheCluster.getTopologyListeners()) {
            if (firedToListeners.contains(listener)) {
                continue;
            }
            fireRejoinEvents(oldNode, newNode, listener);
        }
    }

    private void fireRejoinEvents(ClusterNode oldNode, ClusterNode newNode, ClusterTopologyListener listener) {
        try {
            listener.nodeJoined(newNode);
            listener.clusterOnline(newNode);
            listener.clusterRejoined(new DisconnectedClusterNode(oldNode), newNode);
        } catch (Throwable e) {
            LOGGER.error("Caught exception while firing rejoin event", e);
        }
    }

    /**
     * {@inheritDoc}
     */
    public boolean addTopologyListener(ClusterTopologyListener listener) {
        checkIfInitialized();
        boolean added = realCacheCluster.addTopologyListener(listener);
        if (added) {
            listeners.add(listener);
        }
        return added;
    }

    /**
     * {@inheritDoc}
     */
    public boolean removeTopologyListener(ClusterTopologyListener listener) {
        checkIfInitialized();
        boolean removed = realCacheCluster.removeTopologyListener(listener);
        if (removed) {
            listeners.remove(listener);
        }
        return removed;
    }

    /**
     * {@inheritDoc}
     */
    public ClusterNode getCurrentNode() {
        checkIfInitialized();
        return realCacheCluster.getCurrentNode();
    }

    /**
     * {@inheritDoc}
     */
    public Collection<ClusterNode> getNodes() {
        checkIfInitialized();
        return realCacheCluster.getNodes();
    }

    /**
     * {@inheritDoc}
     */
    public ClusterScheme getScheme() {
        checkIfInitialized();
        return realCacheCluster.getScheme();
    }

    /**
     * {@inheritDoc}
     */
    public boolean isClusterOnline() {
        checkIfInitialized();
        return realCacheCluster.isClusterOnline();
    }

    /**
     * {@inheritDoc}
     */
    public ClusterNode waitUntilNodeJoinsCluster() {
        checkIfInitialized();
        return realCacheCluster.waitUntilNodeJoinsCluster();
    }

    private void checkIfInitialized() {
        if (realCacheCluster == null) {
            throw new CacheException(
                    "The underlying cache cluster has not been initialized. Probably the terracotta client has not been configured yet.");
        }
    }

    /**
     * Until fixed in terracotta core, getHostName() and getIp() does not work after node has disconnected.
     * Temporary fix for beta release.
     *
     */
    private static class DisconnectedClusterNode implements ClusterNode {

        private final ClusterNode delegateNode;

        /**
         * Constructor accepting the actual delegate node
         *
         * @param actualNode
         */
        public DisconnectedClusterNode(final ClusterNode actualNode) {
            this.delegateNode = actualNode;
        }

        /**
         * {@inheritDoc}
         */
        public String getId() {
            return delegateNode.getId();
        }

        /**
         * {@inheritDoc}
         */
        public String getHostname() {
            String hostName = "";
            try {
                hostName = InetAddress.getLocalHost().getHostName();
            } catch (UnknownHostException e) {
                hostName = "[Can't determine hostname and " + delegateNode.getId() + " has DISCONNECTED]";
            }
            return hostName;
        }

        /**
         * {@inheritDoc}
         */
        public String getIp() {
            String ip = "";
            try {
                ip = InetAddress.getLocalHost().getHostAddress();
            } catch (UnknownHostException e) {
                ip = "[Can't determine IP and " + delegateNode.getId() + " has DISCONNECTED]";
            }
            return ip;
        }

    }

    /**
     * {@inheritDoc}
     */
    public List<ClusterTopologyListener> getTopologyListeners() {
        return this.listeners;
    }

}
