/*
 * Decompiled with CFR 0.152.
 */
package com.terracottatech.offheapstore.paging;

import com.terracottatech.offheapstore.paging.Page;
import com.terracottatech.offheapstore.paging.PageSource;
import com.terracottatech.offheapstore.storage.allocator.BestFitAllocator;
import com.terracottatech.offheapstore.util.DebuggingUtils;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OffHeapStorageArea {
    private static final Logger LOGGER = LoggerFactory.getLogger(OffHeapStorageArea.class);
    private final int initialPageSize;
    private final int maximalPageSize;
    private final int pageGrowthAreaSize;
    private final Owner owner;
    private final PageSource pageSource;
    private final BestFitAllocator allocator;
    private final Map<Integer, Page> pages = new ConcurrentHashMap<Integer, Page>(1, 0.75f, 1);
    private final boolean thief;
    private final boolean victim;

    public OffHeapStorageArea(Owner owner, PageSource pageSource, int pageSize, boolean thief, boolean victim) {
        this(owner, pageSource, pageSize, pageSize, thief, victim);
    }

    public OffHeapStorageArea(Owner owner, PageSource pageSource, int initialPageSize, int maximalPageSize, boolean thief, boolean victim) {
        if (victim && maximalPageSize != initialPageSize) {
            throw new IllegalArgumentException("Variable page-size offheap storage areas cannot be victims as they do not support stealing.");
        }
        this.owner = owner;
        this.pageSource = pageSource;
        this.initialPageSize = Integer.bitCount(initialPageSize = Math.max(BestFitAllocator.MINIMAL_SIZE, initialPageSize)) == 1 ? initialPageSize : Integer.highestOneBit(initialPageSize) << 1;
        this.maximalPageSize = maximalPageSize < initialPageSize ? initialPageSize : (Integer.bitCount(maximalPageSize) == 1 ? maximalPageSize : Integer.highestOneBit(maximalPageSize) << 1);
        this.pageGrowthAreaSize = this.maximalPageSize - this.initialPageSize;
        this.allocator = new BestFitAllocator(this);
        this.thief = thief;
        this.victim = victim;
    }

    public void clear() {
        this.allocator.clear();
        Iterator<Page> it = this.pages.values().iterator();
        while (it.hasNext()) {
            Page p = it.next();
            it.remove();
            this.pageSource.free(p);
        }
        assert (this.validatePages());
    }

    public int readInt(int address) {
        int pageSize;
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + 4 <= (pageSize = this.pageSizeFor(pageIndex))) {
            return this.pages.get(pageIndex).asByteBuffer().getInt(pageAddress);
        }
        int value = 0;
        for (int i = 0; i < 4; ++i) {
            ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
            value |= (0xFF & buffer.get(pageAddress)) << 8 * (3 - i);
            pageIndex = this.pageIndexFor(++address);
            pageAddress = this.pageAddressFor(address);
        }
        return value;
    }

    public ByteBuffer readBuffer(int address, int length) {
        int pageSize;
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + length <= (pageSize = this.pageSizeFor(pageIndex))) {
            ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
            return ((ByteBuffer)buffer.duplicate().limit(pageAddress + length).position(pageAddress)).slice().asReadOnlyBuffer();
        }
        ByteBuffer data = ByteBuffer.allocate(length);
        while (data.hasRemaining()) {
            ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer().duplicate();
            buffer.position(pageAddress);
            if (buffer.remaining() > data.remaining()) {
                buffer.limit(buffer.position() + data.remaining());
            }
            data.put(buffer);
            pageIndex = this.pageIndexFor(address += buffer.remaining());
            pageAddress = this.pageAddressFor(address);
        }
        return ((ByteBuffer)data.flip()).asReadOnlyBuffer();
    }

    public void writeInt(int address, int value) {
        int pageSize;
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + 4 <= (pageSize = this.pageSizeFor(pageIndex))) {
            this.pages.get(pageIndex).asByteBuffer().putInt(pageAddress, value);
        } else {
            for (int i = 0; i < 4; ++i) {
                ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
                buffer.position(pageAddress);
                buffer.put((byte)(value >> 8 * (3 - i)));
                pageIndex = this.pageIndexFor(++address);
                pageAddress = this.pageAddressFor(address);
            }
        }
    }

    public void writeLong(int address, long value) {
        int pageSize;
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + 8 <= (pageSize = this.pageSizeFor(pageIndex))) {
            this.pages.get(pageIndex).asByteBuffer().putLong(pageAddress, value);
        } else {
            for (int i = 0; i < 8; ++i) {
                ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
                buffer.position(pageAddress);
                buffer.put((byte)(value >> 8 * (7 - i)));
                pageIndex = this.pageIndexFor(++address);
                pageAddress = this.pageAddressFor(address);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void writeBuffer(int address, ByteBuffer data) {
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        int pageSize = this.pageSizeFor(pageIndex);
        if (pageAddress + data.remaining() <= pageSize) {
            ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
            buffer.position(pageAddress);
            buffer.put(data);
        } else {
            while (data.hasRemaining()) {
                ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
                buffer.position(pageAddress);
                if (data.remaining() > buffer.remaining()) {
                    int originalLimit = data.limit();
                    try {
                        data.limit(data.position() + buffer.remaining());
                        address += data.remaining();
                        buffer.put(data);
                    }
                    finally {
                        data.limit(originalLimit);
                    }
                } else {
                    address += data.remaining();
                    buffer.put(data);
                }
                pageIndex = this.pageIndexFor(address);
                pageAddress = this.pageAddressFor(address);
            }
        }
    }

    public void writeBuffers(int address, ByteBuffer[] data) {
        for (ByteBuffer buffer : data) {
            int length = buffer.remaining();
            this.writeBuffer(address, buffer);
            address += length;
        }
    }

    public void free(int address) {
        this.allocator.free(address);
    }

    public void destroy() {
        this.allocator.clear();
        Iterator<Page> it = this.pages.values().iterator();
        while (it.hasNext()) {
            Page p = it.next();
            it.remove();
            this.pageSource.free(p);
        }
        assert (this.validatePages());
    }

    public int allocate(int size) {
        do {
            int address;
            if ((address = this.allocator.allocate(size)) < 0) continue;
            return address;
        } while (this.expandData());
        return -1;
    }

    private boolean expandData() {
        int newPageSize = this.nextPageSize();
        if ((int)this.getAllocatedMemory() + newPageSize < 0) {
            return false;
        }
        Page newPage = this.pageSource.allocate(newPageSize, this.thief, this.victim, this);
        if (newPage == null) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Data area expansion from {} failed", (Object)this.getAllocatedMemory());
            }
            return false;
        }
        if (this.pages.put(this.pages.size(), newPage) != null) {
            this.pageSource.free(newPage);
            assert (this.validatePages());
            throw new AssertionError();
        }
        assert (this.validatePages());
        this.allocator.expand(newPageSize);
        if (LOGGER.isDebugEnabled()) {
            long before = this.getAllocatedMemory();
            long after = before + (long)newPageSize;
            LOGGER.debug("Data area expanded from {}B to {}B [occupation={}]", new Object[]{DebuggingUtils.toBase2SuffixedString(before), DebuggingUtils.toBase2SuffixedString(after), Float.valueOf((float)this.allocator.occupied() / (float)after)});
        }
        return true;
    }

    public long getAllocatedMemory() {
        return this.addressForPage(this.pages.size());
    }

    public long getOccupiedMemory() {
        return this.allocator.occupied();
    }

    public String toString() {
        Page p;
        StringBuilder sb = new StringBuilder("OffHeapStorageArea\n");
        int i = 0;
        while (i < this.pages.size() && (p = this.pages.get(i++)) != null) {
            Page q;
            int size = p.size();
            int count = 1;
            while (i < this.pages.size() && (q = this.pages.get(i)) != null && q.size() == size) {
                ++count;
                ++i;
            }
            sb.append("\t").append(count).append(" ").append(DebuggingUtils.toBase2SuffixedString(size)).append("B page").append(count == 1 ? "\n" : "s\n");
        }
        sb.append("Allocator: ").append(this.allocator).append('\n');
        sb.append("Page Source: ").append(this.pageSource);
        return sb.toString();
    }

    private int pageIndexFor(int address) {
        if (address > this.pageGrowthAreaSize) {
            return (address - this.pageGrowthAreaSize) / this.maximalPageSize + this.pageIndexFor(this.pageGrowthAreaSize);
        }
        return 32 - Integer.numberOfLeadingZeros(address / this.initialPageSize + 1) - 1;
    }

    private int addressForPage(int index) {
        int postIndex = index - this.pageIndexFor(this.pageGrowthAreaSize);
        if (postIndex > 0) {
            return this.pageGrowthAreaSize + this.maximalPageSize * postIndex;
        }
        return (this.initialPageSize << index) - this.initialPageSize;
    }

    private int pageAddressFor(int address) {
        return address - this.addressForPage(this.pageIndexFor(address));
    }

    private int pageSizeFor(int index) {
        if (index < this.pageIndexFor(this.pageGrowthAreaSize)) {
            return this.initialPageSize << index;
        }
        return this.maximalPageSize;
    }

    private int nextPageSize() {
        return this.pageSizeFor(this.pages.size());
    }

    public void validate() {
        this.allocator.validate();
    }

    public void release(int address) {
        Integer lastPage = this.pageIndexFor(address);
        for (int i = this.pages.size() - 1; i > lastPage; --i) {
            Page p = this.pages.remove(i);
            assert (p != null);
            this.allocator.expand(-p.size());
            this.pageSource.free(p);
        }
        assert (this.validatePages());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Collection<Page> release(Collection<Page> targets) {
        Lock ownerLock = this.owner.writeLock();
        if (this.thief || this.owner.isThief()) {
            if (!ownerLock.tryLock()) {
                return Collections.emptyList();
            }
        } else {
            ownerLock.lock();
        }
        try {
            LinkedList<Page> recovered = new LinkedList<Page>();
            LinkedList<Page> freed = new LinkedList<Page>();
            while (freed.size() < targets.size()) {
                int remove = this.allocator.getLastUsedAddress();
                if (remove < 0) {
                    for (int i = this.pages.size() - 1; i >= 0; --i) {
                        Page free = this.pages.get(i);
                        assert (free != null);
                        this.allocator.expand(-free.size());
                        this.pages.remove(i);
                        if (targets.remove(free)) {
                            recovered.add(free);
                            continue;
                        }
                        freed.add(free);
                    }
                    assert (this.validatePages());
                    break;
                }
                int start = this.pageIndexFor(remove + BestFitAllocator.TOP_FOOT_OFFSET) + 1;
                int end = this.pages.size();
                HashMap<Integer, Page> switched = new HashMap<Integer, Page>(end - start);
                for (int i = start; i < end; ++i) {
                    Page free = this.pages.get(i);
                    switched.put(i, free);
                    this.pages.put(i, new Page(free.asByteBuffer(), this));
                    if (targets.remove(free)) {
                        recovered.add(free);
                        continue;
                    }
                    freed.add(free);
                }
                if (!this.owner.evictAtAddress(remove)) {
                    this.pages.putAll(switched);
                    recovered.removeAll(switched.values());
                    freed.removeAll(switched.values());
                    break;
                }
                if (this.pages.size() > start) {
                    throw new AssertionError();
                }
                assert (this.validatePages());
            }
            Iterator freePageSource = freed.iterator();
            for (Page t : targets) {
                int index = this.getIndexForPage(t);
                if (index < 0 || !freePageSource.hasNext()) continue;
                Page f = (Page)freePageSource.next();
                assert (f != t);
                assert (f.size() == t.size());
                ((ByteBuffer)f.asByteBuffer().clear()).put((ByteBuffer)t.asByteBuffer().clear());
                this.pages.put(index, f);
                recovered.add(t);
            }
            assert (this.validatePages());
            while (freePageSource.hasNext()) {
                this.pageSource.free((Page)freePageSource.next());
            }
            LinkedList<Page> linkedList = recovered;
            return linkedList;
        }
        finally {
            ownerLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean shrink() {
        Lock ownerLock = this.owner.writeLock();
        ownerLock.lock();
        try {
            if (this.pages.isEmpty()) {
                boolean bl = false;
                return bl;
            }
            int initialSize = this.pages.size();
            for (Page p : this.release(new LinkedList<Page>(Collections.singletonList(this.pages.get(this.pages.size() - 1))))) {
                this.pageSource.free(p);
            }
            boolean bl = this.pages.size() < initialSize;
            return bl;
        }
        finally {
            ownerLock.unlock();
        }
    }

    private Integer getIndexForPage(Page p) {
        for (Map.Entry<Integer, Page> e : this.pages.entrySet()) {
            if (e.getValue() != p) continue;
            return e.getKey();
        }
        return -1;
    }

    private boolean validatePages() {
        for (int i = 0; i < this.pages.size(); ++i) {
            if (this.pages.get(i) != null) continue;
            ArrayList<Integer> pageIndices = new ArrayList<Integer>(this.pages.keySet());
            Collections.sort(pageIndices);
            throw new AssertionError((Object)("Page Indices " + pageIndices));
        }
        return true;
    }

    public static interface Owner {
        public boolean evictAtAddress(int var1);

        public Lock writeLock();

        public boolean isThief();
    }
}

