/*
 * Decompiled with CFR 0.152.
 */
package com.terracottatech.offheapstore.paging;

import com.terracottatech.offheapstore.paging.Page;
import com.terracottatech.offheapstore.paging.PageSource;
import com.terracottatech.offheapstore.storage.allocator.BestFitAllocator;
import com.terracottatech.offheapstore.util.DebuggingUtils;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class OffHeapStorageArea {
    private static final Logger LOGGER = LoggerFactory.getLogger(OffHeapStorageArea.class);
    private final int pageSize;
    private final int pageAddressMask;
    private final int pageIndexShift;
    private final Owner owner;
    private final PageSource pageSource;
    private final BestFitAllocator allocator;
    private final Map<Integer, Page> pages = new ConcurrentHashMap<Integer, Page>(1, 0.75f, 1);
    private final boolean thief;
    private final boolean victim;

    public OffHeapStorageArea(Owner owner, PageSource pageSource, int pageSize, boolean thief, boolean victim) {
        this.owner = owner;
        this.pageSource = pageSource;
        pageSize = Math.max(BestFitAllocator.MINIMAL_SIZE, pageSize);
        this.pageSize = Integer.bitCount(pageSize) == 1 ? pageSize : Integer.highestOneBit(pageSize) << 1;
        this.pageAddressMask = this.pageSize - 1;
        this.pageIndexShift = Integer.bitCount(this.pageAddressMask);
        this.allocator = new BestFitAllocator(this);
        this.thief = thief;
        this.victim = victim;
    }

    public void clear() {
        this.allocator.clear();
        Iterator<Page> it = this.pages.values().iterator();
        while (it.hasNext()) {
            Page p = it.next();
            it.remove();
            this.pageSource.free(p);
        }
        assert (this.validatePages());
    }

    public int readInt(int address) {
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + 4 <= this.pageSize) {
            return this.pages.get(pageIndex).asByteBuffer().getInt(pageAddress);
        }
        int value = 0;
        for (int i = 0; i < 4; ++i) {
            ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
            value |= (0xFF & buffer.get(pageAddress)) << 8 * (3 - i);
            pageIndex = this.pageIndexFor(++address);
            pageAddress = this.pageAddressFor(address);
        }
        return value;
    }

    public ByteBuffer readBuffer(int address, int length) {
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + length <= this.pageSize) {
            ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
            return ((ByteBuffer)buffer.duplicate().limit(pageAddress + length).position(pageAddress)).slice();
        }
        ByteBuffer data = ByteBuffer.allocate(length);
        while (data.hasRemaining()) {
            ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer().duplicate();
            buffer.position(pageAddress);
            if (buffer.remaining() > data.remaining()) {
                buffer.limit(buffer.position() + data.remaining());
            }
            data.put(buffer);
            pageIndex = this.pageIndexFor(address += buffer.remaining());
            pageAddress = this.pageAddressFor(address);
        }
        return (ByteBuffer)data.flip();
    }

    public void writeInt(int address, int value) {
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + 4 <= this.pageSize) {
            this.pages.get(pageIndex).asByteBuffer().putInt(pageAddress, value);
        } else {
            for (int i = 0; i < 4; ++i) {
                ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
                buffer.position(pageAddress);
                buffer.put((byte)(value >> 8 * (3 - i)));
                pageIndex = this.pageIndexFor(++address);
                pageAddress = this.pageAddressFor(address);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void writeBuffer(int address, ByteBuffer data) {
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + data.remaining() <= this.pageSize) {
            ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
            buffer.position(pageAddress);
            buffer.put(data);
        } else {
            while (data.hasRemaining()) {
                ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
                buffer.position(pageAddress);
                if (data.remaining() > buffer.remaining()) {
                    int originalLimit = data.limit();
                    try {
                        data.limit(data.position() + buffer.remaining());
                        address += data.remaining();
                        buffer.put(data);
                    }
                    finally {
                        data.limit(originalLimit);
                    }
                } else {
                    address += data.remaining();
                    buffer.put(data);
                }
                pageIndex = this.pageIndexFor(address);
                pageAddress = this.pageAddressFor(address);
            }
        }
    }

    public void free(int address) {
        this.allocator.free(address);
    }

    public void destroy() {
        this.allocator.clear();
        Iterator<Page> it = this.pages.values().iterator();
        while (it.hasNext()) {
            Page p = it.next();
            it.remove();
            this.pageSource.free(p);
        }
        assert (this.validatePages());
    }

    public int allocate(int size) {
        do {
            int address;
            if ((address = this.allocator.allocate(size)) < 0) continue;
            return address;
        } while (this.expandData());
        return -1;
    }

    private boolean expandData() {
        Page newPage = this.pageSource.allocate(this.pageSize, this.thief, this.victim, this);
        if (newPage == null) {
            if (LOGGER.isDebugEnabled()) {
                int before = this.pages.size() * this.pageSize;
                LOGGER.debug("Data area expansion from {} failed", (Object)before);
            }
            return false;
        }
        if (this.pages.put(this.pages.size(), newPage) != null) {
            this.pageSource.free(newPage);
            assert (this.validatePages());
            throw new AssertionError();
        }
        assert (this.validatePages());
        this.allocator.expand(this.pageSize);
        if (LOGGER.isDebugEnabled()) {
            int before = this.pages.size() * this.pageSize;
            int after = (this.pages.size() + 1) * this.pageSize;
            LOGGER.debug("Data area expanded from {}B to {}B [occupation={}]", new Object[]{DebuggingUtils.toBase2SuffixedString(before), DebuggingUtils.toBase2SuffixedString(after), Float.valueOf((float)this.allocator.occupied() / (float)after)});
        }
        return true;
    }

    public long getAllocatedMemory() {
        return this.pages.size() * this.pageSize;
    }

    public long getOccupiedMemory() {
        return this.allocator.occupied();
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("OffHeapStorageArea ");
        sb.append(this.pages.size()).append(" ").append(DebuggingUtils.toBase2SuffixedString(this.pageSize)).append("B pages\n");
        sb.append("Allocator: ").append(this.allocator).append('\n');
        sb.append("Page Source: ").append(this.pageSource);
        return sb.toString();
    }

    private int pageIndexFor(int address) {
        return address >>> this.pageIndexShift;
    }

    private int pageAddressFor(int address) {
        return address & this.pageAddressMask;
    }

    public void validate() {
        this.allocator.validate();
    }

    public void release(int address) {
        Integer lastPage = this.pageIndexFor(address);
        for (int i = this.pages.size() - 1; i > lastPage; --i) {
            Page p = this.pages.remove(i);
            assert (p != null);
            this.allocator.expand(-this.pageSize);
            this.pageSource.free(p);
        }
        assert (this.validatePages());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Collection<Page> release(Collection<Page> targets) {
        this.owner.writeLock();
        try {
            LinkedList<Page> recovered = new LinkedList<Page>();
            LinkedList<Page> freed = new LinkedList<Page>();
            while (freed.size() < targets.size()) {
                Page free;
                int i;
                int remove = this.allocator.getLastUsedAddress();
                if (remove < 0) {
                    for (i = this.pages.size() - 1; i >= 0; --i) {
                        free = this.pages.get(i);
                        assert (free != null);
                        this.allocator.expand(-this.pageSize);
                        this.pages.remove(i);
                        if (targets.remove(free)) {
                            recovered.add(free);
                            continue;
                        }
                        freed.add(free);
                    }
                    assert (this.validatePages());
                    break;
                }
                for (i = this.pageIndexFor(remove + BestFitAllocator.TOP_FOOT_OFFSET) + 1; i < this.pages.size(); ++i) {
                    free = this.pages.get(i);
                    this.pages.put(i, new Page(free.asByteBuffer()));
                    if (targets.remove(free)) {
                        recovered.add(free);
                        continue;
                    }
                    freed.add(free);
                }
                this.owner.removeAtAddress(remove);
                assert (this.validatePages());
            }
            Iterator freePageSource = freed.iterator();
            for (Page t : targets) {
                int index = this.getIndexForPage(t);
                if (index < 0) continue;
                Page f = (Page)freePageSource.next();
                assert (f != t);
                assert (f.size() == t.size());
                ((ByteBuffer)f.asByteBuffer().clear()).put((ByteBuffer)t.asByteBuffer().clear());
                this.pages.put(index, f);
                recovered.add(t);
            }
            assert (this.validatePages());
            while (freePageSource.hasNext()) {
                this.pageSource.free((Page)freePageSource.next());
            }
            LinkedList<Page> linkedList = recovered;
            return linkedList;
        }
        finally {
            this.owner.writeUnlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean shrink() {
        this.owner.writeLock();
        try {
            if (this.pages.isEmpty()) {
                boolean bl = false;
                return bl;
            }
            int initialSize = this.pages.size();
            for (Page p : this.release(new LinkedList<Page>(Collections.singletonList(this.pages.get(this.pages.size() - 1))))) {
                this.pageSource.free(p);
            }
            boolean bl = this.pages.size() < initialSize;
            return bl;
        }
        finally {
            this.owner.writeUnlock();
        }
    }

    private Integer getIndexForPage(Page p) {
        for (Map.Entry<Integer, Page> e : this.pages.entrySet()) {
            if (e.getValue() != p) continue;
            return e.getKey();
        }
        return -1;
    }

    private boolean validatePages() {
        for (int i = 0; i < this.pages.size(); ++i) {
            if (this.pages.get(i) != null) continue;
            ArrayList<Integer> pageIndices = new ArrayList<Integer>(this.pages.keySet());
            Collections.sort(pageIndices);
            throw new AssertionError((Object)("Page Indices " + pageIndices));
        }
        return true;
    }

    public static interface Owner {
        public void removeAtAddress(int var1);

        public void writeLock();

        public void writeUnlock();
    }
}

