package net.pubnative.sdk.layouts;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.util.Log;

import net.pubnative.sdk.core.exceptions.PNException;

public abstract class PNLayout extends PNLayoutWaterfall {

    private static final String TAG = PNLayout.class.getSimpleName();

    private boolean mIsRunning = false;
    private Handler mHandler = null;

    //==============================================================================================
    // Listener
    //==============================================================================================
    public interface LoadListener {

        void onPNLayoutLoadFinish(PNLayout layout);

        void onPNLayoutLoadFail(PNLayout layout, Exception exception);

    }

    public interface TrackListener {

        void onPNLayoutTrackImpression(PNLayout layout);

        void onPNLayoutTrackClick(PNLayout layout);
    }

    protected LoadListener  mLoadListener;
    protected TrackListener mTrackListener;

    //==============================================================================================
    // Public methods
    //==============================================================================================
    public void load(Context context, String appToken, String placement) {
        load(context, appToken, placement, mLoadListener);
    }

    public void load(Context context, String appToken, String placement, LoadListener listener) {
        if (listener ==  null) {
            Log.w(TAG, "load - Error: listener not specified, dropping the call");
        } else if (mIsRunning) {
            Log.w(TAG, "load - Error: layout already loading, dropping the call");
        } else {
            mIsRunning = true;
            mLoadListener = listener;
            initialize(context, appToken, placement);
        }
    }

    //==============================================================================================
    // HELPERS
    //==============================================================================================
    protected Handler getMainHandler() {
        if (mHandler == null) {
            mHandler = new Handler(Looper.getMainLooper());
        }
        return mHandler;
    }

    protected void invokeFinish() {
        final LoadListener listener = mLoadListener;
        mLoadListener = null;
        getMainHandler().post(new Runnable() {
            @Override
            public void run() {
                if (listener != null) {
                    mIsRunning = false;
                    listener.onPNLayoutLoadFinish(PNLayout.this);
                }
            }
        });
    }

    protected void invokeFail(final Exception exception) {
        final LoadListener listener = mLoadListener;
        mLoadListener = null;
        getMainHandler().post(new Runnable() {
            @Override
            public void run() {
                if (listener != null) {
                    mIsRunning = false;
                    listener.onPNLayoutLoadFail(PNLayout.this, exception);
                }
            }
        });
    }

    protected void invokeImpression() {
        if (mInsight != null) {
            mInsight.sendImpressionInsight();
        }
        if (mTrackListener != null) {
            mTrackListener.onPNLayoutTrackImpression(this);
        }
    }

    protected void invokeClick() {
        if (mInsight != null) {
            mInsight.sendClickInsight();
        }
        if (mTrackListener != null) {
            mTrackListener.onPNLayoutTrackClick(this);
        }
    }

    public void setLoadListener(LoadListener listener) {
        mLoadListener = listener;
    }

    public void setTrackListener(TrackListener listener) {
        mTrackListener = listener;
    }
}
