package net.peanuuutz.fork.util.common

import org.joml.Math.lerp
import kotlin.contracts.InvocationKind
import kotlin.contracts.contract

public fun Color(
    red: Int,
    green: Int,
    blue: Int,
    alpha: Int = 0xFF
): Color {
    return Color(packColor(red, green, blue, alpha))
}

public fun Color(
    red: Float,
    green: Float,
    blue: Float,
    alpha: Float = 1.0f
): Color {
    return Color(packColor(red, green, blue, alpha))
}

public fun Color(argb: Long): Color {
    return Color(argb.toULong())
}

@JvmInline
public value class Color internal constructor(private val packedLong: ULong) {
    public val red: Int
        get() = unpackRed(packedLong.toLong())

    public val nRed: Float
        get() = red / 255.0f

    public val green: Int
        get() = unpackGreen(packedLong.toLong())

    public val nGreen: Float
        get() = green / 255.0f

    public val blue: Int
        get() = unpackBlue(packedLong.toLong())

    public val nBlue: Float
        get() = blue / 255.0f

    public val alpha: Int
        get() = unpackAlpha(packedLong.toLong())

    public val nAlpha: Float
        get() = alpha / 255.0f

    public val rgb: Int
        get() = (packedLong.toLong() and 0xFFFFFF).toInt()

    public val argb: Int
        get() = (packedLong.toLong() and 0xFFFFFFFF).toInt()

    public operator fun component1(): Int = red

    public operator fun component2(): Int = green

    public operator fun component3(): Int = blue

    public operator fun component4(): Int = alpha

    public fun orElse(default: Color): Color {
        return if (isSpecified()) this else default
    }

    public fun copy(
        alpha: Int = this.alpha,
        red: Int = this.red,
        green: Int = this.green,
        blue: Int = this.blue
    ): Color {
        return Color(red, green, blue, alpha)
    }

    public fun copy(
        alpha: Float = nAlpha,
        red: Float = nRed,
        green: Float = nGreen,
        blue: Float = nBlue
    ): Color {
        return Color(red, green, blue, alpha)
    }

    override fun toString(): String {
        return if (isSpecified()) {
            "Color(red=$red, green=$green, blue=$blue, alpha=$alpha)"
        } else {
            "Color.Unspecified"
        }
    }

    public companion object {
        public val Black: Color = Color(0xFF000000)

        public val DarkGray: Color = Color(0xFF444444)

        public val Gray: Color = Color(0xFF888888)

        public val LightGray: Color = Color(0xFFCCCCCC)

        public val White: Color = Color(0xFFFFFFFF)

        public val Red: Color = Color(0xFFFF0000)

        public val Green: Color = Color(0xFF00FF00)

        public val Blue: Color = Color(0xFF0000FF)

        public val Yellow: Color = Color(0xFFFFFF00)

        public val Magenta: Color = Color(0xFFFF00FF)

        public val Cyan: Color = Color(0xFF00FFFF)

        public val Transparent: Color = Color(0x00FFFFFF)

        public val Unspecified: Color = Color(0x100FFFFFF)

        public fun lerp(
            from: Color,
            to: Color,
            fraction: Float
        ): Color {
            return when {
                from.isUnspecified() && to.isUnspecified() -> Transparent
                from.isUnspecified() -> to.copy(alpha = to.nAlpha * fraction)
                to.isUnspecified() -> from.copy(alpha = from.nAlpha * (1.0f - fraction))
                else -> {
                    Color(
                        red = lerp(from.nRed, to.nRed, fraction),
                        green = lerp(from.nGreen, to.nGreen, fraction),
                        blue = lerp(from.nBlue, to.nBlue, fraction),
                        alpha = lerp(from.nAlpha, to.nAlpha, fraction)
                    )
                }
            }
        }
    }
}

public fun Color.isUnspecified(): Boolean {
    return this == Color.Unspecified
}

public fun Color.isSpecified(): Boolean {
    return isUnspecified().not()
}

public inline fun Color.orElse(provider: () -> Color): Color {
    contract { callsInPlace(provider, InvocationKind.EXACTLY_ONCE) }

    return orElse(provider())
}

public operator fun Color.times(other: Color): Color {
    return when {
        isUnspecified() -> other
        other.isSpecified() -> this
        else -> {
            Color(
                red = nRed * other.nRed,
                green = nGreen * other.nGreen,
                blue = nBlue * other.nBlue,
                alpha = nAlpha * other.nAlpha
            )
        }
    }
}
