/*
 * BEGIN_HEADER - DO NOT EDIT
 * 
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the "License").  You may not use this file except
 * in compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://open-jbi-components.dev.java.net/public/CDDLv1.0.html.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * HEADER in each file and include the License file at
 * https://open-jbi-components.dev.java.net/public/CDDLv1.0.html.
 * If applicable add the following below this CDDL HEADER,
 * with the fields enclosed by brackets "[]" replaced with
 * your own identifying information: Portions Copyright
 * [year] [name of copyright owner]
 */

/*
 * @(#)NotificationEventImpl.java 
 *
 * Copyright 2004-2007 Sun Microsystems, Inc. All Rights Reserved.
 * 
 * END_HEADER - DO NOT EDIT
 */
package com.sun.esb.eventmanagement.impl;

import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.sun.esb.eventmanagement.api.NotificationEvent;

/**
 * This class represents a Notification Event
 *
 * @author bgebbie
 *
 */
public class NotificationEventImpl  extends EventImpl implements NotificationEvent {

    static final long serialVersionUID = 5298513832820910083L;
        


    /////////
    // Fields
    /////////

    /**
     * The notification event id (autogenerated).
     */
    private long mId;

    /**
     * The type of notification:
     *      EVENT_TYPE_ALERT
     *      EVENT_TYPE_TCA
     *      EVENT_TYPE_EVENT
     */
    private String mType;

    /**
     * The severity:
     *      SEVERITY_TYPE_FATAL        0
     *      SEVERITY_TYPE_CRITICAL     1
     *      SEVERITY_TYPE_MAJOR        2
     *      SEVERITY_TYPE_MINOR        3
     *      SEVERITY_TYPE_WARNING      4
     *      SEVERITY_TYPE_INFO         5
     */
    private int mSeverity;
    private String[] severityStrings = {"FATAL","CRITICAL","MAJOR","MINOR","WARNING","INFO"};
    
    /**
     * The operational State:
     *      OPERATIONAL_STATE_UNKNOWN        0
     *      OPERATIONAL_STATE_STARTING        1
     *      OPERATIONAL_STATE_SUSPENDING     2
     *      OPERATIONAL_STATE_SUSPENDED      3
     *      OPERATIONAL_STATE_STOPPING       4
     *      OPERATIONAL_STATE_STOPPED        5
     *      OPERATIONAL_STATE_RUNNING        6
     */
    private int mOperationalState;
    private String[] opStateStrings = { "UNKNOWN", "STARTING", "SUSPENDING", "SUSPENDED", "STOPPING", "STOPPED", "RUNNING"};

    /**
     * The message key.
     */
    private String mMessageCode;

    /**
     * The message code arguments.
     */
    private String[] mMessageCodeArgs;

    /**
     * A non i18n comment such as a stacktrace.
     */
    private String mMessageDetails;

    /**
     * The observation state:
     *      OBSERVATIONAL_STATE_UNOBSERVED      0
     *      OBSERVATIONAL_STATE_OBSERVED        1
     *      OBSERVATIONAL_STATE_RESOLVED        2
     */
    private int mObservationalState;
    private String[] obStateStrings = { "UNOBSERVED", "OBSERVED", "RESOLVED" };

    /**
     * Flag to keep track of whether the listeners have been notified.
     */
    private boolean mListenersNotified;

    /**
     * The logger.
     */
    private static final Logger mLogger;
    private static boolean isDebugEnabled;
    static {
       mLogger = Logger.getLogger(NotificationEventImpl.class.getName());
    
    }

    /**
     * NotificationEvent Constructor.
     */
    public NotificationEventImpl() {
    }
    
    
    public NotificationEventImpl(String componentType,
                 String componentProjectPathName,
                 String componentName,
                 long timeStamp) {
        super(componentType,
              componentProjectPathName,
              componentName,
              timeStamp); 
    }       


    /**
     * NotificationEvent Constructor.
     *
     *
     * @param physicalHostName the name of the physical host
     * @param environmentName the name of the environment
     * @param logicalHostName the name of the logical host
     * @param serverType the type of the server (INTEGRATION, MESSAGE)
     * @param serverName the name of the server
     * @param componentType the type of the component (COLLABORATION_COMPONENT)
     * @param componentProjectPathName full path name of the project in
     *                                 which the component is in. Note the names
     *                                 of the project and its sub-projects, if
     *                                 any, are separated by '/'.
     * @param componentName the name of the component
     *             .
     *             .
     *             .
     *             .
     *             .
     */
    public NotificationEventImpl(
        String physicalHostName,
        String environmentName,
        String logicalHostName,
        String serverType,
        String serverName,
        String componentType,
        String componentProjectPathName,
        String componentName,
        String type,
        int severity,
        int operationalState,
        String messageCode,
        String[] messageCodeArgs,
        String messageDetails
        ) {

        super(physicalHostName,
              environmentName,
              logicalHostName,
              serverType,
              serverName,
              componentType,
              componentProjectPathName,
              componentName,
              new Date().getTime()
        );

        mType = type;
        mSeverity = severity;
        mOperationalState = operationalState;
        mMessageCode = messageCode;
        mMessageCodeArgs = messageCodeArgs;
        mMessageDetails = messageDetails;

        mObservationalState = OBSERVATIONAL_STATE_UNOBSERVED;
        mListenersNotified = false;
    }

    /**
     * The notification event id
     *
     * @return the notification event id
     *
     * @hibernate.id generator-class="hilo.long"
     */
    public long getId() {
        return mId;
    }

    /**
     * Set the id.  This should never be called explicitly as persistance
     * mechanism will override this value.
     *
     * @param id the id
     */
    public void setId(long id) {
        mId = id;
    }

    /**
     * The getter method for the notification event type.
     *
     * @return the event type:
     *      EVENT_TYPE_ALERT
     *      EVENT_TYPE_TCA
     *      EVENT_TYPE_EVENT
     *
     * @hibernate.property
     */
    public String getType() {

        return mType;
    }


    /**
     * The setter method for the notification event type.
     *
     * @param type the notification event type:
     *      EVENT_TYPE_ALERT
     *      EVENT_TYPE_TCA
     *      EVENT_TYPE_EVENT
     */
    public void setType(String type) {

        mType = type;
    }


    /**
     * The getter method for the severity.
     *
     * @return the severity:
     *      SEVERITY_TYPE_FATAL
     *      SEVERITY_TYPE_CRITICAL
     *      SEVERITY_TYPE_MAJOR
     *      SEVERITY_TYPE_MINOR
     *      SEVERITY_TYPE_WARNING
     *      SEVERITY_TYPE_INFO
     *S
     * @hibernate.property
     */
    public int getSeverity() {
        return mSeverity;
    }

    /**
     * @return a String representation of the Severity
     * This String is not localized, it is one of the following:
     * "FATAL", "CRITICAL","MAJOR","MINOR","WARNING","INFO"
     */
    public String getSeverityString(int severity) {
         // perform boundary check
         if (severity>=0 && severity<severityStrings.length ) { 
             return severityStrings[severity] ;
         } else {
             if(isDebugEnabled) {
                 mLogger.log(Level.FINEST, "WARNING: getSeverityStrings: invalid index: {0}", severity);
             }
             return "";  
         }
    }

    /**
     * The setter method for the severity.
     *
     * @param severity the severity:
     *      SEVERITY_TYPE_FATAL
     *      SEVERITY_TYPE_CRITICAL
     *      SEVERITY_TYPE_MAJOR
     *      SEVERITY_TYPE_MINOR
     *      SEVERITY_TYPE_WARNING
     *      SEVERITY_TYPE_INFO
     */
    public void setSeverity(int severity) {
        mSeverity = severity;
    }


    /**
     * The getter method for the details about a notification event.
     *
     * @return  the details about a notification event
     *
     * @hibernate.property
     */
    public String getMessageDetails() {

        return mMessageDetails;
    }


    /**
     * The setter method for the details about a notification event.
     *
     * @param messageDetails the details about a notification event
     */
    public void setMessageDetails(String messageDetails) {

        mMessageDetails = messageDetails;
    }


    /**
     * The getter method for the operational state
     *
     * @return the operational state:
     *      OPERATIONAL_STATE_UNKNOWN
     *      OPERATIONAL_STATE_STARTING
     *      OPERATIONAL_STATE_SUSPENDING
     *      OPERATIONAL_STATE_SUSPENDED
     *      OPERATIONAL_STATE_STOPPING
     *      OPERATIONAL_STATE_STOPPED
     *      OPERATIONAL_STATE_RUNNING
     *
     * @hibernate.property
     */
    public int getOperationalState() {

        return mOperationalState;
    }


    /**
     * @return a String representation of the OperationalState. 
     * This String is not localized, it is one of the following:
     * "UNKNOWN", "STARTING", "SUSPENDING", "SUSPENDED", "STOPPING", "STOPPED", "RUNNING"
     */
    public String getOpStateString(int opState) {
        if ( opState>=0 && opState<opStateStrings.length ) { 
        	return opStateStrings[opState];
        } else {
            if(isDebugEnabled) {
                mLogger.log(Level.FINEST, "WARNING: getOpStateString: invalid index: {0}", opState);
            }
            return "";   
        }
    }
   

    /**
     * The setter method for the operational state.
     *
     * @param operationalState the operational state:
     *      OPERATIONAL_STATE_UNKNOWN
     *      OPERATIONAL_STATE_STARTING
     *      OPERATIONAL_STATE_SUSPENDING
     *      OPERATIONAL_STATE_SUSPENDED
     *      OPERATIONAL_STATE_STOPPING
     *      OPERATIONAL_STATE_STOPPED
     *      OPERATIONAL_STATE_RUNNING
     */
    public void setOperationalState(int operationalState) {

        mOperationalState = operationalState;
    }


    /**
     * The getter method to retrieve the observational state.
     *
     * @return the observational state:
     *      OBSERVATIONAL_STATE_UNOBSERVED
     *      OBSERVATIONAL_STATE_OBSERVED
     *      OBSERVATIONAL_STATE_RESOLVED
     *
     * @hibernate.property
     */
    public int getObservationalState() {
        return mObservationalState;
    }


    /**
     * @return a String representation of the ObservationalState
     * This String is not localized, it is one of the following:
     * "UNOBSERVED", "OBSERVED", "RESOLVED"
     */    
    public String getObStateString(int obState) {
        if ( obState>=0 && obState<obStateStrings.length ) { 
           return obStateStrings[obState];
        } else {
            if(isDebugEnabled) {
                mLogger.log(Level.FINEST, "WARNING: getObStateString: invalid index: {0}", obState);
            }
           return "";
        }
    }

    /**
     * Set the observational state.
     *
     * @param observationalState the observational state:
     *      OBSERVATIONAL_STATE_UNOBSERVED
     *      OBSERVATIONAL_STATE_OBSERVED
     *      OBSERVATIONAL_STATE_RESOLVED
     */
    public void setObservationalState(int observationalState) {
        mObservationalState = observationalState;
    }


    /**
     * The getter method for whether the listeners interested in the
     * notifications event have been notified.  Since a schemaorg_apache_xmlbeans.system failure may
     * occur during this process, all of the notification events will need
     * to be resent when the schemaorg_apache_xmlbeans.system is started up again.  This field keeps
     * track of this.  Possible values are:
     *      TRUE - All the listeners have been notified
     *      FALSE - All the listeners may have not been notified
     * Note that it is possible a listener may be called more then once
     * whenever all the listeners could not be notified in an attempt.
     *
     * @return <code>true</code> if all the listeners have been notified and
     * <code>false</code> if they may have not been notified
     *
     * @hibernate.property
     */
    public boolean getListenersNotified() {

        return mListenersNotified;
    }


    /**
     * The setter method for whether the listeners interested in the
     * notification events have been notified.
     *
     * @param listenersNotified <code>true</code> if all the listeners have
     * been notified and <code>false</code> if they may have not been notified
     */
    public void setListenersNotified(boolean listenersNotified) {

        mListenersNotified = listenersNotified;
    }


    /**
     * The getter method for the message code which is a unique integer that
     * is used to identify the root cause of the notification events as
     * specifically as possible.  It can be used by customer support to
     * easily look up common problems and solutions in a knowledge base.
     * It also helps track the cause of issues and helps duplicate issues
     * to be found.  This code will also be used to look up an i18n String
     * from a resource bundle which will have any message code arguments
     * applied to it.
     *
     * @return the message code
     *
     * @hibernate.property
     */
    public String getMessageCode() {

        return mMessageCode;
    }


    /**
     * The setter method for the message code.
     *
     * @param messageCode the message code
     */
    public void setMessageCode(String messageCode) {

        mMessageCode = messageCode;
    }


    /**
     * The getter method for the message code arguments.
     *
     * @return an array of Strings that represent the arguments to
     * the message template specified by the message code.  These strings
     * will not be internationalized.
     *
     * @hibernate.array role="messageCodeArgs" table="T_MSG_CODE_ARGS"
     * hibernate.collection-key column="id" generator-class="hilo"
     * @hibernate.collection-key column="id"
     * @hibernate.collection-index column="argIndex" type="integer"
     * @hibernate.collection-element column="arg" type="string"
     */
    public String[] getMessageCodeArgs() {

        return mMessageCodeArgs;
    }


    /**
     * The setter method for the message code arguments.
     *
     * @param messageCodeArgs the arguments to the message template specified
     * by the message code.  These strings will not be internationalized.
     */
    public void setMessageCodeArgs(String[] messageCodeArgs) {

        mMessageCodeArgs = messageCodeArgs;
    }


    /**
     * Compare two notification event objects.
     *
     * @param o the other notification event to compare to
     *
     * @return <code>true</code> if they are equivalent and <code>false</code>
     * if they are not
     */
    public boolean equals(Object o) {

        if (this == o) {
            return true;
        }

        if (!(o instanceof NotificationEvent)) {
            return false;
        }

        final NotificationEvent event = (NotificationEvent) o;

        //
        // Put cheapest comparisons first followed by ones must likely to be
        // false.
        //

        // Ignore ID

        if (mListenersNotified != event.getListenersNotified()) {
            return false;
        }

        if (mSeverity != event.getSeverity()) {
            return false;
        }

        if (mObservationalState != event.getObservationalState()) {
            return false;
        }

        if (mOperationalState != event.getOperationalState()) {
            return false;
        }

        if (mMessageCode != null
            ? !mMessageCode.equals(event.getMessageCode())
            : event.getMessageCode() != null) {
            return false;
        }

        if (mType != null
            ? !mType.equals(event.getType())
            : event.getType() != null) {
            return false;
        }

        if (!super.equals(o)) {
            return false;
        }

        if (mMessageDetails != null
            ? !mMessageDetails.equals(event.getMessageDetails())
            : event.getMessageDetails() != null) {
            return false;
        }

        if (mMessageCodeArgs == null) {
            if (event.getMessageCodeArgs() != null) {
                return false;
            }
        } else {
            if (mMessageCodeArgs.length != event.getMessageCodeArgs().length) {
                return false;
            }
           String[] messageCodeArgs = event.getMessageCodeArgs();
            for (int i = 0; i < mMessageCodeArgs.length; i++) {
                if (!mMessageCodeArgs[i].equals(messageCodeArgs[i])) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Generate a hashcode for the notification event.
     *
     * @return the hashcode for the notification event
     */
    public int hashCode() {

        int result = super.hashCode();

        // ID ?

        result = 29 * result + (mType != null ? mType.hashCode() : 0);

        result = 29 * result + mSeverity;

        result = 29 * result + mOperationalState;

        result = 29 * result + (mMessageCode != null
            ? mMessageCode.hashCode()
            : 0);

        if (mMessageCodeArgs == null) {
            result = 29 * result;
        } else {
            for (String mMessageCodeArg : mMessageCodeArgs) {
                result = 29 * result + mMessageCodeArg.hashCode();
            }
        }

        result = 29 * result + (mMessageDetails != null
            ? mMessageDetails.hashCode()
            : 0);

        result = 29 * result + mObservationalState;

        result = 29 * result + (mListenersNotified ? 1 : 0);

        return result;
    }


    /**
     * Generate a string representation of the notification event.
     *
     * @return the string representation of the notification event
     */
    public String toString() {
        String event = super.toString();
        StringBuffer sb = new StringBuffer();

        sb.append("\n\n\n");

        sb.append("\tAlert type:                ");
        sb.append(mType);
        sb.append("\n");

        sb.append("\tSeverity:                  ");
        sb.append(getSeverityString(mSeverity));
        sb.append("\n");

        sb.append("\tEvent ID:                  ");
        sb.append(mId);
        sb.append("\n");

        sb.append("\tOperational state:         ");
        sb.append(getOpStateString(mOperationalState));
        sb.append("\n");

        sb.append("\tMessage code:              ");
        sb.append(mMessageCode);
        sb.append("\n");

        //sb.append("\tMessage CodeArgs:        ");
        //sb.append(mMessageCodeArgs);
        //sb.append("\n");

        sb.append("\tMessage details:           ");
        sb.append(mMessageDetails);
        sb.append("\n");

        sb.append("\tObservational state:       ");
        sb.append(getObStateString(mObservationalState));
        sb.append("\n");

        sb.append("\tListeners notified:        ");
        sb.append(mListenersNotified ? "True": "False");
        sb.append("\n");
        
        // append the base class details
        sb.append(event);

        return sb.toString();
    }

    /**
     *   Create a query for Notifications
     *
     * @param notificationType         the type of notification to query for
     * @param environmentName          the domainHostName to retrieve
     *                                 notifications for. Pass <code>null</code>
     *                                 if this may be any value.
     * @param logicalHostName          the logicalHostName to retrieve
     *                                 nofitications for. Pass <code>null</code>
     *                                 if this may be any value.
     * @param serverType               the type of server to retrieve
     *                                 notifications for. Pass <code>null</code>
     *                                 if this may be any value.
     * @param serverName               the type of server type to retrieve
     *                                 notifications for. Pass <code>null</code>
     *                                 if this may be any value.
     * @param componentType            the componentType to retrieve
     *                                 nofitications for. Pass <code>null</code>
     *                                 if this may be any value.
     * @param componentProjectPathName the full project path name. the full path
     *                                 name of the project in which the
     *                                 component is in. Note the names of the
     *                                 project and its sub-projects, if any, are
     *                                 separated by '/'.
     * @param componentName            the componentName to retrieve
     *                                 nofitications for. Pass <code>null</code>
     *                                 if this may be any value.
     * @param observationalState       the acknowledged state to retrieve
     *                                 nofitications for. Pass <code>null</code>
     *                                 if this may be any value.
     */
    public String createQuery(
        String notificationType,
        String environmentName,
        String logicalHostName,
        String serverType,
        String serverName,
        String componentType,
        String componentProjectPathName,
        String componentName,
        Integer observationalState) {

        StringBuffer stmt = new StringBuffer("select * from NOTIFICATION_EVENT");
        //stmt.append(NotificationEvent.class.getName());

        StringBuffer whereStmt = new StringBuffer();

        int count = 1;
        if (notificationType != null) {
            whereStmt.append("notificationType = '" + notificationType + "'");
            count++;
        }
        if (environmentName != null) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("environmentName = '" + environmentName + "'");
            count++;
        }
        if (logicalHostName != null) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("logicalHostName = '" + logicalHostName + "'");
            count++;
        }
        if (serverType != null) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("serverType = '" + serverType + "'");
            count++;
        }
        if (serverName != null) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("serverName = '" + serverName + "'");
            count++;
        }
        if (componentType != null) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("componentType = '" + componentType + "'");
            count++;
        }
        if (componentProjectPathName != null) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("componentProjectPathName = '"
                             + componentProjectPathName + "'");
            count++;
        }
        if (componentName != null) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("componentName = '" + componentName + "'");
            count++;
        }
        if (observationalState != null) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("observationalState = " + observationalState);
            count++;
        }

        if (count > 1) {
            stmt.append(" where ");
            stmt.append(whereStmt.toString());
        }
        stmt.append(" order by timeStamp desc");

        return stmt.toString();
    }
    
	/**
	 *   Create a query for Notifications
	 *
	 * @param notificationType         the type of notification to query for
	 * @param deploymentName           the deploymentName to retrieve
	 *                                 notifications for. Pass <code>null</code>
	 *                                 if this may be any value.
	 * @param environmentName          the environmentName to retrieve
	 *                                 notifications for. Pass <code>null</code>
	 *                                 if this may be any value.
	 * @param logicalHostName          the logicalHostName to retrieve
	 *                                 nofitications for. Pass <code>null</code>
	 *                                 if this may be any value.
	 * @param serverType               the type of server to retrieve
	 *                                 notifications for. Pass <code>null</code>
	 *                                 if this may be any value.
	 * @param serverName               the type of server type to retrieve
	 *                                 notifications for. Pass <code>null</code>
	 *                                 if this may be any value.
	 * @param componentType            the componentType to retrieve
	 *                                 nofitications for. Pass <code>null</code>
	 *                                 if this may be any value.
	 * @param componentProjectPathName the full project path name. the full path
	 *                                 name of the project in which the
	 *                                 component is in. Note the names of the
	 *                                 project and its sub-projects, if any, are
	 *                                 separated by '/'.
	 * @param componentName            the componentName to retrieve
	 *                                 nofitications for. Pass <code>null</code>
	 *                                 if this may be any value.
	 * @param observationalState       the acknowledged state to retrieve
	 *                                 nofitications for. Pass <code>null</code>
	 *                                 if this may be any value.
	 * @param dateFrom                 include events starting from this date
	 * @param dateTo                   include events ending with this date
	 * @param severity                 severity - CRITICAL, MAJOR, MINOR, WARN, INFO
	 * @param operationalState         status - UNKNOWN, STOPPED, RUNNING
	 * @param messageDetail            the devil is in the details                
	 *      
	 *
	 */
	public String createQuery2(
		String notificationType,
		String deploymentName,
		String environmentName,
		String logicalHostName,
		String serverType,
		String serverName,
		String componentType,
		String componentProjectPathName,
		String componentName,
		Integer observationalState,
		String dateFrom,
		String dateTo,
		Integer severity,
		Integer operationalState,
		String messageDetail
		) {

		StringBuffer stmt = new StringBuffer("select * from NOTIFICATION_EVENT");
		//stmt.append(NotificationEvent.class.getName());

		StringBuffer whereStmt = new StringBuffer();

		int count = 1;
		if (notificationType != null && notificationType.length()>0 ) {
			whereStmt.append("notificationType = '" + notificationType + "'");
			count++;
		}
		if (deploymentName != null && deploymentName.length()>0 ) {
					if (count > 1) {
						whereStmt.append(" and ");
					}
					whereStmt.append("deploymentName = '" + deploymentName + "'");
					count++;
		}	
		if (environmentName != null && environmentName.length()>0 ) {
			if (count > 1) {
				whereStmt.append(" and ");
			}
			whereStmt.append("environmentName = '" + environmentName + "'");
			count++;
		}
		if (logicalHostName != null && logicalHostName.length()>0 ) {
			if (count > 1) {
				whereStmt.append(" and ");
			}
			whereStmt.append("logicalHostName = '" + logicalHostName + "'");
			count++;
		}
		if (serverType != null && serverType.length()>0 ) {
			if (count > 1) {
				whereStmt.append(" and ");
			}
			whereStmt.append("serverType = '" + serverType + "'");
			count++;
		}
		if (serverName != null && serverName.length()>0 ) {
			if (count > 1) {
				whereStmt.append(" and ");
			}
			whereStmt.append("serverName = '" + serverName + "'");
			count++;
		}
		if (componentType != null && componentType.length()>0 ) {
			if (count > 1) {
				whereStmt.append(" and ");
			}
			whereStmt.append("componentType = '" + componentType + "'");
			count++;
		}
		if (componentProjectPathName != null && componentProjectPathName.length()>0 ) {
			if (count > 1) {
				whereStmt.append(" and ");
			}
			whereStmt.append("componentProjectPathName = '"
							 + componentProjectPathName + "'");
			count++;
		}
		if (componentName != null && componentName.length()>0 ) {
			if (count > 1) {
				whereStmt.append(" and ");
			}
			whereStmt.append("componentName = '" + componentName + "'");
			count++;
		}
		if (observationalState != null) {
			if (count > 1) {
				whereStmt.append(" and ");
			}
			whereStmt.append("observationalState = " + observationalState);
			count++;
		}

		if ( dateFrom!=null && dateFrom.length()>0 ) {
		   try {
			  SimpleDateFormat formatter = new SimpleDateFormat("MM/dd/yyyy");
			  ParsePosition pos = new ParsePosition(0);
			  Date date = formatter.parse(dateFrom, pos);
			  long t = date.getTime();
			  if ( count > 1 ) {
				 whereStmt.append(" and ");
			  }
			  whereStmt.append("timeStamp >= " + t);
			  count++;
		   } catch(Exception e) {
                if(mLogger.isLoggable(Level.SEVERE)) {
                    mLogger.log(Level.SEVERE, "Unable to parse dateFrom:{0}", dateFrom);
                }
		   }
		}

		if ( dateTo!=null && dateTo.length()>0 ) {
		   try {
			  SimpleDateFormat formatter = new SimpleDateFormat("MM/dd/yyyy hh:mm:sss");
			  ParsePosition pos = new ParsePosition(0);
			  // make this date to the end of day
			  Date date = formatter.parse(dateTo+" 23:59:59", pos);
			  long t = date.getTime();
			  if ( count > 1 ) {
				 whereStmt.append(" and ");
			  }
			  whereStmt.append("timeStamp <= " + t);
			  count++;
		   } catch(Exception e) {
                if(mLogger.isLoggable(Level.SEVERE)){
                    mLogger.log(Level.SEVERE, "Unable to parse dateTo:{0}", dateTo);
                }
		   }
		}

		if (severity != null) {
			if (count > 1) {
				whereStmt.append(" and ");
			}
			whereStmt.append("severity <= " + severity);
			count++;
		}

		if (operationalState != null) {
			if (count > 1) {
				whereStmt.append(" and ");
			}
			whereStmt.append("operationalState = " + operationalState);
			count++;
		}

		if (messageDetail != null && messageDetail.length()>0 ) {
			if (count > 1) {
				whereStmt.append(" and ");
			}
			whereStmt.append("messageDetail like '" + messageDetail + "%'");
			count++;
		}

		if (count > 1) {
			stmt.append(" where ");
			stmt.append(whereStmt.toString());
		}
		stmt.append(" order by timeStamp desc");
        
		return stmt.toString();
	}
	    
    /**
     *   Create a query for Notifications
     *
     * @param notificationType         the type of notification to query for
     * @param environmentName          the domainHostName to retrieve
     *                                 notifications for. Pass <code>null</code>
     *                                 if this may be any value.
     * @param logicalHostName          the logicalHostName to retrieve
     *                                 nofitications for. Pass <code>null</code>
     *                                 if this may be any value.
     * @param serverType               the type of server to retrieve
     *                                 notifications for. Pass <code>null</code>
     *                                 if this may be any value.
     * @param serverName               the type of server type to retrieve
     *                                 notifications for. Pass <code>null</code>
     *                                 if this may be any value.
     * @param componentType            the componentType to retrieve
     *                                 nofitications for. Pass <code>null</code>
     *                                 if this may be any value.
     * @param componentProjectPathName the full project path name. the full path
     *                                 name of the project in which the
     *                                 component is in. Note the names of the
     *                                 project and its sub-projects, if any, are
     *                                 separated by '/'.
     * @param componentName            the componentName to retrieve
     *                                 nofitications for. Pass <code>null</code>
     *                                 if this may be any value.
     * @param observationalState       the acknowledged state to retrieve
     *                                 nofitications for. Pass <code>null</code>
     *                                 if this may be any value.
     * @param dateFrom                 include events starting from this date
     * @param dateTo                   include events ending with this date
     * @param severity                 severity - CRITICAL, MAJOR, MINOR, WARN, INFO
     * @param operationalState         status - UNKNOWN, STOPPED, RUNNING
     * @param messageDetail            the devil is in the details                
     *      
     *
     */
    public String createQuery2(
        String notificationType,
        String environmentName,
        String logicalHostName,
        String serverType,
        String serverName,
        String componentType,
        String componentProjectPathName,
        String componentName,
        Integer observationalState,
        String dateFrom,
        String dateTo,
        Integer severity,
        Integer operationalState,
        String messageDetail
        ) {

        StringBuffer stmt = new StringBuffer("select * from NOTIFICATION_EVENT");
        //stmt.append(NotificationEvent.class.getName());

        StringBuffer whereStmt = new StringBuffer();

        int count = 1;
        if (notificationType != null && notificationType.length()>0 ) {
            whereStmt.append("notificationType = '" + notificationType + "'");
            count++;
        }
        if (environmentName != null && environmentName.length()>0 ) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("environmentName = '" + environmentName + "'");
            count++;
        }
        if (logicalHostName != null && logicalHostName.length()>0 ) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("logicalHostName = '" + logicalHostName + "'");
            count++;
        }
        if (serverType != null && serverType.length()>0 ) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("serverType = '" + serverType + "'");
            count++;
        }
        if (serverName != null && serverName.length()>0 ) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("serverName = '" + serverName + "'");
            count++;
        }
        if (componentType != null && componentType.length()>0 ) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("componentType = '" + componentType + "'");
            count++;
        }
        if (componentProjectPathName != null && componentProjectPathName.length()>0 ) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("componentProjectPathName = '"
                             + componentProjectPathName + "'");
            count++;
        }
        if (componentName != null && componentName.length()>0 ) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("componentName = '" + componentName + "'");
            count++;
        }
        if (observationalState != null) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("observationalState = " + observationalState);
            count++;
        }

        if ( dateFrom!=null && dateFrom.length()>0 ) {
           try {
              SimpleDateFormat formatter = new SimpleDateFormat("MM/dd/yyyy");
              ParsePosition pos = new ParsePosition(0);
              Date date = formatter.parse(dateFrom, pos);
              long t = date.getTime();
              if ( count > 1 ) {
                 whereStmt.append(" and ");
              }
              whereStmt.append("timeStamp >= " + t);
              count++;
           } catch(Exception e) {
               if(mLogger.isLoggable(Level.SEVERE)){
                   mLogger.log(Level.SEVERE, "Unable to parse dateFrom:{0}", dateFrom);
               }
           }
        }

        if ( dateTo!=null && dateTo.length()>0 ) {
           try {
              SimpleDateFormat formatter = new SimpleDateFormat("MM/dd/yyyy hh:mm:sss");
              ParsePosition pos = new ParsePosition(0);
              // make this date to the end of day
              Date date = formatter.parse(dateTo+" 23:59:59", pos);
              long t = date.getTime();
              if ( count > 1 ) {
                 whereStmt.append(" and ");
              }
              whereStmt.append("timeStamp <= " + t);
              count++;
           } catch(Exception e) {
               if(mLogger.isLoggable(Level.SEVERE)){
                   mLogger.log(Level.SEVERE, "Unable to parse dateTo:{0}", dateTo);
               }
           }
        }

        if (severity != null) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("severity <= " + severity);
            count++;
        }

        if (operationalState != null) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("operationalState = " + operationalState);
            count++;
        }

        if (messageDetail != null && messageDetail.length()>0 ) {
            if (count > 1) {
                whereStmt.append(" and ");
            }
            whereStmt.append("messageDetail like '" + messageDetail + "%'");
            count++;
        }

        if (count > 1) {
            stmt.append(" where ");
            stmt.append(whereStmt.toString());
        }
        stmt.append(" order by timeStamp desc");
        
        return stmt.toString();
    }
    
}    

