package net.minidev.json.parser;

/**
 * JSONParserBase is the common code between {@link JSONParser} and
 * {@link JSONParserStream}
 * 
 * @see JSONParser
 * @see JSONParserStream
 * 
 * @author Uriel Chemouni <uchemouni@gmail.com>
 */
public abstract class JSONParserBase {
	protected static boolean[] stopArray = new boolean[126];
	protected static boolean[] stopKey = new boolean[126];
	protected static boolean[] stopValue = new boolean[126];
	protected static boolean[] stopX = new boolean[126];
	static {
		stopKey[':'] = true;
		stopValue[','] = stopValue['}'] = true;
		stopArray[','] = stopArray[']'] = true;
	}
	/**
	 * allow simple quote as String quoting char
	 */
	public final static int ACCEPT_SIMPLE_QUOTE = 1;
	/**
	 * allow non quoted test
	 */
	public final static int ACCEPT_NON_QUOTE = 2;
	/**
	 * Parse NaN as Float.NaN
	 */
	public final static int ACCEPT_NAN = 4;
	/**
	 * Ignore control char in input text.
	 */
	public final static int IGNORE_CONTROL_CHAR = 8;
	/**
	 * Use int datatype to store number when it's possible.
	 * 
	 * @since 1.0.7
	 */
	public final static int USE_INTEGER_STORAGE = 16;
	/**
	 * Throws exception on excessive 0 leading in digits
	 * 
	 * @since 1.0.7
	 */
	public final static int ACCEPT_LEADING_ZERO = 32;
	/**
	 * Throws exception on useless comma in object and array
	 * 
	 * @since 1.0.8
	 */
	public final static int ACCEPT_USELESS_COMMA = 64;

	/**
	 * smart mode, fastest parsing mode. accept lots of non standard json syntax
	 * 
	 * @since 1.0.6
	 */
	public final static int MODE_PERMISSIVE = -1;
	/**
	 * strict RFC4627 mode.
	 * 
	 * slower than PERMISIF MODE.
	 * 
	 * @since 1.0.6
	 */
	public final static int MODE_RFC4627 = USE_INTEGER_STORAGE;
	/**
	 * Parse Object like json-simple
	 * 
	 * Best for an iso-bug json-simple API port.
	 * 
	 * @since 1.0.7
	 */
	public final static int MODE_JSON_SIMPLE = ACCEPT_USELESS_COMMA;

	/*
	 * End of static declaration
	 */
	protected ContainerFactory containerFactory;
	protected ContentHandler handler = ContentHandlerDumy.HANDLER;
	protected final MSB sb = new MSB(15);
	protected String xs;
	protected Object xo;
	protected int pos;

	/*
	 * Parssing flags
	 */
	protected final boolean acceptSimpleQuote;
	protected final boolean acceptNonQuote;
	protected final boolean acceptNaN;
	protected final boolean ignoreControlChar;
	protected final boolean useIntegerStorage;
	protected final boolean acceptLeadinZero;
	protected final boolean acceptUselessComma;

	public void setHandler(ContentHandler handler) {
		this.handler = handler;
	}

	public JSONParserBase(int permissiveMode) {
		this.acceptNaN = (permissiveMode & ACCEPT_NAN) > 0;
		this.acceptNonQuote = (permissiveMode & ACCEPT_NON_QUOTE) > 0;
		this.acceptSimpleQuote = (permissiveMode & ACCEPT_SIMPLE_QUOTE) > 0;
		this.ignoreControlChar = (permissiveMode & IGNORE_CONTROL_CHAR) > 0;
		this.useIntegerStorage = (permissiveMode & USE_INTEGER_STORAGE) > 0;
		this.acceptLeadinZero = (permissiveMode & ACCEPT_LEADING_ZERO) > 0;
		this.acceptUselessComma = (permissiveMode & ACCEPT_USELESS_COMMA) > 0;
	}

	public void checkControleChar() throws ParseException {
		if (ignoreControlChar)
			return;
		int l = xs.length();
		for (int i = 0; i < l; i++) {
			char c = xs.charAt(i);
			if (c < 0)
				continue;
			if (c <= 31)
				throw new ParseException(pos + i, ParseException.ERROR_UNEXPECTED_CHAR, c);
			if (c == 127)
				throw new ParseException(pos + i, ParseException.ERROR_UNEXPECTED_CHAR, c);
		}
	}

	public static class MSB {
		char b[];
		int p;

		public MSB(int size) {
			b = new char[size];
			p = -1;
		}

		public void append(char c) {
			p++;
			if (b.length <= p) {
				char[] t = new char[b.length * 2 + 1];
				System.arraycopy(b, 0, t, 0, b.length);
				b = t;
			}
			b[p] = c;
		}

		public void append(int c) {
			p++;
			if (b.length <= p) {
				char[] t = new char[b.length * 2 + 1];
				System.arraycopy(b, 0, t, 0, b.length);
				b = t;
			}
			b[p] = (char) c;
		}

		public String toString() {
			return new String(b, 0, p + 1);
		}

		public void clear() {
			p = -1;
		}
	}
}
