package net.kafujo.samples.wikidata.parser;

import java.time.DateTimeException;
import java.time.ZoneId;
import java.time.zone.ZoneRulesException;
import java.util.Objects;

public class TimeZoneParser extends WikidataParser<ZoneId> {

    public static final TimeZoneParser DEFAULT = new TimeZoneParser();

    @Override
    public ZoneId parseFbk(final String raw, ZoneId fbk) {
        try {
            return parse(raw);
        } catch (Exception any) {
            lgr.debug("could not get timezone from '{}': {}", raw, any.toString());
            fallbackCounter.merge(raw, 1, Integer::sum);
        }
        return fbk;
    }

    /**
     * @param raw
     * @return
     * @throws DateTimeException  if the zone ID has an invalid format
     * @throws ZoneRulesException if the zone ID is a region ID that cannot be found
     */
    @Override
    public ZoneId parse(final String raw) {
        Objects.requireNonNull(raw, "REQUIRE raw input to parse time zone");
        if (raw.isBlank()) {
            throw new IllegalArgumentException("Need not blank raw input to parse time zone");
        }

        String clean = raw.strip();

        switch (clean) {
            case "Pacific Time Zone":
                clean = "UTC-08:00";
                break;
            case "Mountain Time Zone":
                clean = "UTC-07:00";
                break;
            case "Central Time Zone":
                clean = "UTC-06:00";
                break;
            case "Eastern Time Zone":
            case "Pakistan Standard Time":
                clean = "UTC-05:00";
                break;
            case "Atlantic Time Zone":
                clean = "UTC-04:00";
                break;
            case "Central European Summer Time":
                clean = "UTC-02:00";
                break;
            case "Greenwich Mean Time":
            case "Western European Time":
                clean = "UTC+00:00";
                break;
            case "Central European Time":
                clean = "UTC+01:00";
                break;
            case "Eastern European Time":
            case "Central Africa Time":
            case "Israel time zone":
                clean = "UTC+02:00";
                break;
            case "Eastern European Summer Time":
            case "East Africa Time":
            case "Moscow Time":
                clean = "UTC+03:00";
                break;
            case "Indian Standard Time":
                clean = "UTC+05:30";
                break;
            case "بث +00:00":
                clean = "UTC+04:30";
                break;
            case "Nepal Standard Time":
                clean = "UTC+05:45";
                break;

            case "Indonesia Western Standard Time":
            case "Bangladesh Standard Time":
            case "Indochina Time":
                clean = "UTC+07:00";
                break;
            case "Singapore Standard Time":
            case "time in Malaysia":
            case "Taiwan time":
                clean = "UTC+08:00";
                break;
            case "Japan Standard Time":
                clean = "UTC+09:00";
                break;
        }

        if (clean.equals(raw.strip())) {
            clean = clean.replace("UTC−", "UTC-");
            clean = clean.replace("UTC±", "UTC-");
        }

        if (!clean.equals(raw)) {
            lgr.debug("fixed timezone from '" + raw + "' to '" + clean + "'");
        }
        return ZoneId.of(clean);
    }
}
