package net.jangaroo.smartsprites.maven;

/*
* Copyright 2012 CoreMedia AG
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing,
* software distributed under the License is distributed on an "AS
* IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
* express or implied. See the License for the specific language
* governing permissions and limitations under the License.
*/

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;

import org.carrot2.labs.smartsprites.SmartSpritesParameters;
import org.carrot2.labs.smartsprites.SpriteBuilder;
import org.carrot2.labs.smartsprites.message.Message;
import org.carrot2.labs.smartsprites.message.MessageLog;
import org.carrot2.labs.smartsprites.message.PrintStreamMessageSink;

import java.io.IOException;
import java.util.List;

/**
 * Goal which creates Spritesheets from given css and image files
 * Explanation of every variable is taken directly from the smartsprites documentation
 * (http://csssprites.org/)
 *
 * For further information on the use of smartsprites please refer to: http://csssprites.org/
 *
 * @goal createSprites
 * @phase generate-resources
 */
public class SmartSpritesMojo
        extends AbstractMojo {

  /** Variables **/

  /**
   *  directory in which SmartSprites processing should be done, required if css-files not specified or if
   *  output-dir-path specified, default: not specified.
   *
   *  SmartSprites will process all files with the *.css extension found in root-dir-path or any subdirectory of it.
   *  For more fine-grained control over the processed CSS files, see the css-files option.
   *
   *  If the provided root directory path is relative, it will be resolved against the current working directory.
   *
   * @parameter expression="${smart.rootDirPath}" default-value="src/main/sprites"
   */
  private String rootDirPath;

  /**
   * paths of CSS files to process, required if root-dir-path not specified, default: not specified.
   *
   *  SmartSprites will process all CSS files listed using this option.
   *  If css-files is to be used together with output-dir-path, root-dir-path must also be specified so that
   *  SmartSprites can preserve the directory structure found in root-dir-path in output-dir-path.
   *  If root-dir-path and output-dir-path are used, css-files outside of root-dir-path will be ignored.
   *
   *  Relative CSS file paths provided using this option will be resolved against the current working directory.
   *  Please note that SmartSprites will not expand any wildcards (like style/*.css),
   *  it assumes the expansion is performed at the command line shell level.
   *
   *  To specify the list of CSS files to process in the SmartSprites Ant task,
   *  use one or more nested fileset elements. Please see the build.xml file in the distribution archive for an example.
   *
   *  @parameter expression="${smart.cssFiles}"
   */
  private List<String> cssFiles;

  /**
   *  output directory for processed CSS files and CSS-relative sprite images, optional, default: not specified.
   *
   *  If a non-empty output-dir-path is specified, a non-empty root-dir-path must also be provided.
   *  The directory structure relative to the root-dir-path will be preserved in the output directory.
   *  E.g. if CSS files are contained in the css/base directory of root-dir-path,
   *  the processed results will be written to output-dir-path/css/base.
   *  Also, CSS-relative sprite images will be written to the output directory.
   *  Sprite images with document-root-relative URLs will be written relative to the document-root-dir-path.
   *  If the output-dir-path directory does not exist, it will be created.
   *  If the provided output directory path is relative, it will be resolved against the current working directory.
   *
   *  You can leave the output-dir-path empty, in which case the CSS files will be written next to the
   *  original CSS files with the css-file-suffix, and sprite images will be written relative to the original CSS files.
   *  If you are using a non-empty output-dir-path, you might want to use an empty css-file-suffix.
   *
   *  @parameter expression="${smart.outPutDirPath}" default-value="${basedir}/target/generated-resources/META-INF/resources/spritesheets"
   */
  private String outPutDirPath;

  /**
   *  Document root path for document-root-relative (starting with /) image urls in CSS, optional, default: not specified.
   *
   *  All document-root-relative image and sprite URLs will be taken relative to document-root-dir-path.
   *  Also document-root-relative sprite URLs will be written relative to document-root-dir-path.
   *  You can leave this property empty if your CSS uses only CSS-relative image URLs.
   *  If the provided document root directory path is relative,
   *  it will be resolved against the current working directory.
   *
   *  @parameter expression="${smart.documentRootDirPath}
   */
  private String documentRootDirPath;

  /**
   *  Message logging level, optional, default: INFO.
   *
   *  Messages less important than log-level will not be shown.
   *  SmartSprites has 3 levels of log messages (in the increasing order of importance):
   *  INFO: information messages, can be safely ignored
   *  IE6NOTICE: notices related to possible quality loss when creating IE6-friendly sprite images,
   *  see also the IE6-friendly PNG option
   *  WARN: warnings related to syntax, IO and sprite rendering quality loss problems
   *  that may cause the converted sprite-based designs look broken
   *
   *  @parameter  expression="${smart.logLevel}" default-value="WARN"
   */
  private String logLevel;

  /**
   *  Color depth of sprites in the PNG format, optional, default: AUTO.
   *  AUTO: PNG color depth will be chosen automatically. If the sprite image does not contain partial transparencies
   *  (alpha channel) and has less than 256 colors, PNG8 will be used. Otherwise, the sprite will be saved in PNG24.
   *  DIRECT: PNG sprites will always be saved in the PNG24 format.
   *  INDEXED: PNG sprites will always be saved in the PNG8 format.
   *  If the sprite image contains partial transparencies (alpha channel) or has more than 255 colors,
   *  image quality loss may occur and appropriate warnings will be issued. See also the sprite-matte-color property.
   *
   *  @parameter expression="${smart.spritePngDepth}" default-value="AUTO"
   */
  private String spritePngDepth;

  /**
   *  Enables generation of IE6-friendly sprite images, optional, default: disabled.
   *
   *  If sprite-png-ie6 is specified, for each PNG sprite image with partial transparencies (alpha channel) or
   *  more than 255 colors and any transparencies, SmartSprites will generate a corresponding
   *  color-reduced PNG8 file for IE6.
   *
   *  An extra IE6-only CSS rule will be added to the generated CSS file to ensure
   *  that IE6 (and only IE6) uses the color-reduced version:
   *  #web { width: 17px; height: 17px; background-repeat: no-repeat;
   *  background-image: url('../img/mysprite.png');
   *  -background-image: url('../img/mysprite-ie6.png');
   *  background-position: left -0px; }
   *
   *  See also the sprite-matte-color property.
   *
   *  @parameter expression="${smart.spritePngIeSix}" default-value="false"
   */
  private boolean spritePngIeSix;

  /**
   *  The encoding to assume for input and output CSS files, default: UTF-8.
   *  For the list of allowed values, please see the list of encodings supported in Java.
   *
   *  @parameter expression="${smart.cssFileEncoding}" default-value="UTF-8"
   */
  private String cssFileEncoding;

  /**
   *  Suffix to be appended to the processed CSS file name, optional, default: .
   *
   * @parameter expression="${smart.cssFileSuffix}" default-value=""
   */
  private String cssFileSuffix;

  public void execute()
          throws MojoExecutionException {

    /*
        public SmartSpritesParameters(String rootDir, List<String> cssFiles,
        String outputDir, String documentRootDir, MessageLevel logLevel,
        String cssFileSuffix, PngDepth spritePngDepth, boolean spritePngIe6,
        String cssEncoding)
     */
    Message.MessageLevel msgLogLevel;
    try{
      msgLogLevel = Message.MessageLevel.valueOf(logLevel.toUpperCase());
    } catch (Exception e) {
      throw new MojoExecutionException("LogLevel Error: ", e);
    }

    SmartSpritesParameters.PngDepth pngDepth;
    try{
      pngDepth = SmartSpritesParameters.PngDepth.valueOf(spritePngDepth);
    } catch (Exception e) {
      throw new MojoExecutionException("PngDepth Error: ", e);
    }

    SmartSpritesParameters smartParameters;

    smartParameters = new SmartSpritesParameters(
            rootDirPath,
            cssFiles,
            outPutDirPath,
            documentRootDirPath,
            msgLogLevel,
            cssFileSuffix,
            pngDepth,
            spritePngIeSix,
            cssFileEncoding);

    final MessageLog messageLog = new MessageLog(new PrintStreamMessageSink(
            System.out, smartParameters.getLogLevel()));
    SpriteBuilder spriteBuilder;

    try{
      spriteBuilder = new SpriteBuilder(smartParameters, messageLog);
      spriteBuilder.buildSprites();
    } catch (IOException e) {
      throw new MojoExecutionException("Smartsprites error: ", e);
    }
  }
}