/*
 * Copyright (c) 2006 Israfil Consulting Services Corporation
 * Copyright (c) 2006 Christian Edward Gruber
 * All Rights Reserved
 * 
 * This software is licensed under the Berkeley Standard Distribution license,
 * (BSD license), as defined below:
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this 
 *    list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of Israfil Consulting Services nor the names of its contributors 
 *    may be used to endorse or promote products derived from this software without 
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND 
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
 * IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, 
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
 * OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
 * OF SUCH DAMAGE.
 * 
 * $Id: AbstractFlexModule.java 529 2007-09-05 21:34:22Z christianedwardgruber $
 */
package net.israfil.mojo.flex2;

import java.util.Iterator;
import java.util.Set;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;

/**
 * A base implementation of an {@link FlexModule}.
 *
 * @author <a href="snicoll@apache.org">Stephane Nicoll</a>
 * @author <a href="cgruber@israfil.net">Christian Edward Gruber</a>
 * @version $Id: AbstractFlexModule.java 529 2007-09-05 21:34:22Z christianedwardgruber $
 */
public abstract class AbstractFlexModule implements FlexModule {

	/**
	 * The artifact this module wraps.
	 */
    private Artifact artifact = null;

    // Those are set by the configuration

	/**
	 * The groupId for this artifact
	 */
    private String groupId = null;

	/**
	 * The artifactId for this artifact
	 */
    private String artifactId = null;

    /**
	 * The classifier for this artifact
	 */
    private String classifier = null;

	/**
	 * The target folder into which this module's artifact will be deployed
	 */
    protected String targetPath = null;

	/**
	 * The name by which this module will be deployed
	 */
    protected String targetFilename = null;

    public AbstractFlexModule() { }

    public AbstractFlexModule( Artifact artifact ) {
        this.artifact = artifact;
        this.artifactId = artifact.getArtifactId();
        this.groupId = artifact.getGroupId();
        this.classifier = artifact.getClassifier();
    }

    public void resolveArtifact( MavenProject project, Set artifacts ) throws MojoFailureException {

        if ( artifact != null ) return; // succeed fast.
        if ( groupId == null ) groupId = project.getGroupId();
        if ( artifactId == null ) throw new MojoFailureException( "No artifact specified" );
        Iterator i = artifacts.iterator();
        while (i.hasNext()) {
        	Artifact a = (Artifact) i.next();
        	if (a.getGroupId().equals(groupId) && a.getArtifactId().equals(artifactId))
        		this.artifact = a;
        }
        if (artifact == null) 
        	throw new MojoFailureException("Could not find artifact: " + groupId + ":" 
        			+ artifactId + " from declared dependencies" );
        if (targetFilename == null) targetFilename = artifact.getFile().getName();
    }

    public Artifact getArtifact() { return this.artifact; }

  
    /**
     * Returns the groupId of the artifact wrapped by this module.
     */
    public String getGroupId()  { return groupId; }

    /**
     * Returns the artifactId of the artifact wrapped by this module.
     */
    public String getArtifactId() { return artifactId; }

    /**
     * Returns the classifier (if any) of the artifact wrapped by this module.
     */
    public String getClassifier() { return classifier; }

    /**
     * Returns the directory into which this artifact should be deployed.
     */
    public String getTargetPath() {
    		return targetPath;
    }

    /**
     * Returns the filename by which this artifact should be deployed
     */
    public String getTargetFilename() {
    	return targetFilename;
    }

  
    public String toString()
    {
    	return "FlexModule[" + groupId + ":" + artifactId + 
			((classifier!=null) ? (":" + classifier) : "") +
		    ((artifact!=null) ? (":" + artifact.getVersion()) : "") +
		    ":" + this.getType() +"]";
    }

}
