/*
 * #%L
 * ImageJ software for multidimensional image processing and analysis.
 * %%
 * Copyright (C) 2009 - 2019 Board of Regents of the University of
 * Wisconsin-Madison, Broad Institute of MIT and Harvard, and Max Planck
 * Institute of Molecular Cell Biology and Genetics.
 * %%
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * #L%
 */

package net.imagej.axis;

/**
 * Implement an axis that uses Chapman and Richards method of scaling. This axis
 * based on ImageJ 1.x's CurveFitter algorithm.
 * 
 * @author Barry DeZonia
 */
public class ChapmanRichardsAxis extends Variable3Axis {

	// -- constructor --

	public ChapmanRichardsAxis(AxisType type, String unit, double a, double b,
		double c)
	{
		super(type, unit, a, b, c);
	}

	// -- CalibratedAxis methods --

	@Override
	public double calibratedValue(double rawValue) {
		return Math.pow(a() * (1 - Math.exp(-b() * rawValue)), c());
	}

	@Override
	public double rawValue(double calibratedValue) {
		return Math.log(1 - Math.pow(calibratedValue / a(), 1 / c())) / -b();
	}

	@Override
	public String generalEquation() {
		return "a*(1-exp(-b*x))^c";
	}

	@Override
	public ChapmanRichardsAxis copy() {
		return new ChapmanRichardsAxis(type(), unit(), a(), b(), c());
	}
}
