package name.remal.reflection;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;
import java.security.CodeSource;
import java.security.ProtectionDomain;
import java.util.Optional;
import java.util.function.Function;

import static java.lang.Thread.currentThread;
import static name.remal.SneakyThrow.sneakyThrow;
import static name.remal.UncheckedCast.uncheckedCast;
import static name.remal.reflection.ExtendedURLClassLoader.LoadingOrder.PARENT_ONLY;
import static name.remal.reflection.ExtendedURLClassLoader.LoadingOrder.THIS_ONLY;

@SuppressWarnings("JavaReflectionMemberAccess")
public class ClassLoaderUtils {

    static class ClassLoaderWrapper extends ClassLoader {
        public ClassLoaderWrapper(@NotNull ClassLoader classLoader) {
            super(classLoader);
        }

        @Nullable
        public Package getPackageOrNull(@NotNull String name) {
            return this.getPackage(name);
        }
    }

    @Nullable
    public static Package getPackageOrNull(@NotNull ClassLoader classLoader, @NotNull String packageName) {
        return new ClassLoaderWrapper(classLoader).getPackageOrNull(packageName);
    }

    private static final Method ADD_URL_METHOD;

    static {
        try {
            ADD_URL_METHOD = URLClassLoader.class.getDeclaredMethod("addURL", URL.class);
            ADD_URL_METHOD.setAccessible(true);
        } catch (NoSuchMethodException e) {
            throw sneakyThrow(e);
        }
    }

    public static void addURLsToClassLoader(@NotNull ClassLoader classLoader, @NotNull URL... urls) {
        if (0 == urls.length) return;

        ClassLoader systemClassLoader = ClassLoader.getSystemClassLoader();
        do {
            if (classLoader instanceof URLClassLoader) {
                synchronized (classLoader) {
                    for (URL url : urls) {
                        try {
                            ADD_URL_METHOD.invoke(classLoader, url);
                        } catch (@NotNull IllegalAccessException | InvocationTargetException e) {
                            throw sneakyThrow(e);
                        }
                    }
                }
                return;
            }
            if (systemClassLoader == classLoader) break;
            classLoader = classLoader.getParent();
            if (null == classLoader) classLoader = systemClassLoader;
        } while (true);

        throw new IllegalStateException("New URL can't be added to system ClassLoader: " + systemClassLoader);
    }

    public static <T, R> R forInstantiated(@NotNull ClassLoader classLoader, @NotNull Class<T> type, @NotNull Class<? extends T> implementationType, @NotNull Function<T, R> action) {
        if (!type.isAssignableFrom(implementationType) || type == implementationType) throw new IllegalArgumentException(implementationType + " is not subtype of " + type);

        String implTypeName = implementationType.getName();
        String implTypeNamePrefix = implTypeName + '$';

        URL sourceURL = Optional.ofNullable(implementationType.getProtectionDomain()).map(ProtectionDomain::getCodeSource).map(CodeSource::getLocation).orElseThrow(() -> new IllegalStateException(implementationType + ": null == protectionDomain?.codeSource?.location"));
        try (URLClassLoader childClassLoader = new ExtendedURLClassLoader(
            className -> className.equals(implTypeName) || className.startsWith(implTypeNamePrefix) ? THIS_ONLY : PARENT_ONLY,
            new URL[]{sourceURL},
            classLoader
        )) {
            Thread currentThread = currentThread();
            ClassLoader prevContextClassLoader = currentThread.getContextClassLoader();
            currentThread.setContextClassLoader(classLoader);
            try {
                T implementation = uncheckedCast(childClassLoader.loadClass(implementationType.getName()).newInstance());
                return action.apply(implementation);

            } finally {
                currentThread.setContextClassLoader(prevContextClassLoader);
            }

        } catch (Exception e) {
            throw sneakyThrow(e);
        }
    }

}
