package name.remal

import java.io.File
import java.lang.System.currentTimeMillis
import java.nio.file.Path
import java.util.zip.Deflater.BEST_COMPRESSION
import java.util.zip.ZipEntry
import java.util.zip.ZipOutputStream

val TEMP_DIR: File get() = File(System.getProperty("java.io.tmpdir")).absoluteFile.createDirectories()
val WORKING_DIR: File get() = File(System.getProperty("user.dir")).absoluteFile
val USER_HOME_DIR: File get() = File(System.getProperty("user.home")).absoluteFile

@JvmOverloads
fun newTempFile(prefix: String = "temp-", suffix: String = ".temp", doDeleteOnExit: Boolean = true, baseTempDir: File = TEMP_DIR): File {
    val now = currentTimeMillis()
    var counter = 0
    while (true) {
        val file = File(baseTempDir, encodeURIComponent("$prefix$now-$counter$suffix"))
        if (file.createNewFile()) {
            return file.absoluteFile.also {
                if (doDeleteOnExit) it.deleteOnExit()
            }
        }
        ++counter
    }
}

@JvmOverloads
fun newTempDir(prefix: String = "temp-", doDeleteOnExit: Boolean = true, baseTempDir: File = TEMP_DIR): File {
    val now = currentTimeMillis()
    var counter = 0
    while (true) {
        val dir = File(baseTempDir, encodeURIComponent("$prefix$now-$counter"))
        if (dir.mkdir()) {
            return dir.absoluteFile.also {
                if (doDeleteOnExit) it.deleteOnExitRecursively()
            }
        }
        ++counter
    }
}


fun File.forceDelete() = apply { toPath().delete() }
fun File.forceDeleteRecursively() = apply { toPath().deleteRecursively() }

fun File.createDirectories() = apply { toPath().createDirectories() }
fun File.createParentDirectories() = apply { toPath().createParentDirectories() }

fun File.permitRead() = apply { setReadable(true, false) }
fun File.permitWrite() = apply { setWritable(true, false) }
fun File.permitExecute() = apply { setExecutable(true, false) }
fun File.permitAll() = permitRead().permitWrite().permitExecute()

fun File.allowReadOnlyForOwner() = apply { setReadable(false, false); setReadable(true, true) }
fun File.allowWriteOnlyForOwner() = apply { setWritable(false, false); setWritable(true, true) }
fun File.allowExecuteOnlyForOwner() = apply { setExecutable(false, false); setExecutable(true, true) }
fun File.allowAllOnlyForOwner() = allowReadOnlyForOwner().allowWriteOnlyForOwner().allowExecuteOnlyForOwner()

fun File.forbidRead() = apply { setReadable(false, false) }
fun File.forbidWrite() = apply { setWritable(false, false) }
fun File.forbidExecute() = apply { setExecutable(false, false) }
fun File.forbidAll() = forbidRead().forbidWrite().forbidExecute()

inline fun File.forSelfAndEachParent(action: (file: File) -> Unit) {
    var file: File? = this.absoluteFile
    while (null != file) {
        action(file)
        file = file.parentFile
    }
}


private fun File.zipToImpl(target: File, overwrite: Boolean, zipOnlyContent: Boolean) {
    if (!this.exists()) throw NoSuchFileException(file = this, reason = "The source file doesn't exist.")

    if (target.exists()) {
        val stillExists = if (!overwrite) true else !target.deleteRecursively()
        if (stillExists) {
            throw FileAlreadyExistsException(file = this, other = target, reason = "The destination file already exists.")
        }
    }

    target.createParentDirectories().outputStream().use {
        ZipOutputStream(it).use { zipOutputStream ->
            zipOutputStream.setLevel(BEST_COMPRESSION)

            if (isDirectory) {
                val entryNamePrefix: String
                if (!zipOnlyContent) {
                    entryNamePrefix = name + "/"
                    zipOutputStream.putNextEntry(ZipEntry(entryNamePrefix))
                    zipOutputStream.closeEntry()
                } else {
                    entryNamePrefix = ""
                }

                val paths = sortedMapOf<String, Path>()
                val sourcePath = toPath().toAbsolutePath()
                sourcePath.walk().forEach { path ->
                    val entryName = sourcePath.relativize(path)
                        .let { if ("/" != it.fileSystem.separator) it.toString().replace(it.fileSystem.separator, "/") else it.toString() }
                        .let { if (path.isDirectory) it + "/" else it }
                        .let { entryNamePrefix + it }
                    paths[entryName] = path
                }

                paths.forEach { entryName, path ->
                    zipOutputStream.putNextEntry(ZipEntry(entryName))
                    if (!entryName.endsWith("/")) {
                        path.newInputStream().use { it.copyTo(zipOutputStream) }
                    }
                    zipOutputStream.closeEntry()
                }


            } else {
                zipOutputStream.putNextEntry(ZipEntry(name))
                inputStream().use { it.copyTo(zipOutputStream) }
                zipOutputStream.closeEntry()
            }
        }
    }
}

fun File.zipTo(target: File, overwrite: Boolean = false) = zipToImpl(target, overwrite, false)
fun File.zipContentTo(target: File, overwrite: Boolean = false) = zipToImpl(target, overwrite, true)


fun File.deleteOnExitRecursively() = apply { FilesToDeleteContainer.files.add(this.absoluteFile) }

private object FilesToDeleteContainer {

    val files = concurrentSetOf<File>()

    init {
        Runtime.getRuntime().addShutdownHook(Thread {
            while (files.isNotEmpty()) {
                val iterator = files.iterator()
                while (iterator.hasNext()) {
                    iterator.next().deleteRecursively()
                    iterator.remove()
                }
            }
        })
    }

}
