package name.remal.annotation;

import com.google.common.base.Joiner;
import name.remal.gradle_plugins.api.RelocateClasses;
import name.remal.proxy.CompositeInvocationHandler;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.annotation.Annotation;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Target;
import java.lang.reflect.*;
import java.util.*;
import java.util.Map.Entry;

import static java.lang.String.format;
import static java.lang.reflect.Modifier.isStatic;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static name.remal.ArrayUtils.*;
import static name.remal.PrimitiveTypeUtils.unwrap;
import static name.remal.PrimitiveTypeUtils.wrap;
import static name.remal.SneakyThrow.sneakyThrow;
import static name.remal.UncheckedCast.uncheckedCast;
import static name.remal.reflection.HierarchyUtils.getHierarchy;
import static name.remal.reflection.HierarchyUtils.getPackageHierarchy;

public class AnnotationUtils {

    public static boolean isLangCoreAnnotation(@NotNull Class<? extends Annotation> type) {
        String typeName = type.getName();
        if (typeName.startsWith("java.lang.")) return true;
        if (typeName.startsWith("kotlin.")) return true;
        return false;
    }

    @NotNull
    public static List<@NotNull Method> getAttributeMethods(@NotNull Class<? extends Annotation> type) {
        List<Method> result = new ArrayList<>();
        for (Method method : type.getDeclaredMethods()) {
            if (isStatic(method.getModifiers())) continue;
            if (method.isSynthetic()) continue;
            if (0 != method.getParameterCount()) continue;
            if (Void.TYPE == method.getReturnType()) continue;
            result.add(method);
        }
        return result;
    }

    @NotNull
    @RelocateClasses(Joiner.class)
    public static <T extends Annotation> T newAnnotationInstance(@NotNull Class<T> type, @Nullable Map<String, Object> attributes) {
        List<Method> attributeMethods = getAttributeMethods(type);
        attributeMethods.forEach(method -> method.setAccessible(true));

        Map<String, Object> fullAttributes = new HashMap<>();
        attributeMethods.forEach(attributeMethod -> {
            String attributeName = attributeMethod.getName();
            Object value = null != attributes ? attributes.get(attributeName) : null;
            if (null == value) value = attributeMethod.getDefaultValue();
            if (null == value) throw new IllegalArgumentException("Missing value for " + attributeName);
            if (!wrap(attributeMethod.getReturnType()).isInstance(value)) {
                throw new ErrorCreatingAnnotationInstanceException(type, format(
                    "Attribute %s must be %s, but %s provided",
                    attributeName,
                    unwrap(attributeMethod.getReturnType()),
                    unwrap(value.getClass())
                ));
            }
            fullAttributes.put(attributeName, value);
        });

        if (null != attributes) {
            Set<String> unknownAttrs = new LinkedHashSet<>();
            for (String attr : attributes.keySet()) {
                if (!fullAttributes.containsKey(attr)) unknownAttrs.add(attr);
            }
            if (!unknownAttrs.isEmpty()) {
                if (1 == unknownAttrs.size()) {
                    throw new ErrorCreatingAnnotationInstanceException(type, "Unknown attribute: " + unknownAttrs.iterator().next());
                } else {
                    throw new ErrorCreatingAnnotationInstanceException(type, "Unknown attribute: " + Joiner.on(", ").join(unknownAttrs));
                }
            }
        }

        final int hashCode;
        {
            int value = 0;
            for (Entry<String, Object> entry : fullAttributes.entrySet()) {
                value += 127 * entry.getKey().hashCode() ^ arrayHashCode(entry.getValue());
            }
            hashCode = value;
        }

        final String stringRepresentation;
        {
            StringBuilder sb = new StringBuilder();
            sb.append('@').append(type.getName()).append('(');
            boolean isFirst = true;
            for (Entry<String, Object> entry : fullAttributes.entrySet()) {
                if (isFirst) isFirst = false;
                else sb.append(", ");
                sb.append(entry.getKey()).append('=').append(arrayToString(entry.getValue()));
            }
            sb.append(')');
            stringRepresentation = sb.toString();
        }

        return uncheckedCast(Proxy.newProxyInstance(
            type.getClassLoader(),
            new Class[]{type},
            new CompositeInvocationHandler()
                .appendEqualsHandler((proxy, other) -> {
                    if (!type.isInstance(other)) return false;
                    for (Method attributeMethod : attributeMethods) {
                        if (!arrayEquals(
                            fullAttributes.get(attributeMethod.getName()),
                            attributeMethod.invoke(other)
                        )) {
                            return false;
                        }
                    }
                    return true;
                })
                .appendHashCodeHandler(hashCode)
                .appendToStringHandler(stringRepresentation)
                .appendConstMethodHandler(
                    method -> 0 == method.getParameterCount() && "annotationType".equals(method.getName()),
                    type
                )
                .appendMethodHandler(
                    method -> 0 == method.getParameterCount() && fullAttributes.containsKey(method.getName()),
                    (proxy, method, args) -> fullAttributes.get(method.getName())
                )
        ));
    }

    @NotNull
    public static <T extends Annotation> Map<@NotNull String, @NotNull Object> getAttributes(@NotNull T annotation) {
        Map<String, Object> attrs = new LinkedHashMap<>();
        for (Method method : getAttributeMethods(annotation.annotationType())) {
            method.setAccessible(true);
            try {
                attrs.put(method.getName(), method.invoke(annotation));
            } catch (Throwable throwable) {
                throw sneakyThrow(throwable);
            }
        }
        return attrs;
    }

    @NotNull
    public static <T extends Annotation> T withAttributes(@NotNull T annotation, @Nullable Map<String, Object> attributes) {
        if (null == attributes || attributes.isEmpty()) return annotation;

        Map<String, Object> fullAttributes = new HashMap<>(getAttributes(annotation));
        fullAttributes.putAll(attributes);
        return newAnnotationInstance(uncheckedCast(annotation.annotationType()), fullAttributes);
    }

    public static <T extends Annotation> boolean canAnnotate(@NotNull Class<T> type, @NotNull ElementType elementType) {
        Target target = type.getDeclaredAnnotation(Target.class);
        return null == target || contains(target.value(), elementType);
    }

    public static boolean canAnnotate(@NotNull Class<? extends Annotation> type, @NotNull AnnotatedElement annotatedElement) {
        if (annotatedElement instanceof Annotation) return canAnnotate(type, ElementType.ANNOTATION_TYPE);
        if (annotatedElement instanceof Class) return canAnnotate(type, ElementType.TYPE);
        if (annotatedElement instanceof Field) return canAnnotate(type, ElementType.FIELD);
        if (annotatedElement instanceof Method) return canAnnotate(type, ElementType.METHOD);
        if (annotatedElement instanceof Parameter) return canAnnotate(type, ElementType.PARAMETER);
        if (annotatedElement instanceof Constructor) return canAnnotate(type, ElementType.CONSTRUCTOR);
        if (annotatedElement instanceof Package) return canAnnotate(type, ElementType.PACKAGE);
        if (annotatedElement instanceof AnnotatedTypeVariable) return canAnnotate(type, ElementType.TYPE_PARAMETER) || canAnnotate(type, ElementType.TYPE_USE);
        if (annotatedElement instanceof AnnotatedType) return canAnnotate(type, ElementType.TYPE_USE);
        return false;
    }

    @Nullable
    public static <T extends Annotation> T getMetaAnnotation(@NotNull AnnotatedElement annotatedElement, @NotNull Class<T> type) {
        List<T> annotations = getMetaAnnotationsImpl(annotatedElement, type, true);
        if (!annotations.isEmpty()) return annotations.get(0);
        return null;
    }

    @NotNull
    public static <T extends Annotation> List<@NotNull T> getMetaAnnotations(@NotNull AnnotatedElement annotatedElement, @NotNull Class<T> type) {
        return getMetaAnnotationsImpl(annotatedElement, type, false);
    }

    @NotNull
    private static <T extends Annotation> List<@NotNull T> getMetaAnnotationsImpl(@NotNull AnnotatedElement rootAnnotatedElement, @NotNull Class<T> type, boolean doReturnOnlyFirst) {
        Set<T> result = null;

        boolean canAnnotateAnnotations = canAnnotate(type, ElementType.ANNOTATION_TYPE);

        Set<MetaAnnotationScanContext> processedMetaAnnotationScanContexts = new HashSet<>();
        Queue<MetaAnnotationScanContext> annotatedElementsQueue = new LinkedList<>();
        annotatedElementsQueue.add(new MetaAnnotationScanContext(rootAnnotatedElement, null));
        while (true) {
            MetaAnnotationScanContext metaAnnotationScanContext = annotatedElementsQueue.poll();
            if (null == metaAnnotationScanContext) break;
            if (!processedMetaAnnotationScanContexts.add(metaAnnotationScanContext)) continue;

            AnnotatedElement annotatedElement = metaAnnotationScanContext.getAnnotatedElement();
            List<AnnotationAttribute> attributes = metaAnnotationScanContext.getAttributes();

            {
                // Get directly declared annotation:
                T annotation = annotatedElement.getDeclaredAnnotation(type);
                if (null != annotation) {
                    T resultAnnotation = withAttributes(annotation, attributes);
                    if (doReturnOnlyFirst) return singletonList(resultAnnotation);
                    if (null == result) result = new LinkedHashSet<>();
                    result.add(resultAnnotation);
                }
            }

            List<Annotation> otherDeclaredAnnotations = new ArrayList<>();
            for (Annotation annotation : annotatedElement.getDeclaredAnnotations()) {
                Class<? extends Annotation> annotationType = annotation.annotationType();
                if (type != annotationType && !isLangCoreAnnotation(annotationType)) {
                    otherDeclaredAnnotations.add(annotation);
                }
            }

            // Parse repeatable-container annotations:
            for (Annotation annotation : otherDeclaredAnnotations) {
                List<Method> attributeMethods = getAttributeMethods(annotation.annotationType());
                if (1 != attributeMethods.size()) continue;
                Method valueMethod = attributeMethods.get(0);
                if (!valueMethod.getReturnType().isArray() || type != valueMethod.getReturnType().getComponentType()) continue;
                if (!"value".equals(valueMethod.getName())) continue;
                valueMethod.setAccessible(true);
                final T[] values;
                try {
                    values = uncheckedCast(valueMethod.invoke(annotation));
                } catch (Throwable throwable) {
                    throw sneakyThrow(throwable);
                }
                for (T value : values) {
                    T resultAnnotation = withAttributes(value, attributes);
                    if (doReturnOnlyFirst) return singletonList(resultAnnotation);
                    if (null == result) result = new LinkedHashSet<>();
                    result.add(resultAnnotation);
                }
            }

            // Parse annotations on annotations:
            if (canAnnotateAnnotations) {
                for (Annotation annotation : otherDeclaredAnnotations) {
                    Class<? extends Annotation> annotationType = annotation.annotationType();
                    List<AnnotationAttribute> innerAttributes = null;
                    for (Method attributeMethod : getAttributeMethods(annotationType)) {
                        for (AnnotationAttributeAlias alias : attributeMethod.getDeclaredAnnotationsByType(AnnotationAttributeAlias.class)) {
                            if (null == innerAttributes) {
                                innerAttributes = new ArrayList<>();
                                if (null != attributes) innerAttributes.addAll(attributes);
                            }

                            Object value = null;
                            if (null != attributes) {
                                for (AnnotationAttribute attr : attributes) {
                                    if (annotationType == attr.getAnnotationType() && Objects.equals(attributeMethod.getName(), attr.getName())) {
                                        value = attr.getValue();
                                        break;
                                    }
                                }
                            }
                            if (null == value) {
                                attributeMethod.setAccessible(true);
                                try {
                                    value = attributeMethod.invoke(annotation);
                                } catch (Throwable throwable) {
                                    throw sneakyThrow(throwable);
                                }
                            }

                            innerAttributes.add(new AnnotationAttribute(
                                alias.annotationClass(),
                                alias.attributeName(),
                                value
                            ));
                        }
                    }
                    annotatedElementsQueue.add(new MetaAnnotationScanContext(
                        annotationType,
                        null != innerAttributes ? innerAttributes : attributes
                    ));
                }
            }

            // Scan inheritance hierarchy for root annotated element:
            if (rootAnnotatedElement != annotatedElement) continue;
            if (null == type.getDeclaredAnnotation(Inherited.class)) continue;

            if (annotatedElement instanceof Parameter) {
                List<Parameter> hierarchy = getHierarchy((Parameter) annotatedElement);
                if (2 <= hierarchy.size()) {
                    for (AnnotatedElement curAnnotatedElement : hierarchy.subList(1, hierarchy.size())) {
                        annotatedElementsQueue.add(new MetaAnnotationScanContext(curAnnotatedElement, attributes));
                    }
                }

            } else if (annotatedElement instanceof Method) {
                List<Method> hierarchy = getHierarchy((Method) annotatedElement);
                if (2 <= hierarchy.size()) {
                    for (AnnotatedElement curAnnotatedElement : hierarchy.subList(1, hierarchy.size())) {
                        annotatedElementsQueue.add(new MetaAnnotationScanContext(curAnnotatedElement, attributes));
                    }
                }

            } else if (annotatedElement instanceof Class) {
                List<Class<?>> hierarchy = uncheckedCast(getHierarchy((Class<?>) annotatedElement));
                if (2 <= hierarchy.size()) {
                    for (AnnotatedElement curAnnotatedElement : hierarchy.subList(1, hierarchy.size())) {
                        annotatedElementsQueue.add(new MetaAnnotationScanContext(curAnnotatedElement, attributes));
                    }
                }
                if (canAnnotate(type, ElementType.PACKAGE)) {
                    for (Class<?> curClass : hierarchy) {
                        for (Package curPackage : getPackageHierarchy(curClass)) {
                            annotatedElementsQueue.add(new MetaAnnotationScanContext(curPackage, attributes));
                        }
                    }
                }
            }
        }

        return null == result ? emptyList() : new ArrayList<>(result);
    }

    @NotNull
    private static <T extends Annotation> T withAttributes(@NotNull T annotation, @Nullable Collection<AnnotationAttribute> attributes) {
        if (null == attributes || attributes.isEmpty()) return annotation;

        Map<String, Object> attrsMap = new HashMap<>();
        Class<T> annotationType = uncheckedCast(annotation.annotationType());
        for (AnnotationAttribute attr : attributes) {
            if (annotationType == attr.getAnnotationType()) {
                attrsMap.putIfAbsent(attr.getName(), attr.getValue());
            }
        }
        getAttributes(annotation).forEach(attrsMap::putIfAbsent);
        return newAnnotationInstance(annotationType, attrsMap);
    }

}
