package me.ramendev.expokert;

import java.util.Set;

/**
 * The wildcard, which is a card that can be considered as another "real" {@link Card}.
 * There are variations of wildcards, such as the {@link Joker}, the "universal" wildcard.
 */
public abstract class Wildcard extends Card {
	/**
	 * Constructor for a wildcard. A wildcard will always use {@link Pip#WILD} and
	 * {@link Suit#WILD}.
	 */
	public Wildcard() {
		super(Pip.WILD, Suit.WILD);
	}
	
	/**
	 * Checks whether this wildcard "can be" another card, in this case, whether
	 * it can be converted to the given {@link Card}.
	 * @param  card The card to check against.
	 * @return 	Whether this wildcard "can be" another card.
	 */
	public final boolean canBe(final Card card) {
		return getConversions().contains(card);
	}
	
	/**
	 * Gets all the {@link Card} that this wildcard can convert to. This is implementation-
	 * specific, to different kinds of wildcards in different games.
	 *
	 * @return   The set of all cards that this wildcard can convert to.
	 * @implNote Rationale for this class and this method being abstract: this
	 *           wildcard should implement the logic to get the conversions by storing
	 *           additional game state (done by the subclasses), and THEN return
	 *           the cards it can convert to in a dynamic manner.
	 */
	public abstract Set<Card> getConversions();
}
