package me.ramendev.expokert;

import java.util.EnumSet;

import lombok.Getter;
import me.ramendev.expokert.exception.IllegalCardException;

/**
 * The pip of a {@link Card}, or a card's value. It is notable that the term "value"
 * might not be reflected throughout this enum's usages, as different games
 * might consider the pip to be comparable, and implement an ordering of pips,
 * or not at all. By default, this enum is ordered as {@link #TWO} to {@link #ACE},
 * in ascending order.
 *
 * @implNote Some methods in this enum that concern the order of pips use {@link #ordinal()}
 *           in their implementation. The implementation of the enum here declares
 *           {@link #TWO} first and {@link #ACE} last, as it is regarded as the
 *           <strong>most</strong> common ordering of pips, though that is not
 *           to say that <em>some</em> variants use alternate or no orderings at all.
 * @apiNote  To get the so-called "value" of a card, use {@link #getValue()},
 *           as in several games, the value is often more meaningful than
 *           the {@link #ordinal()}, although {@link #getValue()} is equivalent
 *           to <code>ordinal() + 2</code>.
 * @see      Suit
 * @see      Card
 */
@Getter
public enum Pip {
	/**
	 * Two pips by default, but in some variants, two is the largest pip.
	 */
	TWO('2'),
	
	/**
	 * Three pips.
	 */
	THREE('3'),
	
	/**
	 * Four pips.
	 */
	FOUR('4'),
	
	/**
	 * Five pips.
	 */
	FIVE('5'),
	
	/**
	 * Six pips.
	 */
	SIX('6'),
	
	/**
	 * Seven pips.
	 */
	SEVEN('7'),
	
	/**
	 * Eight pips.
	 */
	EIGHT('8'),
	
	/**
	 * Nine pips.
	 */
	NINE('9'),
	
	/**
	 * Ten pips.
	 */
	TEN('T'),
	
	/**
	 * The Jack pip, or eleven pips.
	 */
	JACK('J'),
	
	/**
	 * The Queen pip, or twelve pips.
	 */
	QUEEN('Q'),
	
	/**
	 * The King pip, or thirteen pips.
	 */
	KING('K'),
	
	/**
	 * The Ace pip, or fourteen pips by default, and in some scenarios, only
	 * worth one pip and is the smallest of the bunch.
	 */
	ACE('A'),
	
	/**
	 * The wildcard pip. Its value always depends on the circumstances of a
	 * particular card game.
	 */
	WILD('!');
	
	/**
	 * The character that represents the pip in text.
	 */
	private final char character;
	
	/**
	 * Trivial all-arguments constructor. Creates a new pip with the given character.
	 *
	 * @param character The character of the pip.
	 */
	Pip(final char character) {
		this.character = character;
	}
	
	/**
	 * Returns the pip that is notated with the provided character.
	 *
	 * @param  character The character of the pip to be returned.
	 * @return The pip with the provided character.
	 */
	public static Pip from(char character) {
		character = Character.toUpperCase(character);
		for (final Pip pip : values()) {
			if (pip.getCharacter() == character) {
				return pip;
			}
		}
		throw new IllegalCardException("Unexpected pip character: " + character);
	}
	
	/**
	 * Trivial getter of the {@link #character} field.
	 *
	 * @return The character of the pip.
	 */
	public final char getCharacter() {
		return character;
	}
	
	/**
	 * Gets the numerical value of the pip. "Value" in this case often means a numerical
	 * value, used in some games, and not necessarily the pip's value itself. By default,
	 * this method returns <code>ordinal() + 2</code>, which assumes that {@link #TWO} is
	 * the smallest and {@link #ACE} is the largest.
	 *
	 * @return   The "value" of the pip.
	 * @implNote It is recommended that you never use this method with the {@link #WILD}
	 *           pip, as per the enum's documentation.
	 */
	public final int getValue() {
		return ordinal() + 2;
	}
	
	/**
	 * Checks if the given pip is directly next to the given pip in the default
	 * pip order. It is universal that {@link #TWO} and {@link #ACE} is "next to
	 * each other", and the {@link #WILD} is next to nothing.
	 * <p>
	 * Formally, this method checks if:<br>
	 * <code>Math.abs(this.ordinal() - pip.ordinal()) == 1</code>.
	 *
	 * @param  pip The other pip to check against.
	 * @return Whether the two pips are next to each other.
	 */
	public final boolean isNextTo(final Pip pip) {
		return (!EnumSet.of(this, pip).contains(WILD)
			&& (Math.abs(ordinal() - pip.ordinal()) == 1
			|| EnumSet.of(this, pip).containsAll(EnumSet.of(TWO, ACE))));
	}
}
