package me.ramendev.expokert;

import java.util.ArrayList;
import java.util.List;

import lombok.Value;
import lombok.experimental.NonFinal;
import me.ramendev.expokert.exception.IllegalCardException;

/**
 * A playing card. <a href="https://en.wikipedia.org/wiki/Playing_card">Playing card Wikipedia article</a>.
 * This class only contains the basic information about a playing card, which is
 * its {@link Pip} (its value, synonymously) and its {@link Suit}.
 *
 * @apiNote  Technically, one could create a {@link Card} with the {@link Pip#WILD}
 *           pip and the {@link Suit#WILD} suit, thus not implementing the wildcard
 *           {@link Wildcard#getConversions()} functionality one would do by instantiating,
 *           a {@link Wildcard} or its subclass instance. This, obviously, is not
 *           recommended.
 * @implNote If you want to create a wildcard-kind of card, you should always use
 *           the abstract class {@link Wildcard}, or one of the wildcard implementations
 *           such as {@link Joker}.
 */
@Value
@NonFinal
public class Card {
	/**
	 * The pip of a card, or a card's value.
	 */
	Pip pip;
	
	/**
	 * The suit of a card.
	 */
	Suit suit;
	
	/**
	 * Creates a card from the given pip and suit.
	 *
	 * @param  pip  The pip of the card.
	 * @param  suit The suit of the card.
	 * @throws IllegalCardException When any of the pip or the suit of the card is
	 *                              WILD. It is recommended that you implement a
	 *                              {@link Wildcard}, or use one of its subclasses
	 *                              instead, such as {@link Joker}.
	 */
	public Card(final Pip pip, final Suit suit) {
		if ((pip == Pip.WILD || suit == Suit.WILD) && !(this instanceof Wildcard)) {
			throw new IllegalCardException(
				"Card instance cannot have any properties as WILD, use Wildcard implementations instead"
			);
		}
		this.pip = pip;
		this.suit = suit;
	}
	
	/**
	 * Creates a card from the given string, which should notate the card's properties.
	 *
	 * @param  string The string of this card.
	 * @throws IllegalCardException When the string's notation cannot be parsed to a card.
	 */
	public Card(final String string) {
		this(Pip.from(string.charAt(0)), Suit.from(Character.toLowerCase(string.charAt(1))));
		if (string.length() != 2) {
			throw new IllegalCardException("Invalid string for card: " + string);
		}
	}
	
	/**
	 * Gets all the cards that can be created from all of the {@link Pip}s (except
	 * for {@link Pip#WILD}), and all of the {@link Suit}s (except for
	 * {@link Suit#WILD})
	 *
	 * @return The list of all cards.
	 */
	public static List<Card> getAllCards() {
		final List<Card> cards = new ArrayList<>(Deck.DEFAULT_DECK_SIZE - 1);
		for (final Pip pip : Pip.values()) {
			if (pip != Pip.WILD) {
				for (final Suit suit : Suit.values()) {
					if (suit != Suit.WILD) {
						cards.add(new Card(pip, suit));
					}
				}
			}
		}
		return cards;
	}
	
	/**
	 * Gets the notation form of this card.
	 *
	 * @return The notation form of the card.
	 */
	@Override
	public String toString() {
		return getPip().getCharacter() + "" + getSuit().getCharacter();
	}
}
