package me.ramendev.expokert;

import java.util.Arrays;
import java.util.Deque;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.Stack;
import java.util.stream.Collectors;

import lombok.Value;
import me.ramendev.expokert.exception.DeckSizeException;
import me.ramendev.expokert.exception.DuplicateCardException;

/**
 * The standard 52-card deck, with an option to include a {@link Wildcard}.
 */
@Value
public class Deck {
	/**
	 * The default deck size, which is 52, or the number of possible permutations
	 * of {@link Pip}s (excluding the {@link Pip#WILD}) and {@link Suit}s
	 * (excluding the {@link Suit#WILD}).
	 */
	public static final int DEFAULT_DECK_SIZE
		= (Pip.values().length - 1) * (Suit.values().length - 1);
	
	/**
	 * The set of cards that this deck holds. Using a set ensures that no duplicates
	 * of cards can be present, and using the {@link LinkedHashSet} implementation
	 * maintains insertion order.
	 */
	LinkedHashSet<Card> cards;
	
	/**
	 * Constructor for a list of {@link Card}s from the provided {@link List}.
	 *
	 * @param cards The list of cards for this deck to hold.
	 */
	public Deck(final List<Card> cards) {
		this.cards = new LinkedHashSet<>(cards);
	}
	
	/**
	 * Constructor for an empty deck of {@link Card}s.
	 */
	public Deck() {
		this.cards = new LinkedHashSet<>(DEFAULT_DECK_SIZE);
	}
	
	/**
	 * Constructor for a standard 52-{@link Card} deck, with the option to add
	 * a {@link Wildcard}. Note that in most decks, there will only be one
	 * wildcard present.
	 *
	 * @param wildcard The ({@link Optional}) wildcard to add to the deck.
	 */
	public Deck(final Optional<Wildcard> wildcard) {
		this(Card.getAllCards());
		wildcard.ifPresent(this::addCard);
	}
	
	/**
	 * Constructor for a deck generated from a list of whitespace-separated cards.
	 *
	 * @param cards The string of cards to generate the deck from.
	 */
	public Deck(final String cards) {
		this(Arrays
			.stream(cards.split("[\s,]+"))
			.map(Card::new)
			.toList()
		);
	}
	
	/**
	 * Returns the string form of this deck. More formally, returns a string that
	 * is a result of {@link String#join(CharSequence, CharSequence...)}ing the
	 * cards' string forms, separated by spaces.
	 *
	 * @return The string form of this deck.
	 */
	@Override
	public String toString() {
		return cards.stream().map(Card::toString).collect(Collectors.joining(" "));
	}
	
	/**
	 * Returns how many cards are in this deck.
	 *
	 * @return The size of the set containing cards.
	 * @see Set#size()
	 */
	public int getSize() {
		return getCards().size();
	}
	
	/**
	 * Adds the given {@link Card} to the deck.
	 *
	 * @param card The card to add to the deck.
	 * @throws DeckSizeException When the deck is already full.
	 * @throws DuplicateCardException When the card being added already has a
	 * 	                              duplicate in the list.
	 */
	public void addCard(final Card card) {
		if (getSize() >= DEFAULT_DECK_SIZE + 1) {
			throw new DeckSizeException("Deck size exceeded " + DEFAULT_DECK_SIZE);
		} else if (!getCards().add(card)) {
			throw new DuplicateCardException("Card " + card + "already exists in the deck");
		}
	}
	
	/**
	 * "Pops" the card (similar to the pop operation in the {@link Stack} or
	 * {@link Deque} data structure). Formally, removes the top card in the
	 * deck (or "last" when referring to {@link LinkedHashSet}s) and returns it.
	 *
	 * @return The card at the top of the deck.
	 * @throws DeckSizeException When the deck is empty, meaning there is no cards.
	 * 	                         to pop from it.
	 * @see Stack#pop()
	 */
	public Card popCard() {
		if (getSize() == 0) {
			throw new DeckSizeException("Nothing to pop from deck with size 0");
		}
		final Card card = getCards().toArray(Card[]::new)[getSize() - 1];
		getCards().remove(card);
		return card;
	}
}
