/*
 * Decompiled with CFR 0.152.
 */
package me.desair.tus.server;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Set;
import me.desair.tus.server.HttpMethod;
import me.desair.tus.server.TusExtension;
import me.desair.tus.server.checksum.ChecksumExtension;
import me.desair.tus.server.concatenation.ConcatenationExtension;
import me.desair.tus.server.core.CoreProtocol;
import me.desair.tus.server.creation.CreationExtension;
import me.desair.tus.server.download.DownloadExtension;
import me.desair.tus.server.exception.TusException;
import me.desair.tus.server.expiration.ExpirationExtension;
import me.desair.tus.server.termination.TerminationExtension;
import me.desair.tus.server.upload.UploadIdFactory;
import me.desair.tus.server.upload.UploadInfo;
import me.desair.tus.server.upload.UploadLock;
import me.desair.tus.server.upload.UploadLockingService;
import me.desair.tus.server.upload.UploadStorageService;
import me.desair.tus.server.upload.UuidUploadIdFactory;
import me.desair.tus.server.upload.cache.ThreadLocalCachedStorageAndLockingService;
import me.desair.tus.server.upload.disk.DiskLockingService;
import me.desair.tus.server.upload.disk.DiskStorageService;
import me.desair.tus.server.util.TusServletRequest;
import me.desair.tus.server.util.TusServletResponse;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TusFileUploadService {
    public static final String TUS_API_VERSION = "1.0.0";
    private static final Logger log = LoggerFactory.getLogger(TusFileUploadService.class);
    private UploadStorageService uploadStorageService;
    private UploadLockingService uploadLockingService;
    private UploadIdFactory idFactory = new UuidUploadIdFactory();
    private final LinkedHashMap<String, TusExtension> enabledFeatures = new LinkedHashMap();
    private final Set<HttpMethod> supportedHttpMethods = EnumSet.noneOf(HttpMethod.class);
    private boolean isThreadLocalCacheEnabled = false;
    private boolean isChunkedTransferDecodingEnabled = false;

    public TusFileUploadService() {
        String storagePath = FileUtils.getTempDirectoryPath() + File.separator + "tus";
        this.uploadStorageService = new DiskStorageService(this.idFactory, storagePath);
        this.uploadLockingService = new DiskLockingService(this.idFactory, storagePath);
        this.initFeatures();
    }

    protected void initFeatures() {
        this.addTusExtension(new CoreProtocol());
        this.addTusExtension(new CreationExtension());
        this.addTusExtension(new ChecksumExtension());
        this.addTusExtension(new TerminationExtension());
        this.addTusExtension(new ExpirationExtension());
        this.addTusExtension(new ConcatenationExtension());
    }

    public TusFileUploadService withUploadUri(String uploadUri) {
        this.idFactory.setUploadUri(uploadUri);
        return this;
    }

    public TusFileUploadService withMaxUploadSize(Long maxUploadSize) {
        Validate.exclusiveBetween((long)0L, (long)Long.MAX_VALUE, (long)maxUploadSize, (String)"The max upload size must be bigger than 0");
        this.uploadStorageService.setMaxUploadSize(maxUploadSize);
        return this;
    }

    public TusFileUploadService withUploadIdFactory(UploadIdFactory uploadIdFactory) {
        Validate.notNull((Object)uploadIdFactory, (String)"The UploadIdFactory cannot be null", (Object[])new Object[0]);
        String previousUploadURI = this.idFactory.getUploadUri();
        this.idFactory = uploadIdFactory;
        this.idFactory.setUploadUri(previousUploadURI);
        this.uploadStorageService.setIdFactory(this.idFactory);
        this.uploadLockingService.setIdFactory(this.idFactory);
        return this;
    }

    public TusFileUploadService withUploadStorageService(UploadStorageService uploadStorageService) {
        Validate.notNull((Object)uploadStorageService, (String)"The UploadStorageService cannot be null", (Object[])new Object[0]);
        uploadStorageService.setMaxUploadSize(this.uploadStorageService.getMaxUploadSize());
        uploadStorageService.setUploadExpirationPeriod(this.uploadStorageService.getUploadExpirationPeriod());
        uploadStorageService.setIdFactory(this.idFactory);
        this.uploadStorageService = uploadStorageService;
        this.prepareCacheIfEnabled();
        return this;
    }

    public TusFileUploadService withUploadLockingService(UploadLockingService uploadLockingService) {
        Validate.notNull((Object)uploadLockingService, (String)"The UploadStorageService cannot be null", (Object[])new Object[0]);
        uploadLockingService.setIdFactory(this.idFactory);
        this.uploadLockingService = uploadLockingService;
        this.prepareCacheIfEnabled();
        return this;
    }

    public TusFileUploadService withStoragePath(String storagePath) {
        Validate.notBlank((CharSequence)storagePath, (String)"The storage path cannot be blank", (Object[])new Object[0]);
        this.withUploadStorageService(new DiskStorageService(storagePath));
        this.withUploadLockingService(new DiskLockingService(storagePath));
        this.prepareCacheIfEnabled();
        return this;
    }

    public TusFileUploadService withThreadLocalCache(boolean isEnabled) {
        this.isThreadLocalCacheEnabled = isEnabled;
        this.prepareCacheIfEnabled();
        return this;
    }

    public TusFileUploadService withChunkedTransferDecoding(boolean isEnabled) {
        this.isChunkedTransferDecodingEnabled = isEnabled;
        return this;
    }

    public TusFileUploadService withUploadExpirationPeriod(Long expirationPeriod) {
        this.uploadStorageService.setUploadExpirationPeriod(expirationPeriod);
        return this;
    }

    public TusFileUploadService withDownloadFeature() {
        this.addTusExtension(new DownloadExtension());
        return this;
    }

    public TusFileUploadService addTusExtension(TusExtension feature) {
        Validate.notNull((Object)feature, (String)"A custom feature cannot be null", (Object[])new Object[0]);
        this.enabledFeatures.put(feature.getName(), feature);
        this.updateSupportedHttpMethods();
        return this;
    }

    public TusFileUploadService disableTusExtension(String extensionName) {
        Validate.notNull((Object)extensionName, (String)"The extension name cannot be null", (Object[])new Object[0]);
        Validate.isTrue((!StringUtils.equals((CharSequence)"core", (CharSequence)extensionName) ? 1 : 0) != 0, (String)"The core protocol cannot be disabled", (Object[])new Object[0]);
        this.enabledFeatures.remove(extensionName);
        this.updateSupportedHttpMethods();
        return this;
    }

    public Set<HttpMethod> getSupportedHttpMethods() {
        return EnumSet.copyOf(this.supportedHttpMethods);
    }

    public Set<String> getEnabledFeatures() {
        return new LinkedHashSet<String>(this.enabledFeatures.keySet());
    }

    public void process(HttpServletRequest servletRequest, HttpServletResponse servletResponse) throws IOException {
        this.process(servletRequest, servletResponse, null);
    }

    public void process(HttpServletRequest servletRequest, HttpServletResponse servletResponse, String ownerKey) throws IOException {
        Validate.notNull((Object)servletRequest, (String)"The HTTP Servlet request cannot be null", (Object[])new Object[0]);
        Validate.notNull((Object)servletResponse, (String)"The HTTP Servlet response cannot be null", (Object[])new Object[0]);
        HttpMethod method = HttpMethod.getMethodIfSupported(servletRequest, this.supportedHttpMethods);
        log.debug("Processing request with method {} and URL {}", (Object)method, (Object)servletRequest.getRequestURL());
        TusServletRequest request = new TusServletRequest(servletRequest, this.isChunkedTransferDecodingEnabled);
        TusServletResponse response = new TusServletResponse(servletResponse);
        try (UploadLock lock = this.uploadLockingService.lockUploadByUri(request.getRequestURI());){
            this.processLockedRequest(method, request, response, ownerKey);
        }
        catch (TusException e) {
            log.error("Unable to lock upload for request URI " + request.getRequestURI(), (Throwable)e);
        }
    }

    public InputStream getUploadedBytes(String uploadUri) throws IOException, TusException {
        return this.getUploadedBytes(uploadUri, null);
    }

    public InputStream getUploadedBytes(String uploadUri, String ownerKey) throws IOException, TusException {
        try (UploadLock lock = this.uploadLockingService.lockUploadByUri(uploadUri);){
            InputStream inputStream = this.uploadStorageService.getUploadedBytes(uploadUri, ownerKey);
            return inputStream;
        }
    }

    public UploadInfo getUploadInfo(String uploadUri) throws IOException, TusException {
        return this.getUploadInfo(uploadUri, null);
    }

    public UploadInfo getUploadInfo(String uploadUri, String ownerKey) throws IOException, TusException {
        try (UploadLock lock = this.uploadLockingService.lockUploadByUri(uploadUri);){
            UploadInfo uploadInfo = this.uploadStorageService.getUploadInfo(uploadUri, ownerKey);
            return uploadInfo;
        }
    }

    public void deleteUpload(String uploadUri) throws IOException, TusException {
        this.deleteUpload(uploadUri, null);
    }

    public void deleteUpload(String uploadUri, String ownerKey) throws IOException, TusException {
        try (UploadLock lock = this.uploadLockingService.lockUploadByUri(uploadUri);){
            UploadInfo uploadInfo = this.uploadStorageService.getUploadInfo(uploadUri, ownerKey);
            if (uploadInfo != null) {
                this.uploadStorageService.terminateUpload(uploadInfo);
            }
        }
    }

    public void cleanup() throws IOException {
        this.uploadLockingService.cleanupStaleLocks();
        this.uploadStorageService.cleanupExpiredUploads(this.uploadLockingService);
    }

    protected void processLockedRequest(HttpMethod method, TusServletRequest request, TusServletResponse response, String ownerKey) throws IOException {
        try {
            this.validateRequest(method, (HttpServletRequest)request, ownerKey);
            this.executeProcessingByFeatures(method, request, response, ownerKey);
        }
        catch (TusException e) {
            this.processTusException(method, request, response, ownerKey, e);
        }
    }

    protected void executeProcessingByFeatures(HttpMethod method, TusServletRequest servletRequest, TusServletResponse servletResponse, String ownerKey) throws IOException, TusException {
        for (TusExtension feature : this.enabledFeatures.values()) {
            if (servletRequest.isProcessedBy(feature)) continue;
            servletRequest.addProcessor(feature);
            feature.process(method, servletRequest, servletResponse, this.uploadStorageService, ownerKey);
        }
    }

    protected void validateRequest(HttpMethod method, HttpServletRequest servletRequest, String ownerKey) throws TusException, IOException {
        for (TusExtension feature : this.enabledFeatures.values()) {
            feature.validate(method, servletRequest, this.uploadStorageService, ownerKey);
        }
    }

    protected void processTusException(HttpMethod method, TusServletRequest request, TusServletResponse response, String ownerKey, TusException exception) throws IOException {
        int status = exception.getStatus();
        String message = exception.getMessage();
        log.warn("Unable to process request {} {}. Sent response status {} with message \"{}\"", new Object[]{method, request.getRequestURL(), status, message});
        try {
            for (TusExtension feature : this.enabledFeatures.values()) {
                if (request.isProcessedBy(feature)) continue;
                request.addProcessor(feature);
                feature.handleError(method, request, response, this.uploadStorageService, ownerKey);
            }
            UploadInfo uploadInfo = this.uploadStorageService.getUploadInfo(request.getRequestURI(), ownerKey);
            this.uploadStorageService.removeLastNumberOfBytes(uploadInfo, request.getBytesRead());
        }
        catch (TusException ex) {
            log.warn("An exception occurred while handling another exception", (Throwable)ex);
        }
        response.sendError(status, message);
    }

    private void updateSupportedHttpMethods() {
        this.supportedHttpMethods.clear();
        for (TusExtension tusFeature : this.enabledFeatures.values()) {
            this.supportedHttpMethods.addAll(tusFeature.getMinimalSupportedHttpMethods());
        }
    }

    private void prepareCacheIfEnabled() {
        if (this.isThreadLocalCacheEnabled && this.uploadStorageService != null && this.uploadLockingService != null) {
            ThreadLocalCachedStorageAndLockingService service = new ThreadLocalCachedStorageAndLockingService(this.uploadStorageService, this.uploadLockingService);
            service.setIdFactory(this.idFactory);
            this.uploadStorageService = service;
            this.uploadLockingService = service;
        }
    }
}

