package li.vin.my.apikit.network

/**
 * 10/31/18.
 */

interface Poolable {

    val poolId: String
    var isCanceled: Boolean

    fun execute()
    fun completed(poolable: Poolable)

}

object ServicePool {

    private val servicePool = mutableMapOf<String, List<Poolable>>()
    private val requestOrder = mutableListOf<String>()
    private val runningServices = mutableListOf<Poolable>()

    /**
     * Add services to the pool.
     * These services will be executed at some point in the future based on when a similar service was added to the pool.
     */
    fun enter(poolable: Poolable) {
        val similarServices = servicePool[poolable.poolId]?.toMutableList()
        if (similarServices != null) {
            similarServices.add(poolable)
            servicePool[poolable.poolId] = similarServices
        } else {
            if (!requestOrder.contains(poolable.poolId)) {
                requestOrder.add(poolable.poolId)
            }
            servicePool[poolable.poolId] = listOf(poolable)
        }
        startNextService()
    }

    /**
     * Takes the completed service out of the pool and notify its result to all of the similar services in the pool.
     */
    fun leave(poolable: Poolable) {
        val similarServices = servicePool[poolable.poolId]?.toMutableList() ?: return

        servicePool.remove(poolable.poolId)
        requestOrder.remove(poolable.poolId)

        for (similarService in similarServices) {
            if (similarService != poolable) {
                similarService.completed(poolable)
            }
        }
        similarServices.remove(poolable)
        startNextService()
    }

    /**
     * Starts next service from the pool.
     */
    @Suppress("SENSELESS_COMPARISON")
    private fun startNextService() {
        refreshPool()

        if (requestOrder.isEmpty()) {
            return
        }

        val nextPoolId = requestOrder.first()
        val services = servicePool[nextPoolId] ?: return

        val service = services.first()

        if (service != null) {
            runningServices.add(service)
            service.execute()
        } else {
            servicePool.remove(nextPoolId)
        }
        requestOrder.remove(nextPoolId)
    }

    /**
     * Removes the canceled services
     */
    private fun refreshPool() {
        if (requestOrder.isEmpty()) {
            return
        }

        val nextPoolId = requestOrder.first()
        var services = servicePool[nextPoolId] ?: return

        services = services.filter { !it.isCanceled }
        servicePool[nextPoolId] = services

        if (services.isEmpty()) {
            requestOrder.remove(nextPoolId)
            servicePool.remove(nextPoolId)
            refreshPool()
        }
    }
}
