package li.vin.my.apikit.network

import com.google.gson.Gson
import io.sentry.Sentry
import li.vin.my.apikit.ApiKit
import li.vin.my.apikit.AuthSettings
import li.vin.my.apikit.errors.*
import okhttp3.Cache
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import retrofit2.Call
import retrofit2.Callback
import retrofit2.Response
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import retrofit2.converter.scalars.ScalarsConverterFactory
import java.security.KeyStore
import java.util.*
import java.util.concurrent.TimeUnit
import javax.net.ssl.SSLContext
import javax.net.ssl.TrustManager
import javax.net.ssl.TrustManagerFactory
import javax.net.ssl.X509TrustManager

//import retrofit2.converter.scalars.ScalarsConverterFactory

/**
 * 10/31/18.
 */

abstract class BaseService<Pojo> : Queueable, Poolable {

    var call: Call<Pojo>? = null
    private var dedupe = false
    var id = ""
    protected abstract fun createCall()

    var responseHandler: BaseResponseHandler? = null

    private var serviceQueue: ServiceQueue? = null
    private var queuePosition: Int = 0
    private var result: Pojo? = null
    private var throwable: Throwable? = null
    override var isCanceled: Boolean = false
    var successful: Boolean = false

    fun requestData(responseHandler: ResponseHandler<Pojo>) {
        this.dedupe = false
        makeCall(responseHandler)
    }

    fun requestNoDedupeData(id: String, responseHandler: ResponseHandler<Pojo>) {
        this.dedupe = true
        this.id = id
        makeCall(responseHandler)
    }

    private fun makeCall(responseHandler: ResponseHandler<Pojo>) {
        this.responseHandler = responseHandler
        createCall()
        if (!isCanceled) {
            ServicePool.enter(this)
        }
    }

    fun getDeviceId(): String {
        return AuthSettings.instance.getDeviceId()
    }

    fun getDriverId(): String {
        return AuthSettings.instance.getDriverId()
    }

    fun getVehicleId(): String {
        return AuthSettings.instance.getVehicleId()
    }

    fun getCookies(): String {
        return AuthSettings.instance.getSessionId()
    }

    fun getAuthToken(): String {
        return AuthSettings.instance.getAuthToken()
    }

    private fun startService() {
        call?.clone()?.enqueue(object : Callback<Pojo> {
            override fun onResponse(call: Call<Pojo>, response: Response<Pojo>) {
                if (response.isSuccessful) {
                    successful = true
                    result = response.body()
                } else {
                    if (response.code() == 409) {
                        successful = false
                        throwable = ConflictException()
                    } else {
                        if (response.body() == null || (response.body() is List<*> && (response.body() as List<*>).size == 0)) {
                            successful = false
                            try {
                                throwable = DataNotFoundException(getErrorMessage(response))
                            } catch (e: Exception) {
                                throwable = DataNotFoundException("")
                            }
                        } else if (response.code() != 200) {
                            throwable = GenericErrorException(getErrorMessage(response))
                            successful = false
                        }
                    }
                }
                ServicePool.leave(this@BaseService)
                endService()
            }

            override fun onFailure(call: Call<Pojo>, t: Throwable) {
                Sentry.capture("API Failure: " + t.message)
                successful = false
                throwable = Exception(t)
                ServicePool.leave(this@BaseService)
                endService()
            }
        })
    }

    @Suppress("UNCHECKED_CAST")
    fun endService() {
        if (serviceQueue != null) {
            if (successful) {
                serviceQueue?.serviceCompleted(result as Any, queuePosition)
            } else {
                throwable?.let {
                    serviceQueue?.serviceError(it)
                }
            }
            return
        }
        if (successful) {
            (responseHandler as ResponseHandler<Pojo>).onSuccess(result)
        } else {
            throwable?.let {
                (responseHandler as ResponseHandler<Pojo>).onError(it)
            }
        }
    }

    fun getErrorMessage(response: Response<Pojo>): String {
        var errorMessage = ""
        if (response.errorBody() != null) {
            val gson = Gson()
            val genericError = gson.fromJson(response.errorBody()?.string(), GenericError::class.java)
            errorMessage = genericError.message
        }
        return if (errorMessage.isEmpty()) {
            GenericError.getErrorMessage(response.code())
        } else {
            errorMessage
        }
    }

    fun getErrorMessageFor400(response: Response<Pojo>): String {
        var errorMessage = ""
        if (response.errorBody() != null) {
            val gson = Gson()
            val genericError = gson.fromJson(response.errorBody()?.string(), ErrorMessage::class.java)
            errorMessage = genericError.message
        }
        return errorMessage
    }

    fun <T> createService(serviceClass: Class<T>, baseUrl: String): T {
        return Retrofit.Builder()
            .baseUrl(baseUrl)
            .addConverterFactory(ScalarsConverterFactory.create())
            .addConverterFactory(GsonConverterFactory.create(GsonConverter.gsonConverter))
            .client(ApiKit.okHttpClient)
            .build()
            .create(serviceClass)
    }

    fun <T> createService(serviceClass: Class<T>, baseUrl: String, authToken: String): T {
        val okHttpClientBuilder = OkHttpClient.Builder()

        if (AuthSettings.instance.getAuthToken() != "Bearer") {
            okHttpClientBuilder.addInterceptor(Interceptor { chain ->
                val original = chain.request()
                val requestBuilder = original.newBuilder()
                    .header("Authorization", authToken)
                    .method(original.method, original.body)
                val request = requestBuilder.build()
                chain.proceed(request)
            })
        }
        val trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm())
        trustManagerFactory.init(null as KeyStore?)
        val trustManagers = trustManagerFactory.trustManagers
        check(!(trustManagers.size != 1 || trustManagers[0] !is X509TrustManager)) {
            ("Unexpected default trust managers:" + Arrays.toString(trustManagers))
        }
        val trustManager = trustManagers[0] as X509TrustManager
        val sslContext = SSLContext.getInstance("TLS")
        sslContext.init(null, arrayOf<TrustManager>(trustManager), null)
        val cacheSize: Long = 10 * 1024 * 1024 // 10 MB
        val cacheDir = Cache(ApiKit.instance.cacheDir, cacheSize)
        okHttpClientBuilder.cache(cacheDir)
        val logging = HttpLoggingInterceptor()
        logging.level = if (ApiKit.debugLogging) HttpLoggingInterceptor.Level.BODY else HttpLoggingInterceptor.Level.BASIC
        okHttpClientBuilder.addInterceptor(logging)
        okHttpClientBuilder.addInterceptor(ReceivedCookiesInterceptor())
        okHttpClientBuilder.sslSocketFactory(sslContext.socketFactory, trustManager)
        okHttpClientBuilder.connectTimeout(45, TimeUnit.SECONDS)
        okHttpClientBuilder.readTimeout(45, TimeUnit.SECONDS)
        return Retrofit.Builder()
            .baseUrl(baseUrl)
            .addConverterFactory(GsonConverterFactory.create(GsonConverter.gsonConverter))
            .client(okHttpClientBuilder.build())
            .build()
            .create(serviceClass)

    }

    override fun startServiceInQueue(serviceQueue: ServiceQueue, queuePosition: Int) {
        this.serviceQueue = serviceQueue
        this.queuePosition = queuePosition
        createCall()
        if (!isCanceled) {
            ServicePool.enter(this)
        }
    }

    override fun setBaseResponseHandler(responseHandler: BaseResponseHandler) {
        this.responseHandler = responseHandler
    }

    override fun execute() {
        startService()
    }

    @Suppress("UNCHECKED_CAST")
    override fun completed(poolable: Poolable) {
        if (poolable is BaseService<*>) {
            val duplicate = poolable.result as Pojo
            duplicate?.let {
                successful = true
                result = duplicate
                endService()
                return
            }
            poolable.throwable.let {
                successful = false
                throwable = it
                endService()
            }
        }
    }

    override val poolId: String
        get() {
            return if (dedupe) call?.request()?.url.toString() + id else call?.request()?.url.toString()
        }
}

fun <Object> BaseService<Object>.setCanceled(canceled: Boolean) {
    isCanceled = canceled
}