package li.vin.my.apikit

import android.app.Application
import android.content.Context
import io.sentry.Sentry
import io.sentry.android.AndroidSentryClientFactory
import li.vin.my.apikit.network.ReceivedCookiesInterceptor
import okhttp3.Cache
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import java.io.File
import java.security.KeyStore
import java.util.*
import java.util.concurrent.TimeUnit
import javax.net.ssl.SSLContext
import javax.net.ssl.TrustManager
import javax.net.ssl.TrustManagerFactory
import javax.net.ssl.X509TrustManager

class ApiKit private constructor(val context: Context, val baseUrl: String, val cacheDir: File, val appName: String) {

    companion object {
        private var apikit: ApiKit? = null
        val okHttpClientBuilder = OkHttpClient.Builder()
        lateinit var okHttpClient: OkHttpClient
        var debugLogging = false
        var allowRefresh = false

        fun getInstance(application: Application, appName: String): ApiKit {
            if (apikit == null) {
                apikit = ApiKit(application.applicationContext, "vin.li", application.applicationContext.cacheDir, appName)
                initializeOkHttp(application.applicationContext, application, appName)
            }
            return apikit!!
        }

        fun getInstance(application: Application, baseUrl: String, appName: String): ApiKit {
            if (apikit == null) {
                apikit = ApiKit(application.applicationContext, baseUrl, application.applicationContext.cacheDir, appName)
                initializeOkHttp(application.applicationContext, application, appName)
            }
            return apikit!!
        }

        fun getInstance(application: Application, baseUrl: String, debugLogging: Boolean, appName: String): ApiKit {
            if (apikit == null) {
                apikit = ApiKit(application.applicationContext, baseUrl, application.applicationContext.cacheDir, appName)
                this.debugLogging = debugLogging
                initializeOkHttp(application.applicationContext, application, appName)
            }
            return apikit!!
        }

        fun getInstance(application: Application, baseUrl: String, cacheDir: File, appName: String): ApiKit {
            if (apikit == null) {
                apikit = ApiKit(application.applicationContext, baseUrl, cacheDir, appName)
                initializeOkHttp(application.applicationContext, application, appName)
            }
            return apikit!!
        }

        fun getInstance(application: Application, cacheDir: File, appName: String): ApiKit {
            if (apikit == null) {
                apikit = ApiKit(application.applicationContext, "vin.li", cacheDir, appName)
                initializeOkHttp(application.applicationContext, application, appName)
            }
            return apikit!!
        }

        private fun getAuthToken(): String {
            return AuthSettings.instance.authToken
        }

        private fun initializeOkHttp(context: Context, application: Application, appName: String) {
            Sentry.init("https://74084de8fefb43b7803ab4a5b55474bb@sentry.io/1935791", AndroidSentryClientFactory(application))
            AuthSettings.getInstance(context, appName)
            val cacheSize: Long = 10 * 1024 * 1024 // 10 MB
            val cacheDir = Cache(instance.cacheDir, cacheSize)
            okHttpClientBuilder.cache(cacheDir)
            val logging = HttpLoggingInterceptor()
            logging.level = if (debugLogging) HttpLoggingInterceptor.Level.BODY else HttpLoggingInterceptor.Level.BASIC
            okHttpClientBuilder.addInterceptor(logging)
            okHttpClientBuilder.addInterceptor(ReceivedCookiesInterceptor())
            if (AuthSettings.instance.authToken != "Bearer") {
                okHttpClientBuilder.addInterceptor(Interceptor { chain ->
                    val original = chain.request()
                    val requestBuilder = original.newBuilder()
                        .header("Authorization", getAuthToken())
                        .method(original.method, original.body)
                    val request = requestBuilder.build()
                    chain.proceed(request)
                })
            }
            val trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm())
            trustManagerFactory.init(null as KeyStore?)
            val trustManagers = trustManagerFactory.trustManagers
            check(!(trustManagers.size != 1 || trustManagers[0] !is X509TrustManager)) {
                ("Unexpected default trust managers:" + Arrays.toString(trustManagers))
            }
            val trustManager = trustManagers[0] as X509TrustManager
            val sslContext = SSLContext.getInstance("TLS")
            sslContext.init(null, arrayOf<TrustManager>(trustManager), null)
            okHttpClientBuilder.sslSocketFactory(sslContext.socketFactory, trustManager)
            okHttpClientBuilder.connectTimeout(45, TimeUnit.SECONDS)
            okHttpClientBuilder.readTimeout(45, TimeUnit.SECONDS)
            okHttpClient = okHttpClientBuilder.build()
        }

        val instance: ApiKit
            get() {
                if (apikit == null) {
                    throw IllegalStateException("Apikit context not set")
                }
                return apikit!!
            }
    }
}
