package li.vin.my.apikit

import android.content.Context
import android.content.SharedPreferences
import com.google.gson.Gson
import li.vin.my.apikit.models.LatestVehicle
import li.vin.my.apikit.models.Session
import li.vin.my.apikit.models.SessionData

/**
 * 11/5/18.
 */
class AuthSettings private constructor(appContext: Context) {
    private val sharedPreferences: SharedPreferences
    private val editor: SharedPreferences.Editor

    init {
        sharedPreferences = appContext.getSharedPreferences("li.vin.net.Vinli", Context.MODE_PRIVATE)
        editor = sharedPreferences.edit()
    }

    fun addListener(listener: SharedPreferences.OnSharedPreferenceChangeListener) {
        sharedPreferences.registerOnSharedPreferenceChangeListener(listener)
    }

    companion object {
        private var settings: AuthSettings? = null

        val instance: AuthSettings
            get() {
                if (settings == null) {
                    throw IllegalArgumentException("No Context set in AuthSettings")
                }
                return settings!!
            }

        fun getInstance(context: Context?): AuthSettings {
            if (context == null) {
                throw IllegalArgumentException("No Context set in AuthSettings")
            }
            if (settings == null) {
                settings = AuthSettings(context)
            }

            return settings!!
        }

        private val AUTH_TOKEN = "li.vin.net.Vinli#ACCESS_TOKEN"
        private val DEVICE_ID = "li.vin.my.app.net.SessionManager.deviceId"
        private val VEHICLE_ID = "VEHICLE_ID"
        private val COOKIE = "cookie"
        private val VEHICLE = "VEHICLE"
        private val SESSION = "SESSION"
    }

    fun getAuthToken(): String {
        sharedPreferences.getString(AUTH_TOKEN, "")?.let {
            return "Bearer " + it
        } ?: run {
            return "Bearer"
        }
    }

    fun getVehicleId(): String {
        sharedPreferences.getString(VEHICLE_ID, "")?.let {
            return it
        } ?: run {
            return ""
        }
    }

    fun getDeviceId(): String {
        sharedPreferences.getString(DEVICE_ID, "")?.let {
            return it
        } ?: run {
            return ""
        }
    }

    fun getSessionId(): String {
        sharedPreferences.getString(COOKIE, "")?.let {
            return it
        } ?: run {
            return ""
        }
    }

    fun getCurrentVehicleString(): String {
        sharedPreferences.getString(VEHICLE, "")?.let {
            return it
        } ?: run {
            return ""
        }
    }

    fun getCurrentSessionString(): String {
        sharedPreferences.getString(SESSION, "")?.let {
            return it
        } ?: run {
            return ""
        }
    }

    fun setAuthToken(value: String) {
        setEditPreferenceString(AUTH_TOKEN, value)
    }

    fun setDeviceId(value: String) {
        setEditPreferenceString(DEVICE_ID, value)
    }

    fun setVehicleId(value: String) {
        setEditPreferenceString(VEHICLE_ID, value)
    }

    fun setSessionId(value: String) {
        val sessionIdx = value.indexOf("session=")
        if (sessionIdx != -1) {
            var end = value.indexOf(";", sessionIdx)
            if (end == -1) end = value.length
            val cookie = "session=" + value.substring(sessionIdx + "session=".length, end).trim({ it <= ' ' })
            setEditPreferenceString(COOKIE, cookie)
        }
    }

    fun setCurrentVehicle(value: LatestVehicle?) {
        if (value != null) {
            val vehicleString = Gson().toJson(value)
            setEditPreferenceString(VEHICLE, vehicleString)
        } else {
            setEditPreferenceString(VEHICLE, "")
        }
    }

    fun setCurrentSession(value: Session?) {
        if (value != null) {
            val session = Gson().toJson(value)
            setEditPreferenceString(SESSION, session)
        } else {
            setEditPreferenceString(SESSION, "")
        }
    }

    /**
     * ------------ Setter Utilities ----------
     */

    fun keySession(key: String): Boolean {
        return SESSION == key
    }

    fun keyVehicle(key: String): Boolean {
        return VEHICLE_ID == key
    }

    fun keyDevice(key: String): Boolean {
        return DEVICE_ID == key
    }

    fun keyCookie(key: String): Boolean {
        return COOKIE == key
    }

    fun keyToken(key: String): Boolean {
        return AUTH_TOKEN == key
    }

    private fun setEditPreferenceStringSet(fileKey: String, fileValue: Set<String>) {
        editor.putStringSet(fileKey, fileValue)
        editor.commit()
    }

    private fun setEditPreferenceString(fileKey: String, fileValue: String) {
        editor.putString(fileKey, fileValue)
        editor.commit()
    }

    private fun setEditPreferenceBoolean(fileKey: String, fileValue: Boolean) {
        editor.putBoolean(fileKey, fileValue)
        editor.commit()
    }

    private fun setEditPreferenceInt(fileKey: String, fileValue: Int) {
        editor.putInt(fileKey, fileValue)
        editor.commit()
    }

    private fun setEditPreferenceLong(fileKey: String, fileValue: Long) {
        editor.putLong(fileKey, fileValue)
        editor.commit()
    }
}

fun String.getCurrentSession(): SessionData {
    return Gson().fromJson(this, SessionData::class.java)
}

fun String.getLatestVehicle(): LatestVehicle {
    return Gson().fromJson(this, LatestVehicle::class.java)
}