package li.vin.my.apikit.network

import android.util.Log
import com.google.gson.*
import li.vin.my.apikit.models.*
import java.lang.reflect.Type
import java.text.ParseException
import java.text.SimpleDateFormat
import java.util.*

/**
 * 11/6/18.
 */
object GsonConverter {
    private val DATE_FORMAT_STRINGS = arrayOf("yyyy-MM-dd'T'HH:mm:ss.SSS")

    val gsonConverter: Gson
        get() = GsonBuilder()
            .registerTypeAdapter(Date::class.java, DateSerializer())
            .registerTypeAdapter(BatteryColor::class.java, BatteryColorDeserializer())
            .registerTypeAdapter(TermsType::class.java, TermsTypeDeserializer())
            .registerTypeAdapter(EventType::class.java, EventTypeDeserializer())
            .registerTypeAdapter(AlertRuleType::class.java, AlertRuleTypeDeserializer())
            .registerTypeAdapter(AlertEventType::class.java, AlertEventTypeDeserializer())
            .create()

    internal class DateSerializer : JsonDeserializer<Date> {
        @Throws(JsonParseException::class)
        override fun deserialize(jsonElement: JsonElement, typeOF: Type, context: JsonDeserializationContext): Date? {

            val GMT_TIMEZONE = TimeZone.getTimeZone("GMT")
            for (format in DATE_FORMAT_STRINGS) {
                val dateFormat = SimpleDateFormat(format, Locale.US)
                dateFormat.timeZone = GMT_TIMEZONE

                try {
                    return dateFormat.parse(jsonElement.asString)
                } catch (e: ParseException) {
                    Log.e("GsonConverter", "GSON Date parsing failed: " + e.toString())
                }

            }
            return null
        }
    }

    internal class BatteryColorDeserializer : JsonDeserializer<BatteryColor> {

        @Throws(JsonParseException::class)
        override fun deserialize(json: JsonElement, typeOfT: Type, context: JsonDeserializationContext): BatteryColor {
            return BatteryColor.unitTypeByValue(json.asString)
        }
    }

    internal class TermsTypeDeserializer : JsonDeserializer<TermsType> {

        @Throws(JsonParseException::class)
        override fun deserialize(json: JsonElement, typeOfT: Type, context: JsonDeserializationContext): TermsType {
            return TermsType.unitTypeByValue(json.asString)
        }
    }

    internal class EventTypeDeserializer : JsonDeserializer<EventType> {

        @Throws(JsonParseException::class)
        override fun deserialize(json: JsonElement, typeOfT: Type, context: JsonDeserializationContext): EventType {
            return EventType.unitTypeByValue(json.asString)
        }
    }

    internal class AlertEventTypeDeserializer : JsonDeserializer<AlertEventType> {

        @Throws(JsonParseException::class)
        override fun deserialize(
            json: JsonElement,
            typeOfT: Type,
            context: JsonDeserializationContext
        ): AlertEventType {
            return AlertEventType.unitTypeByValue(json.asString)
        }
    }

    internal class AlertRuleTypeDeserializer : JsonDeserializer<AlertRuleType> {

        @Throws(JsonParseException::class)
        override fun deserialize(json: JsonElement, typeOfT: Type, context: JsonDeserializationContext): AlertRuleType {
            return AlertRuleType.unitTypeByValue(json.asString)
        }
    }
}
