package li.vin.my.apikit.network

import com.google.gson.Gson
import li.vin.my.apikit.ApiKit
import li.vin.my.apikit.AuthSettings
import li.vin.my.apikit.errors.ConflictException
import li.vin.my.apikit.errors.DataNotFoundException
import li.vin.my.apikit.errors.GenericError
import li.vin.my.apikit.errors.GenericErrorException
import okhttp3.Cache
import okhttp3.Dispatcher
import okhttp3.Interceptor
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import retrofit2.Call
import retrofit2.Callback
import retrofit2.Response
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory

//import retrofit2.converter.scalars.ScalarsConverterFactory

/**
 * 10/31/18.
 */

abstract class BaseService<Pojo> : Queueable, Poolable {

    lateinit var call: Call<Pojo>

    protected abstract fun createCall()

    lateinit var responseHandler: BaseResponseHandler

    private var serviceQueue: ServiceQueue? = null
    private var queuePosition: Int = 0
    private var result: Pojo? = null
    private lateinit var throwable: Throwable
    override var isCanceled: Boolean = false
    var successful: Boolean = false

    fun requestData(responseHandler: ResponseHandler<Pojo>) {
        this.responseHandler = responseHandler
        createCall()
        if (!isCanceled) {
            ServicePool.enter(this)
        }
    }

    fun getDeviceId(): String {
        if (ApiKit.instance.isDebug) {
            return "f3e779ea-56ba-41d7-90f9-c074e8a0ec6b"
        } else {
            return AuthSettings.instance.getDeviceId()
        }
    }

    fun getVehicleId(): String {
        if (ApiKit.instance.isDebug) {
            return "e977cf48-f30d-4d9c-b3a6-338be1ef7691"
        } else {
            return AuthSettings.instance.getVehicleId()
        }
    }

    fun getCookies(): String {
        if (ApiKit.instance.isDebug) {
            return "session=Fe26.2**4f8c7b90e3196ae979c1a089201e3e52cea7bdd12c19aebb45abb09b1392180c*lde_vt18XS8oMEptfCy2OQ*0J_IKMWmHmkw5_9h3Cl5oSN9Xm3WwC9A7IRUEXn_wSkO3bQhLCnHAXJVVTosi_85**a006d6a1498f339dc5b1ddc839376247006780bbc75229f2b9b1dfa5bddca15b*Bi3yiZ_qOjdK82KvASOVAmYPS8gD23ii2lXfsYahuss"
        } else {
            return AuthSettings.instance.getSessionId()
        }
    }

    fun getAuthToken(): String {
        if (ApiKit.instance.isDebug) {
            return "Bearer SBJRXcYsxe7jzHnRqgaK1vMyDaHKBesgDh9iigyoOGbMREzf_3NPdIAHRkWcy5hK"
        } else {
            return AuthSettings.instance.getAuthToken()
        }
    }

    private fun startService() {
        call.clone().enqueue(object : Callback<Pojo> {
            override fun onResponse(call: Call<Pojo>, response: Response<Pojo>) {
                if (response.isSuccessful) {
                    successful = true
                    result = response.body()
                } else {
                    if (response.code() == 409) {
                        successful = false
                        throwable = ConflictException()
                    } else {
                        if (response.body() == null || (response.body() is List<*> && (response.body() as List<*>).size == 0)) {
                            successful = false
                            throwable = DataNotFoundException()
                        } else if (response.code() != 200) {
                            throwable = GenericErrorException(getErrorMessage(response))
                            successful = false
                        }
                    }
                }
                ServicePool.leave(this@BaseService)
                endService()
            }

            override fun onFailure(call: Call<Pojo>, t: Throwable) {
                successful = false
                throwable = Exception(t)
                ServicePool.leave(this@BaseService)
                endService()
            }
        })
    }

    @Suppress("UNCHECKED_CAST")
    fun endService() {
        if (serviceQueue != null) {
            if (successful) {
                serviceQueue?.serviceCompleted(result as Any, queuePosition)
            } else {
                serviceQueue?.serviceError(throwable)
            }
            return
        }
        if (successful) {
            (responseHandler as ResponseHandler<Pojo>).onSuccess(result)
        } else {
            (responseHandler as ResponseHandler<Pojo>).onError(throwable)
        }
    }

    fun getErrorMessage(response: Response<Pojo>): String {
        var errorMessage = ""
        if (response.errorBody() != null) {
            val gson = Gson()
            val genericError = gson.fromJson(response.errorBody()?.string(), GenericError::class.java)
            errorMessage = genericError.message
        }
        return if (errorMessage.isEmpty()) {
            GenericError.getErrorMessage(response.code())
        } else {
            errorMessage
        }
    }

    fun <T> createService(serviceClass: Class<T>, baseUrl: String): T {
        val okHttpClientBuilder = OkHttpClient.Builder()

        if (AuthSettings.instance.getAuthToken() != "Bearer" || ApiKit.instance.isDebug) {
            okHttpClientBuilder.addInterceptor(Interceptor { chain ->
                val original = chain.request()
                val requestBuilder = original.newBuilder()
                    .header("Authorization", getAuthToken())
                    .method(original.method(), original.body())
                val request = requestBuilder.build()
                chain.proceed(request)
            })
        }
        val cacheSize: Long = 10 * 1024 * 1024 // 10 MB
        val cacheDir = Cache(ApiKit.instance.cacheDir, cacheSize)
        okHttpClientBuilder.cache(cacheDir)
        val logging = HttpLoggingInterceptor()
        logging.level = HttpLoggingInterceptor.Level.BODY
        okHttpClientBuilder.addInterceptor(logging)
        okHttpClientBuilder.addInterceptor(ReceivedCookiesInterceptor())
        return Retrofit.Builder()
            .baseUrl(baseUrl)
//            .addConverterFactory(ScalarsConverterFactory.create())
            .addConverterFactory(GsonConverterFactory.create(GsonConverter.gsonConverter))
            .client(okHttpClientBuilder.build())
            .build()
            .create(serviceClass)

    }

    override fun startServiceInQueue(serviceQueue: ServiceQueue, queuePosition: Int) {
        this.serviceQueue = serviceQueue
        this.queuePosition = queuePosition
        createCall()
        if (!isCanceled) {
            ServicePool.enter(this)
        }
    }

    override fun setBaseResponseHandler(responseHandler: BaseResponseHandler) {
        this.responseHandler = responseHandler
    }

    override fun execute() {
        startService()
    }

    @Suppress("UNCHECKED_CAST")
    override fun completed(poolable: Poolable) {
        if (poolable is BaseService<*>) {
            val duplicate = poolable.result as Pojo
            duplicate?.let {
                successful = true
                result = duplicate
                endService()
                return
            }
            poolable.throwable.let {
                successful = false
                throwable = it
                endService()
            }
        }
    }

    override val poolId: String
        get() {
            return call.request().url().toString()
        }
}

fun <Object> BaseService<Object>.setCanceled(canceled: Boolean) {
    isCanceled = canceled
}