/*
	* Copyright (C) 2002-2021 Sebastiano Vigna
	*
	* Licensed under the Apache License, Version 2.0 (the "License");
	* you may not use this file except in compliance with the License.
	* You may obtain a copy of the License at
	*
	*     http://www.apache.org/licenses/LICENSE-2.0
	*
	* Unless required by applicable law or agreed to in writing, software
	* distributed under the License is distributed on an "AS IS" BASIS,
	* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	* See the License for the specific language governing permissions and
	* limitations under the License.
	*/
package it.unimi.dsi.fastutil.ints;
import java.util.ListIterator;
/**
 * A type-specific bidirectional iterator that is also a {@link ListIterator}.
 *
 * <p>
 * This interface merges the methods provided by a {@link ListIterator} and a
 * type-specific {@link it.unimi.dsi.fastutil.BidirectionalIterator}. Moreover,
 * it provides type-specific versions of {@link ListIterator#add(Object) add()}
 * and {@link ListIterator#set(Object) set()}.
 *
 * @see java.util.ListIterator
 * @see it.unimi.dsi.fastutil.BidirectionalIterator
 */
public interface IntListIterator extends IntBidirectionalIterator, ListIterator<Integer> {
	/**
	 * Replaces the last element returned by {@link #next} or {@link #previous} with
	 * the specified element (optional operation).
	 * 
	 * @param k
	 *            the element used to replace the last element returned.
	 *
	 *            <p>
	 *            This default implementation just throws an
	 *            {@link UnsupportedOperationException}.
	 * @see ListIterator#set(Object)
	 */
	default void set(final int k) {
		throw new UnsupportedOperationException();
	}
	/**
	 * Inserts the specified element into the list (optional operation).
	 *
	 * <p>
	 * This default implementation just throws an
	 * {@link UnsupportedOperationException}.
	 * 
	 * @param k
	 *            the element to insert.
	 * @see ListIterator#add(Object)
	 */
	default void add(final int k) {
		throw new UnsupportedOperationException();
	}
	/**
	 * Removes from the underlying collection the last element returned by this
	 * iterator (optional operation).
	 *
	 * <p>
	 * This default implementation just throws an
	 * {@link UnsupportedOperationException}.
	 * 
	 * @see ListIterator#remove()
	 */
	@Override
	default void remove() {
		throw new UnsupportedOperationException();
	}
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	default void set(final Integer k) {
		set(k.intValue());
	}
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	default void add(final Integer k) {
		add(k.intValue());
	}
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	default Integer next() {
		return IntBidirectionalIterator.super.next();
	}
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@Deprecated
	@Override
	default Integer previous() {
		return IntBidirectionalIterator.super.previous();
	}
}
