/*
	* Copyright (C) 2002-2024 Sebastiano Vigna
	*
	* Licensed under the Apache License, Version 2.0 (the "License");
	* you may not use this file except in compliance with the License.
	* You may obtain a copy of the License at
	*
	*     http://www.apache.org/licenses/LICENSE-2.0
	*
	* Unless required by applicable law or agreed to in writing, software
	* distributed under the License is distributed on an "AS IS" BASIS,
	* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	* See the License for the specific language governing permissions and
	* limitations under the License.
	*/
package it.unimi.dsi.fastutil.ints;

import it.unimi.dsi.fastutil.Hash;
import it.unimi.dsi.fastutil.HashCommon;
import static it.unimi.dsi.fastutil.HashCommon.arraySize;
import static it.unimi.dsi.fastutil.HashCommon.maxFill;
import java.util.Map;
import java.util.Arrays;
import java.util.NoSuchElementException;
import java.util.function.Consumer;
import it.unimi.dsi.fastutil.objects.ObjectCollection;
import it.unimi.dsi.fastutil.objects.AbstractObjectCollection;
import java.util.Comparator;
import it.unimi.dsi.fastutil.objects.ObjectIterator;
import it.unimi.dsi.fastutil.objects.AbstractObjectSortedSet;
import it.unimi.dsi.fastutil.objects.ObjectListIterator;
import it.unimi.dsi.fastutil.objects.ObjectBidirectionalIterator;
import it.unimi.dsi.fastutil.objects.ObjectSpliterator;
import it.unimi.dsi.fastutil.objects.ObjectSpliterators;
import it.unimi.dsi.fastutil.objects.ObjectSortedSet;

/**
 * A type-specific linked hash map with with a fast, small-footprint implementation.
 *
 * <p>
 * Instances of this class use a hash table to represent a map. The table is filled up to a
 * specified <em>load factor</em>, and then doubled in size to accommodate new entries. If the table
 * is emptied below <em>one fourth</em> of the load factor, it is halved in size; however, the table
 * is never reduced to a size smaller than that at creation time: this approach makes it possible to
 * create maps with a large capacity in which insertions and deletions do not cause immediately
 * rehashing. Moreover, halving is not performed when deleting entries from an iterator, as it would
 * interfere with the iteration process.
 *
 * <p>
 * Note that {@link #clear()} does not modify the hash table size. Rather, a family of
 * {@linkplain #trim() trimming methods} lets you control the size of the table; this is
 * particularly useful if you reuse instances of this class.
 *
 * <p>
 * Entries returned by the type-specific {@link #entrySet()} method implement the suitable
 * type-specific {@link it.unimi.dsi.fastutil.Pair Pair} interface; only values are mutable.
 *
 * <p>
 * Iterators generated by this map will enumerate pairs in the same order in which they have been
 * added to the map (addition of pairs whose key is already present in the map does not change the
 * iteration order). Note that this order has nothing in common with the natural order of the keys.
 * The order is kept by means of a doubly linked list, represented <i>via</i> an array of longs
 * parallel to the table.
 *
 * <p>
 * This class implements the interface of a sorted map, so to allow easy access of the iteration
 * order: for instance, you can get the first key in iteration order with {@code firstKey()} without
 * having to create an iterator; however, this class partially violates the
 * {@link java.util.SortedMap} contract because all submap methods throw an exception and
 * {@link #comparator()} returns always {@code null}.
 *
 * <p>
 * Additional methods, such as {@code getAndMoveToFirst()}, make it easy to use instances of this
 * class as a cache (e.g., with LRU policy).
 *
 * <p>
 * The iterators provided by the views of this class using are type-specific
 * {@linkplain java.util.ListIterator list iterators}, and can be started at any element <em>which
 * is a key of the map</em>, or a {@link NoSuchElementException} exception will be thrown. If,
 * however, the provided element is not the first or last key in the map, the first access to the
 * list index will require linear time, as in the worst case the entire key set must be scanned in
 * iteration order to retrieve the positional index of the starting key. If you use just the methods
 * of a type-specific {@link it.unimi.dsi.fastutil.BidirectionalIterator}, however, all operations
 * will be performed in constant time.
 *
 * @see Hash
 * @see HashCommon
 */
public class Int2ObjectLinkedOpenHashMap<V> extends AbstractInt2ObjectSortedMap<V> implements java.io.Serializable, Cloneable, Hash {
	private static final long serialVersionUID = 0L;
	private static final boolean ASSERTS = false;
	/** The array of keys. */
	protected transient int[] key;
	/** The array of values. */
	protected transient V[] value;
	/** The mask for wrapping a position counter. */
	protected transient int mask;
	/** Whether this map contains the key zero. */
	protected transient boolean containsNullKey;
	/**
	 * The index of the first entry in iteration order. It is valid iff {@link #size} is nonzero;
	 * otherwise, it contains -1.
	 */
	protected transient int first = -1;
	/**
	 * The index of the last entry in iteration order. It is valid iff {@link #size} is nonzero;
	 * otherwise, it contains -1.
	 */
	protected transient int last = -1;
	/**
	 * For each entry, the next and the previous entry in iteration order, stored as
	 * {@code ((prev & 0xFFFFFFFFL) << 32) | (next & 0xFFFFFFFFL)}. The first entry contains predecessor
	 * -1, and the last entry contains successor -1.
	 */
	protected transient long[] link;
	/** The current table size. */
	protected transient int n;
	/** Threshold after which we rehash. It must be the table size times {@link #f}. */
	protected transient int maxFill;
	/** We never resize below this threshold, which is the construction-time {#n}. */
	protected final transient int minN;
	/** Number of entries in the set (including the key zero, if present). */
	protected int size;
	/** The acceptable load factor. */
	protected final float f;
	/** Cached set of entries. */
	protected transient FastSortedEntrySet<V> entries;
	/** Cached set of keys. */
	protected transient IntSortedSet keys;
	/** Cached collection of values. */
	protected transient ObjectCollection<V> values;

	/**
	 * Creates a new hash map.
	 *
	 * <p>
	 * The actual table size will be the least power of two greater than {@code expected}/{@code f}.
	 *
	 * @param expected the expected number of elements in the hash map.
	 * @param f the load factor.
	 */
	@SuppressWarnings("unchecked")
	public Int2ObjectLinkedOpenHashMap(final int expected, final float f) {
		if (f <= 0 || f >= 1) throw new IllegalArgumentException("Load factor must be greater than 0 and smaller than 1");
		if (expected < 0) throw new IllegalArgumentException("The expected number of elements must be nonnegative");
		this.f = f;
		minN = n = arraySize(expected, f);
		mask = n - 1;
		maxFill = maxFill(n, f);
		key = new int[n + 1];
		value = (V[])new Object[n + 1];
		link = new long[n + 1];
	}

	/**
	 * Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor.
	 *
	 * @param expected the expected number of elements in the hash map.
	 */
	public Int2ObjectLinkedOpenHashMap(final int expected) {
		this(expected, DEFAULT_LOAD_FACTOR);
	}

	/**
	 * Creates a new hash map with initial expected {@link Hash#DEFAULT_INITIAL_SIZE} entries and
	 * {@link Hash#DEFAULT_LOAD_FACTOR} as load factor.
	 */
	public Int2ObjectLinkedOpenHashMap() {
		this(DEFAULT_INITIAL_SIZE, DEFAULT_LOAD_FACTOR);
	}

	/**
	 * Creates a new hash map copying a given one.
	 *
	 * @param m a {@link Map} to be copied into the new hash map.
	 * @param f the load factor.
	 */
	public Int2ObjectLinkedOpenHashMap(final Map<? extends Integer, ? extends V> m, final float f) {
		this(m.size(), f);
		putAll(m);
	}

	/**
	 * Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor copying a given one.
	 *
	 * @param m a {@link Map} to be copied into the new hash map.
	 */
	public Int2ObjectLinkedOpenHashMap(final Map<? extends Integer, ? extends V> m) {
		this(m, DEFAULT_LOAD_FACTOR);
	}

	/**
	 * Creates a new hash map copying a given type-specific one.
	 *
	 * @param m a type-specific map to be copied into the new hash map.
	 * @param f the load factor.
	 */
	public Int2ObjectLinkedOpenHashMap(final Int2ObjectMap<V> m, final float f) {
		this(m.size(), f);
		putAll(m);
	}

	/**
	 * Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor copying a given
	 * type-specific one.
	 *
	 * @param m a type-specific map to be copied into the new hash map.
	 */
	public Int2ObjectLinkedOpenHashMap(final Int2ObjectMap<V> m) {
		this(m, DEFAULT_LOAD_FACTOR);
	}

	/**
	 * Creates a new hash map using the elements of two parallel arrays.
	 *
	 * @param k the array of keys of the new hash map.
	 * @param v the array of corresponding values in the new hash map.
	 * @param f the load factor.
	 * @throws IllegalArgumentException if {@code k} and {@code v} have different lengths.
	 */
	public Int2ObjectLinkedOpenHashMap(final int[] k, final V[] v, final float f) {
		this(k.length, f);
		if (k.length != v.length) throw new IllegalArgumentException("The key array and the value array have different lengths (" + k.length + " and " + v.length + ")");
		for (int i = 0; i < k.length; i++) this.put(k[i], v[i]);
	}

	/**
	 * Creates a new hash map with {@link Hash#DEFAULT_LOAD_FACTOR} as load factor using the elements of
	 * two parallel arrays.
	 *
	 * @param k the array of keys of the new hash map.
	 * @param v the array of corresponding values in the new hash map.
	 * @throws IllegalArgumentException if {@code k} and {@code v} have different lengths.
	 */
	public Int2ObjectLinkedOpenHashMap(final int[] k, final V[] v) {
		this(k, v, DEFAULT_LOAD_FACTOR);
	}

	private int realSize() {
		return containsNullKey ? size - 1 : size;
	}

	/**
	 * Ensures that this map can hold a certain number of keys without rehashing.
	 *
	 * @param capacity a number of keys; there will be no rehashing unless the map {@linkplain #size()
	 *            size} exceeds this number.
	 */
	public void ensureCapacity(final int capacity) {
		final int needed = arraySize(capacity, f);
		if (needed > n) rehash(needed);
	}

	private void tryCapacity(final long capacity) {
		final int needed = (int)Math.min(1 << 30, Math.max(2, HashCommon.nextPowerOfTwo((long)Math.ceil(capacity / f))));
		if (needed > n) rehash(needed);
	}

	private V removeEntry(final int pos) {
		final V oldValue = value[pos];
		value[pos] = null;
		size--;
		fixPointers(pos);
		shiftKeys(pos);
		if (n > minN && size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE) rehash(n / 2);
		return oldValue;
	}

	private V removeNullEntry() {
		containsNullKey = false;
		final V oldValue = value[n];
		value[n] = null;
		size--;
		fixPointers(n);
		if (n > minN && size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE) rehash(n / 2);
		return oldValue;
	}

	@Override
	public void putAll(Map<? extends Integer, ? extends V> m) {
		if (f <= .5) ensureCapacity(m.size()); // The resulting map will be sized for m.size() elements
		else tryCapacity(size() + m.size()); // The resulting map will be tentatively sized for size() + m.size()
												// elements
		super.putAll(m);
	}

	private int find(final int k) {
		if (((k) == (0))) return containsNullKey ? n : -(n + 1);
		int curr;
		final int[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k))) & mask]) == (0))) return -(pos + 1);
		if (((k) == (curr))) return pos;
		// There's always an unused entry.
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == (0))) return -(pos + 1);
			if (((k) == (curr))) return pos;
		}
	}

	private void insert(final int pos, final int k, final V v) {
		if (pos == n) containsNullKey = true;
		key[pos] = k;
		value[pos] = v;
		if (size == 0) {
			first = last = pos;
			// Special case of SET_UPPER_LOWER(link[pos], -1, -1);
			link[pos] = -1L;
		} else {
			link[last] ^= ((link[last] ^ (pos & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[pos] = ((last & 0xFFFFFFFFL) << 32) | (-1 & 0xFFFFFFFFL);
			last = pos;
		}
		if (size++ >= maxFill) rehash(arraySize(size + 1, f));
		if (ASSERTS) checkTable();
	}

	@Override
	public V put(final int k, final V v) {
		final int pos = find(k);
		if (pos < 0) {
			insert(-pos - 1, k, v);
			return defRetValue;
		}
		final V oldValue = value[pos];
		value[pos] = v;
		return oldValue;
	}

	/**
	 * Shifts left entries with the specified hash code, starting at the specified position, and empties
	 * the resulting free entry.
	 *
	 * @param pos a starting position.
	 */
	protected final void shiftKeys(int pos) {
		// Shift entries with the same hash.
		int last, slot;
		int curr;
		final int[] key = this.key;
		final V value[] = this.value;
		for (;;) {
			pos = ((last = pos) + 1) & mask;
			for (;;) {
				if (((curr = key[pos]) == (0))) {
					key[last] = (0);
					value[last] = null;
					return;
				}
				slot = (it.unimi.dsi.fastutil.HashCommon.mix((curr))) & mask;
				if (last <= pos ? last >= slot || slot > pos : last >= slot && slot > pos) break;
				pos = (pos + 1) & mask;
			}
			key[last] = curr;
			value[last] = value[pos];
			fixPointers(pos, last);
		}
	}

	@Override

	public V remove(final int k) {
		if (((k) == (0))) {
			if (containsNullKey) return removeNullEntry();
			return defRetValue;
		}
		int curr;
		final int[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k))) & mask]) == (0))) return defRetValue;
		if (((k) == (curr))) return removeEntry(pos);
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == (0))) return defRetValue;
			if (((k) == (curr))) return removeEntry(pos);
		}
	}

	private V setValue(final int pos, final V v) {
		final V oldValue = value[pos];
		value[pos] = v;
		return oldValue;
	}

	/**
	 * Removes the mapping associated with the first key in iteration order.
	 * 
	 * @return the value previously associated with the first key in iteration order.
	 * @throws NoSuchElementException is this map is empty.
	 */
	public V removeFirst() {
		if (size == 0) throw new NoSuchElementException();
		final int pos = first;
		// Abbreviated version of fixPointers(pos)
		if (size == 1) first = last = -1;
		else {
			first = (int)link[pos];
			if (0 <= first) {
				// Special case of SET_PREV(link[first], -1)
				link[first] |= (-1 & 0xFFFFFFFFL) << 32;
			}
		}
		size--;
		final V v = value[pos];
		if (pos == n) {
			containsNullKey = false;
			value[n] = null;
		} else shiftKeys(pos);
		if (n > minN && size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE) rehash(n / 2);
		return v;
	}

	/**
	 * Removes the mapping associated with the last key in iteration order.
	 * 
	 * @return the value previously associated with the last key in iteration order.
	 * @throws NoSuchElementException is this map is empty.
	 */
	public V removeLast() {
		if (size == 0) throw new NoSuchElementException();
		final int pos = last;
		// Abbreviated version of fixPointers(pos)
		if (size == 1) first = last = -1;
		else {
			last = (int)(link[pos] >>> 32);
			if (0 <= last) {
				// Special case of SET_NEXT(link[last], -1)
				link[last] |= -1 & 0xFFFFFFFFL;
			}
		}
		size--;
		final V v = value[pos];
		if (pos == n) {
			containsNullKey = false;
			value[n] = null;
		} else shiftKeys(pos);
		if (n > minN && size < maxFill / 4 && n > DEFAULT_INITIAL_SIZE) rehash(n / 2);
		return v;
	}

	private void moveIndexToFirst(final int i) {
		if (size == 1 || first == i) return;
		if (last == i) {
			last = (int)(link[i] >>> 32);
			// Special case of SET_NEXT(link[last], -1);
			link[last] |= -1 & 0xFFFFFFFFL;
		} else {
			final long linki = link[i];
			final int prev = (int)(linki >>> 32);
			final int next = (int)linki;
			link[prev] ^= ((link[prev] ^ (linki & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[next] ^= ((link[next] ^ (linki & 0xFFFFFFFF00000000L)) & 0xFFFFFFFF00000000L);
		}
		link[first] ^= ((link[first] ^ ((i & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
		link[i] = ((-1 & 0xFFFFFFFFL) << 32) | (first & 0xFFFFFFFFL);
		first = i;
	}

	private void moveIndexToLast(final int i) {
		if (size == 1 || last == i) return;
		if (first == i) {
			first = (int)link[i];
			// Special case of SET_PREV(link[first], -1);
			link[first] |= (-1 & 0xFFFFFFFFL) << 32;
		} else {
			final long linki = link[i];
			final int prev = (int)(linki >>> 32);
			final int next = (int)linki;
			link[prev] ^= ((link[prev] ^ (linki & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[next] ^= ((link[next] ^ (linki & 0xFFFFFFFF00000000L)) & 0xFFFFFFFF00000000L);
		}
		link[last] ^= ((link[last] ^ (i & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
		link[i] = ((last & 0xFFFFFFFFL) << 32) | (-1 & 0xFFFFFFFFL);
		last = i;
	}

	/**
	 * Returns the value to which the given key is mapped; if the key is present, it is moved to the
	 * first position of the iteration order.
	 *
	 * @param k the key.
	 * @return the corresponding value, or the {@linkplain #defaultReturnValue() default return value}
	 *         if no value was present for the given key.
	 */
	public V getAndMoveToFirst(final int k) {
		if (((k) == (0))) {
			if (containsNullKey) {
				moveIndexToFirst(n);
				return value[n];
			}
			return defRetValue;
		}
		int curr;
		final int[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k))) & mask]) == (0))) return defRetValue;
		if (((k) == (curr))) {
			moveIndexToFirst(pos);
			return value[pos];
		}
		// There's always an unused entry.
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == (0))) return defRetValue;
			if (((k) == (curr))) {
				moveIndexToFirst(pos);
				return value[pos];
			}
		}
	}

	/**
	 * Returns the value to which the given key is mapped; if the key is present, it is moved to the
	 * last position of the iteration order.
	 *
	 * @param k the key.
	 * @return the corresponding value, or the {@linkplain #defaultReturnValue() default return value}
	 *         if no value was present for the given key.
	 */
	public V getAndMoveToLast(final int k) {
		if (((k) == (0))) {
			if (containsNullKey) {
				moveIndexToLast(n);
				return value[n];
			}
			return defRetValue;
		}
		int curr;
		final int[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k))) & mask]) == (0))) return defRetValue;
		if (((k) == (curr))) {
			moveIndexToLast(pos);
			return value[pos];
		}
		// There's always an unused entry.
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == (0))) return defRetValue;
			if (((k) == (curr))) {
				moveIndexToLast(pos);
				return value[pos];
			}
		}
	}

	/**
	 * Adds a pair to the map; if the key is already present, it is moved to the first position of the
	 * iteration order.
	 *
	 * @param k the key.
	 * @param v the value.
	 * @return the old value, or the {@linkplain #defaultReturnValue() default return value} if no value
	 *         was present for the given key.
	 */
	public V putAndMoveToFirst(final int k, final V v) {
		int pos;
		if (((k) == (0))) {
			if (containsNullKey) {
				moveIndexToFirst(n);
				return setValue(n, v);
			}
			containsNullKey = true;
			pos = n;
		} else {
			int curr;
			final int[] key = this.key;
			// The starting point.
			if (!((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k))) & mask]) == (0))) {
				if (((curr) == (k))) {
					moveIndexToFirst(pos);
					return setValue(pos, v);
				}
				while (!((curr = key[pos = (pos + 1) & mask]) == (0))) if (((curr) == (k))) {
					moveIndexToFirst(pos);
					return setValue(pos, v);
				}
			}
		}
		key[pos] = k;
		value[pos] = v;
		if (size == 0) {
			first = last = pos;
			// Special case of SET_UPPER_LOWER(link[pos], -1, -1);
			link[pos] = -1L;
		} else {
			link[first] ^= ((link[first] ^ ((pos & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
			link[pos] = ((-1 & 0xFFFFFFFFL) << 32) | (first & 0xFFFFFFFFL);
			first = pos;
		}
		if (size++ >= maxFill) rehash(arraySize(size, f));
		if (ASSERTS) checkTable();
		return defRetValue;
	}

	/**
	 * Adds a pair to the map; if the key is already present, it is moved to the last position of the
	 * iteration order.
	 *
	 * @param k the key.
	 * @param v the value.
	 * @return the old value, or the {@linkplain #defaultReturnValue() default return value} if no value
	 *         was present for the given key.
	 */
	public V putAndMoveToLast(final int k, final V v) {
		int pos;
		if (((k) == (0))) {
			if (containsNullKey) {
				moveIndexToLast(n);
				return setValue(n, v);
			}
			containsNullKey = true;
			pos = n;
		} else {
			int curr;
			final int[] key = this.key;
			// The starting point.
			if (!((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k))) & mask]) == (0))) {
				if (((curr) == (k))) {
					moveIndexToLast(pos);
					return setValue(pos, v);
				}
				while (!((curr = key[pos = (pos + 1) & mask]) == (0))) if (((curr) == (k))) {
					moveIndexToLast(pos);
					return setValue(pos, v);
				}
			}
		}
		key[pos] = k;
		value[pos] = v;
		if (size == 0) {
			first = last = pos;
			// Special case of SET_UPPER_LOWER(link[pos], -1, -1);
			link[pos] = -1L;
		} else {
			link[last] ^= ((link[last] ^ (pos & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[pos] = ((last & 0xFFFFFFFFL) << 32) | (-1 & 0xFFFFFFFFL);
			last = pos;
		}
		if (size++ >= maxFill) rehash(arraySize(size, f));
		if (ASSERTS) checkTable();
		return defRetValue;
	}

	@Override

	public V get(final int k) {
		if (((k) == (0))) return containsNullKey ? value[n] : defRetValue;
		int curr;
		final int[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k))) & mask]) == (0))) return defRetValue;
		if (((k) == (curr))) return value[pos];
		// There's always an unused entry.
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == (0))) return defRetValue;
			if (((k) == (curr))) return value[pos];
		}
	}

	@Override

	public boolean containsKey(final int k) {
		if (((k) == (0))) return containsNullKey;
		int curr;
		final int[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k))) & mask]) == (0))) return false;
		if (((k) == (curr))) return true;
		// There's always an unused entry.
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == (0))) return false;
			if (((k) == (curr))) return true;
		}
	}

	@Override
	public boolean containsValue(final Object v) {
		final int key[] = this.key;
		final V value[] = this.value;
		if (containsNullKey && java.util.Objects.equals(value[n], v)) return true;
		for (int i = n; i-- != 0;) if (!((key[i]) == (0)) && java.util.Objects.equals(value[i], v)) return true;
		return false;
	}

	/** {@inheritDoc} */
	@Override

	public V getOrDefault(final int k, final V defaultValue) {
		if (((k) == (0))) return containsNullKey ? value[n] : defaultValue;
		int curr;
		final int[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k))) & mask]) == (0))) return defaultValue;
		if (((k) == (curr))) return value[pos];
		// There's always an unused entry.
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == (0))) return defaultValue;
			if (((k) == (curr))) return value[pos];
		}
	}

	/** {@inheritDoc} */
	@Override
	public V putIfAbsent(final int k, final V v) {
		final int pos = find(k);
		if (pos >= 0) return value[pos];
		insert(-pos - 1, k, v);
		return defRetValue;
	}

	/** {@inheritDoc} */
	@Override

	public boolean remove(final int k, final Object v) {
		if (((k) == (0))) {
			if (containsNullKey && java.util.Objects.equals(v, value[n])) {
				removeNullEntry();
				return true;
			}
			return false;
		}
		int curr;
		final int[] key = this.key;
		int pos;
		// The starting point.
		if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k))) & mask]) == (0))) return false;
		if (((k) == (curr)) && java.util.Objects.equals(v, value[pos])) {
			removeEntry(pos);
			return true;
		}
		while (true) {
			if (((curr = key[pos = (pos + 1) & mask]) == (0))) return false;
			if (((k) == (curr)) && java.util.Objects.equals(v, value[pos])) {
				removeEntry(pos);
				return true;
			}
		}
	}

	/** {@inheritDoc} */
	@Override
	public boolean replace(final int k, final V oldValue, final V v) {
		final int pos = find(k);
		if (pos < 0 || !java.util.Objects.equals(oldValue, value[pos])) return false;
		value[pos] = v;
		return true;
	}

	/** {@inheritDoc} */
	@Override
	public V replace(final int k, final V v) {
		final int pos = find(k);
		if (pos < 0) return defRetValue;
		final V oldValue = value[pos];
		value[pos] = v;
		return oldValue;
	}

	/** {@inheritDoc} */
	@Override
	public V computeIfAbsent(final int k, final java.util.function.IntFunction<? extends V> mappingFunction) {
		java.util.Objects.requireNonNull(mappingFunction);
		final int pos = find(k);
		if (pos >= 0) return value[pos];
		final V newValue = mappingFunction.apply(k);
		insert(-pos - 1, k, newValue);
		return newValue;
	}

	/** {@inheritDoc} */
	@Override
	public V computeIfAbsent(final int key, final Int2ObjectFunction<? extends V> mappingFunction) {
		java.util.Objects.requireNonNull(mappingFunction);
		final int pos = find(key);
		if (pos >= 0) return value[pos];
		if (!mappingFunction.containsKey(key)) return defRetValue;
		final V newValue = mappingFunction.get(key);
		insert(-pos - 1, key, newValue);
		return newValue;
	}

	/** {@inheritDoc} */
	@Override
	public V computeIfPresent(final int k, final java.util.function.BiFunction<? super Integer, ? super V, ? extends V> remappingFunction) {
		java.util.Objects.requireNonNull(remappingFunction);
		final int pos = find(k);
		if (pos < 0) return defRetValue;
		if (value[pos] == null) return defRetValue;
		final V newValue = remappingFunction.apply(Integer.valueOf(k), (value[pos]));
		if (newValue == null) {
			if (((k) == (0))) removeNullEntry();
			else removeEntry(pos);
			return defRetValue;
		}
		return value[pos] = (newValue);
	}

	/** {@inheritDoc} */
	@Override
	public V compute(final int k, final java.util.function.BiFunction<? super Integer, ? super V, ? extends V> remappingFunction) {
		java.util.Objects.requireNonNull(remappingFunction);
		final int pos = find(k);
		final V newValue = remappingFunction.apply(Integer.valueOf(k), pos >= 0 ? (value[pos]) : null);
		if (newValue == null) {
			if (pos >= 0) {
				if (((k) == (0))) removeNullEntry();
				else removeEntry(pos);
			}
			return defRetValue;
		}
		V newVal = (newValue);
		if (pos < 0) {
			insert(-pos - 1, k, newVal);
			return newVal;
		}
		return value[pos] = newVal;
	}

	/** {@inheritDoc} */
	@Override
	public V merge(final int k, final V v, final java.util.function.BiFunction<? super V, ? super V, ? extends V> remappingFunction) {
		java.util.Objects.requireNonNull(remappingFunction);
		java.util.Objects.requireNonNull(v);
		final int pos = find(k);
		if (pos < 0 || value[pos] == null) {
			if (pos < 0) insert(-pos - 1, k, v);
			else value[pos] = v;
			return v;
		}
		final V newValue = remappingFunction.apply((value[pos]), (v));
		if (newValue == null) {
			if (((k) == (0))) removeNullEntry();
			else removeEntry(pos);
			return defRetValue;
		}
		return value[pos] = (newValue);
	}

	/* Removes all elements from this map.
	 *
	 * <p>To increase object reuse, this method does not change the table size.
	 * If you want to reduce the table size, you must use {@link #trim()}.
	 *
	 */
	@Override
	public void clear() {
		if (size == 0) return;
		size = 0;
		containsNullKey = false;
		Arrays.fill(key, (0));
		Arrays.fill(value, null);
		first = last = -1;
	}

	@Override
	public int size() {
		return size;
	}

	@Override
	public boolean isEmpty() {
		return size == 0;
	}

	/**
	 * The entry class for a hash map does not record key and value, but rather the position in the hash
	 * table of the corresponding entry. This is necessary so that calls to
	 * {@link java.util.Map.Entry#setValue(Object)} are reflected in the map
	 */
	final class MapEntry implements Int2ObjectMap.Entry<V>, Map.Entry<Integer, V>, IntObjectPair<V> {
		// The table index this entry refers to, or -1 if this entry has been deleted.
		int index;

		MapEntry(final int index) {
			this.index = index;
		}

		MapEntry() {
		}

		@Override
		public int getIntKey() {
			return key[index];
		}

		@Override
		public int leftInt() {
			return key[index];
		}

		@Override
		public V getValue() {
			return value[index];
		}

		@Override
		public V right() {
			return value[index];
		}

		@Override
		public V setValue(final V v) {
			final V oldValue = value[index];
			value[index] = v;
			return oldValue;
		}

		@Override
		public IntObjectPair<V> right(final V v) {
			value[index] = v;
			return this;
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method instead.
		 */
		@Deprecated
		@Override
		public Integer getKey() {
			return Integer.valueOf(key[index]);
		}

		@SuppressWarnings("unchecked")
		@Override
		public boolean equals(final Object o) {
			if (!(o instanceof Map.Entry)) return false;
			Map.Entry<Integer, V> e = (Map.Entry<Integer, V>)o;
			return ((key[index]) == ((e.getKey()).intValue())) && java.util.Objects.equals(value[index], (e.getValue()));
		}

		@Override
		public int hashCode() {
			return (key[index]) ^ ((value[index]) == null ? 0 : (value[index]).hashCode());
		}

		@Override
		public String toString() {
			return key[index] + "=>" + value[index];
		}
	}

	/**
	 * Modifies the {@link #link} vector so that the given entry is removed. This method will complete
	 * in constant time.
	 *
	 * @param i the index of an entry.
	 */
	protected void fixPointers(final int i) {
		if (size == 0) {
			first = last = -1;
			return;
		}
		if (first == i) {
			first = (int)link[i];
			if (0 <= first) {
				// Special case of SET_PREV(link[first], -1)
				link[first] |= (-1 & 0xFFFFFFFFL) << 32;
			}
			return;
		}
		if (last == i) {
			last = (int)(link[i] >>> 32);
			if (0 <= last) {
				// Special case of SET_NEXT(link[last], -1)
				link[last] |= -1 & 0xFFFFFFFFL;
			}
			return;
		}
		final long linki = link[i];
		final int prev = (int)(linki >>> 32);
		final int next = (int)linki;
		link[prev] ^= ((link[prev] ^ (linki & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
		link[next] ^= ((link[next] ^ (linki & 0xFFFFFFFF00000000L)) & 0xFFFFFFFF00000000L);
	}

	/**
	 * Modifies the {@link #link} vector for a shift from s to d.
	 * <p>
	 * This method will complete in constant time.
	 *
	 * @param s the source position.
	 * @param d the destination position.
	 */
	protected void fixPointers(int s, int d) {
		if (size == 1) {
			first = last = d;
			// Special case of SET_UPPER_LOWER(link[d], -1, -1)
			link[d] = -1L;
			return;
		}
		if (first == s) {
			first = d;
			link[(int)link[s]] ^= ((link[(int)link[s]] ^ ((d & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
			link[d] = link[s];
			return;
		}
		if (last == s) {
			last = d;
			link[(int)(link[s] >>> 32)] ^= ((link[(int)(link[s] >>> 32)] ^ (d & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			link[d] = link[s];
			return;
		}
		final long links = link[s];
		final int prev = (int)(links >>> 32);
		final int next = (int)links;
		link[prev] ^= ((link[prev] ^ (d & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
		link[next] ^= ((link[next] ^ ((d & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
		link[d] = links;
	}

	/**
	 * Returns the first key of this map in iteration order.
	 *
	 * @return the first key in iteration order.
	 */
	@Override
	public int firstIntKey() {
		if (size == 0) throw new NoSuchElementException();
		return key[first];
	}

	/**
	 * Returns the last key of this map in iteration order.
	 *
	 * @return the last key in iteration order.
	 */
	@Override
	public int lastIntKey() {
		if (size == 0) throw new NoSuchElementException();
		return key[last];
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @implSpec This implementation just throws an {@link UnsupportedOperationException}.
	 */
	@Override
	public Int2ObjectSortedMap<V> tailMap(int from) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @implSpec This implementation just throws an {@link UnsupportedOperationException}.
	 */
	@Override
	public Int2ObjectSortedMap<V> headMap(int to) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @implSpec This implementation just throws an {@link UnsupportedOperationException}.
	 */
	@Override
	public Int2ObjectSortedMap<V> subMap(int from, int to) {
		throw new UnsupportedOperationException();
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @implSpec This implementation just returns {@code null}.
	 */
	@Override
	public IntComparator comparator() {
		return null;
	}

	/**
	 * A list iterator over a linked map.
	 *
	 * <p>
	 * This class provides a list iterator over a linked hash map. The constructor runs in constant
	 * time.
	 */
	private abstract class MapIterator<ConsumerType> {
		/**
		 * The entry that will be returned by the next call to {@link java.util.ListIterator#previous()} (or
		 * {@code null} if no previous entry exists).
		 */
		int prev = -1;
		/**
		 * The entry that will be returned by the next call to {@link java.util.ListIterator#next()} (or
		 * {@code null} if no next entry exists).
		 */
		int next = -1;
		/**
		 * The last entry that was returned (or -1 if we did not iterate or used
		 * {@link java.util.Iterator#remove()}).
		 */
		int curr = -1;
		/**
		 * The current index (in the sense of a {@link java.util.ListIterator}). Note that this value is not
		 * meaningful when this iterator has been created using the nonempty constructor.
		 */
		int index = -1;

		@SuppressWarnings("unused")
		abstract void acceptOnIndex(final ConsumerType action, final int index);

		protected MapIterator() {
			next = first;
			index = 0;
		}

		private MapIterator(final int from) {
			if (((from) == (0))) {
				if (Int2ObjectLinkedOpenHashMap.this.containsNullKey) {
					next = (int)link[n];
					prev = n;
					return;
				} else throw new NoSuchElementException("The key " + from + " does not belong to this map.");
			}
			if (((key[last]) == (from))) {
				prev = last;
				index = size;
				return;
			}
			// The starting point.
			int pos = (it.unimi.dsi.fastutil.HashCommon.mix((from))) & mask;
			// There's always an unused entry.
			while (!((key[pos]) == (0))) {
				if (((key[pos]) == (from))) {
					// Note: no valid index known.
					next = (int)link[pos];
					prev = pos;
					return;
				}
				pos = (pos + 1) & mask;
			}
			throw new NoSuchElementException("The key " + from + " does not belong to this map.");
		}

		public boolean hasNext() {
			return next != -1;
		}

		public boolean hasPrevious() {
			return prev != -1;
		}

		private final void ensureIndexKnown() {
			if (index >= 0) return;
			if (prev == -1) {
				index = 0;
				return;
			}
			if (next == -1) {
				index = size;
				return;
			}
			int pos = first;
			index = 1;
			while (pos != prev) {
				pos = (int)link[pos];
				index++;
			}
		}

		public int nextIndex() {
			ensureIndexKnown();
			return index;
		}

		public int previousIndex() {
			ensureIndexKnown();
			return index - 1;
		}

		public int nextEntry() {
			if (!hasNext()) throw new NoSuchElementException();
			curr = next;
			next = (int)link[curr];
			prev = curr;
			if (index >= 0) index++;
			return curr;
		}

		public int previousEntry() {
			if (!hasPrevious()) throw new NoSuchElementException();
			curr = prev;
			prev = (int)(link[curr] >>> 32);
			next = curr;
			if (index >= 0) index--;
			return curr;
		}

		public void forEachRemaining(final ConsumerType action) {
			while (hasNext()) {
				curr = next;
				next = (int)link[curr];
				prev = curr;
				if (index >= 0) index++;
				acceptOnIndex(action, curr);
			}
		}

		public void remove() {
			ensureIndexKnown();
			if (curr == -1) throw new IllegalStateException();
			if (curr == prev) {
				/* If the last operation was a next(), we are removing an entry that preceeds
						   the current index, and thus we must decrement it. */
				index--;
				prev = (int)(link[curr] >>> 32);
			} else next = (int)link[curr];
			size--;
			/* Now we manually fix the pointers. Because of our knowledge of next
				   and prev, this is going to be faster than calling fixPointers(). */
			if (prev == -1) first = next;
			else link[prev] ^= ((link[prev] ^ (next & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
			if (next == -1) last = prev;
			else link[next] ^= ((link[next] ^ ((prev & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
			int last, slot, pos = curr;
			curr = -1;
			if (pos == n) {
				Int2ObjectLinkedOpenHashMap.this.containsNullKey = false;
				value[n] = null;
			} else {
				int curr;
				final int[] key = Int2ObjectLinkedOpenHashMap.this.key;
				final V value[] = Int2ObjectLinkedOpenHashMap.this.value;
				// We have to horribly duplicate the shiftKeys() code because we need to update next/prev.
				for (;;) {
					pos = ((last = pos) + 1) & mask;
					for (;;) {
						if (((curr = key[pos]) == (0))) {
							key[last] = (0);
							value[last] = null;
							return;
						}
						slot = (it.unimi.dsi.fastutil.HashCommon.mix((curr))) & mask;
						if (last <= pos ? last >= slot || slot > pos : last >= slot && slot > pos) break;
						pos = (pos + 1) & mask;
					}
					key[last] = curr;
					value[last] = value[pos];
					if (next == pos) next = last;
					if (prev == pos) prev = last;
					fixPointers(pos, last);
				}
			}
		}

		public int skip(final int n) {
			int i = n;
			while (i-- != 0 && hasNext()) nextEntry();
			return n - i - 1;
		}

		public int back(final int n) {
			int i = n;
			while (i-- != 0 && hasPrevious()) previousEntry();
			return n - i - 1;
		}

		public void set(@SuppressWarnings("unused") Int2ObjectMap.Entry<V> ok) {
			throw new UnsupportedOperationException();
		}

		public void add(@SuppressWarnings("unused") Int2ObjectMap.Entry<V> ok) {
			throw new UnsupportedOperationException();
		}
	}

	private final class EntryIterator extends MapIterator<Consumer<? super Int2ObjectMap.Entry<V>>> implements ObjectListIterator<Int2ObjectMap.Entry<V>> {
		private MapEntry entry;

		public EntryIterator() {
		}

		public EntryIterator(int from) {
			super(from);
		}

		// forEachRemaining inherited from MapIterator superclass.
		@Override
		final void acceptOnIndex(final Consumer<? super Int2ObjectMap.Entry<V>> action, final int index) {
			action.accept(new MapEntry(index));
		}

		@Override
		public MapEntry next() {
			return entry = new MapEntry(nextEntry());
		}

		@Override
		public MapEntry previous() {
			return entry = new MapEntry(previousEntry());
		}

		@Override
		public void remove() {
			super.remove();
			entry.index = -1; // You cannot use a deleted entry.
		}
	}

	private final class FastEntryIterator extends MapIterator<Consumer<? super Int2ObjectMap.Entry<V>>> implements ObjectListIterator<Int2ObjectMap.Entry<V>> {
		final MapEntry entry = new MapEntry();

		public FastEntryIterator() {
		}

		public FastEntryIterator(int from) {
			super(from);
		}

		// forEachRemaining inherited from MapIterator superclass.
		@Override
		final void acceptOnIndex(final Consumer<? super Int2ObjectMap.Entry<V>> action, final int index) {
			entry.index = index;
			action.accept(entry);
		}

		@Override
		public MapEntry next() {
			entry.index = nextEntry();
			return entry;
		}

		@Override
		public MapEntry previous() {
			entry.index = previousEntry();
			return entry;
		}
	}

	private final class MapEntrySet extends AbstractObjectSortedSet<Int2ObjectMap.Entry<V>> implements FastSortedEntrySet<V> {
		private static final int SPLITERATOR_CHARACTERISTICS = ObjectSpliterators.SET_SPLITERATOR_CHARACTERISTICS | java.util.Spliterator.ORDERED;

		@Override
		public ObjectBidirectionalIterator<Int2ObjectMap.Entry<V>> iterator() {
			return new EntryIterator();
		}

		/**
		 * {@inheritDoc}
		 *
		 * <p>
		 * There isn't a way to split efficiently while still preserving order for a linked data structure,
		 * so this implementation is just backed by the iterator. Thus, this spliterator is not well
		 * optimized for parallel streams.
		 *
		 * <p>
		 * Note, contrary to the specification of {@link java.util.SortedSet}, this spliterator does not,
		 * report {@link java.util.Spliterators.SORTED}. This is because iteration order is based on
		 * insertion order, not natural ordering.
		 */
		@Override
		public ObjectSpliterator<Int2ObjectMap.Entry<V>> spliterator() {
			return ObjectSpliterators.asSpliterator(iterator(), it.unimi.dsi.fastutil.Size64.sizeOf(Int2ObjectLinkedOpenHashMap.this), SPLITERATOR_CHARACTERISTICS);
		}

		@Override
		public Comparator<? super Int2ObjectMap.Entry<V>> comparator() {
			return null;
		}

		@Override
		public ObjectSortedSet<Int2ObjectMap.Entry<V>> subSet(Int2ObjectMap.Entry<V> fromElement, Int2ObjectMap.Entry<V> toElement) {
			throw new UnsupportedOperationException();
		}

		@Override
		public ObjectSortedSet<Int2ObjectMap.Entry<V>> headSet(Int2ObjectMap.Entry<V> toElement) {
			throw new UnsupportedOperationException();
		}

		@Override
		public ObjectSortedSet<Int2ObjectMap.Entry<V>> tailSet(Int2ObjectMap.Entry<V> fromElement) {
			throw new UnsupportedOperationException();
		}

		@Override
		public Int2ObjectMap.Entry<V> first() {
			if (size == 0) throw new NoSuchElementException();
			return new MapEntry(Int2ObjectLinkedOpenHashMap.this.first);
		}

		@Override
		public Int2ObjectMap.Entry<V> last() {
			if (size == 0) throw new NoSuchElementException();
			return new MapEntry(Int2ObjectLinkedOpenHashMap.this.last);
		}

		@Override
		@SuppressWarnings("unchecked")
		public boolean contains(final Object o) {
			if (!(o instanceof Map.Entry)) return false;
			final Map.Entry<?, ?> e = (Map.Entry<?, ?>)o;
			if (e.getKey() == null || !(e.getKey() instanceof Integer)) return false;
			final int k = ((Integer)(e.getKey())).intValue();
			final V v = ((V)e.getValue());
			if (((k) == (0))) return Int2ObjectLinkedOpenHashMap.this.containsNullKey && java.util.Objects.equals(value[n], v);
			int curr;
			final int[] key = Int2ObjectLinkedOpenHashMap.this.key;
			int pos;
			// The starting point.
			if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k))) & mask]) == (0))) return false;
			if (((k) == (curr))) return java.util.Objects.equals(value[pos], v);
			// There's always an unused entry.
			while (true) {
				if (((curr = key[pos = (pos + 1) & mask]) == (0))) return false;
				if (((k) == (curr))) return java.util.Objects.equals(value[pos], v);
			}
		}

		@Override
		@SuppressWarnings("unchecked")
		public boolean remove(final Object o) {
			if (!(o instanceof Map.Entry)) return false;
			final Map.Entry<?, ?> e = (Map.Entry<?, ?>)o;
			if (e.getKey() == null || !(e.getKey() instanceof Integer)) return false;
			final int k = ((Integer)(e.getKey())).intValue();
			final V v = ((V)e.getValue());
			if (((k) == (0))) {
				if (containsNullKey && java.util.Objects.equals(value[n], v)) {
					removeNullEntry();
					return true;
				}
				return false;
			}
			int curr;
			final int[] key = Int2ObjectLinkedOpenHashMap.this.key;
			int pos;
			// The starting point.
			if (((curr = key[pos = (it.unimi.dsi.fastutil.HashCommon.mix((k))) & mask]) == (0))) return false;
			if (((curr) == (k))) {
				if (java.util.Objects.equals(value[pos], v)) {
					removeEntry(pos);
					return true;
				}
				return false;
			}
			while (true) {
				if (((curr = key[pos = (pos + 1) & mask]) == (0))) return false;
				if (((curr) == (k))) {
					if (java.util.Objects.equals(value[pos], v)) {
						removeEntry(pos);
						return true;
					}
				}
			}
		}

		@Override
		public int size() {
			return size;
		}

		@Override
		public void clear() {
			Int2ObjectLinkedOpenHashMap.this.clear();
		}

		/**
		 * Returns a type-specific list iterator on the elements in this set, starting from a given element
		 * of the set. Please see the class documentation for implementation details.
		 *
		 * @param from an element to start from.
		 * @return a type-specific list iterator starting at the given element.
		 * @throws IllegalArgumentException if {@code from} does not belong to the set.
		 */
		@Override
		public ObjectListIterator<Int2ObjectMap.Entry<V>> iterator(final Int2ObjectMap.Entry<V> from) {
			return new EntryIterator(from.getIntKey());
		}

		/**
		 * Returns a type-specific fast list iterator on the elements in this set, starting from the first
		 * element. Please see the class documentation for implementation details.
		 *
		 * @return a type-specific list iterator starting at the first element.
		 */
		@Override
		public ObjectListIterator<Int2ObjectMap.Entry<V>> fastIterator() {
			return new FastEntryIterator();
		}

		/**
		 * Returns a type-specific fast list iterator on the elements in this set, starting from a given
		 * element of the set. Please see the class documentation for implementation details.
		 *
		 * @param from an element to start from.
		 * @return a type-specific list iterator starting at the given element.
		 * @throws IllegalArgumentException if {@code from} does not belong to the set.
		 */
		@Override
		public ObjectListIterator<Int2ObjectMap.Entry<V>> fastIterator(final Int2ObjectMap.Entry<V> from) {
			return new FastEntryIterator(from.getIntKey());
		}

		/** {@inheritDoc} */
		@Override
		public void forEach(final Consumer<? super Int2ObjectMap.Entry<V>> consumer) {
			final long[] link = Int2ObjectLinkedOpenHashMap.this.link;
			for (int i = size, curr, next = first; i-- != 0;) {
				curr = next;
				next = (int)link[curr];
				consumer.accept(new MapEntry(curr));
			}
		}

		/** {@inheritDoc} */
		@Override
		public void fastForEach(final Consumer<? super Int2ObjectMap.Entry<V>> consumer) {
			final MapEntry entry = new MapEntry();
			final long[] link = Int2ObjectLinkedOpenHashMap.this.link;
			for (int i = size, next = first; i-- != 0;) {
				entry.index = next;
				next = (int)link[next];
				consumer.accept(entry);
			}
		}
	}

	@Override
	public FastSortedEntrySet<V> int2ObjectEntrySet() {
		if (entries == null) entries = new MapEntrySet();
		return entries;
	}

	/**
	 * An iterator on keys.
	 *
	 * <p>
	 * We simply override the
	 * {@link java.util.ListIterator#next()}/{@link java.util.ListIterator#previous()} methods (and
	 * possibly their type-specific counterparts) so that they return keys instead of entries.
	 */
	private final class KeyIterator extends MapIterator<java.util.function.IntConsumer> implements IntListIterator {
		public KeyIterator(final int k) {
			super(k);
		}

		@Override
		public int previousInt() {
			return key[previousEntry()];
		}

		public KeyIterator() {
			super();
		}

		// forEachRemaining inherited from MapIterator superclass.
		// Despite the superclass declared with generics, the way Java inherits and generates bridge methods
		// avoids the boxing/unboxing
		@Override
		final void acceptOnIndex(final java.util.function.IntConsumer action, final int index) {
			action.accept(key[index]);
		}

		@Override
		public int nextInt() {
			return key[nextEntry()];
		}
	}

	private final class KeySet extends AbstractIntSortedSet {
		private static final int SPLITERATOR_CHARACTERISTICS = IntSpliterators.SET_SPLITERATOR_CHARACTERISTICS | java.util.Spliterator.ORDERED;

		@Override
		public IntListIterator iterator(final int from) {
			return new KeyIterator(from);
		}

		@Override
		public IntListIterator iterator() {
			return new KeyIterator();
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @see EntrySet#spliterator()
		 */
		@Override
		public IntSpliterator spliterator() {
			return IntSpliterators.asSpliterator(iterator(), it.unimi.dsi.fastutil.Size64.sizeOf(Int2ObjectLinkedOpenHashMap.this), SPLITERATOR_CHARACTERISTICS);
		}

		/** {@inheritDoc} */
		@Override
		public void forEach(final java.util.function.IntConsumer consumer) {
			final long[] link = Int2ObjectLinkedOpenHashMap.this.link;
			final int key[] = Int2ObjectLinkedOpenHashMap.this.key;
			for (int i = size, curr, next = first; i-- != 0;) {
				curr = next;
				next = (int)link[curr];
				consumer.accept(key[curr]);
			}
		}

		@Override
		public int size() {
			return size;
		}

		@Override
		public boolean contains(int k) {
			return containsKey(k);
		}

		@Override
		public boolean remove(int k) {
			final int oldSize = size;
			Int2ObjectLinkedOpenHashMap.this.remove(k);
			return size != oldSize;
		}

		@Override
		public void clear() {
			Int2ObjectLinkedOpenHashMap.this.clear();
		}

		@Override
		public int firstInt() {
			if (size == 0) throw new NoSuchElementException();
			return key[first];
		}

		@Override
		public int lastInt() {
			if (size == 0) throw new NoSuchElementException();
			return key[last];
		}

		@Override
		public IntComparator comparator() {
			return null;
		}

		@Override
		public IntSortedSet tailSet(int from) {
			throw new UnsupportedOperationException();
		}

		@Override
		public IntSortedSet headSet(int to) {
			throw new UnsupportedOperationException();
		}

		@Override
		public IntSortedSet subSet(int from, int to) {
			throw new UnsupportedOperationException();
		}
	}

	@Override
	public IntSortedSet keySet() {
		if (keys == null) keys = new KeySet();
		return keys;
	}

	/**
	 * An iterator on values.
	 *
	 * <p>
	 * We simply override the
	 * {@link java.util.ListIterator#next()}/{@link java.util.ListIterator#previous()} methods (and
	 * possibly their type-specific counterparts) so that they return values instead of entries.
	 */
	private final class ValueIterator extends MapIterator<Consumer<? super V>> implements ObjectListIterator<V> {
		@Override
		public V previous() {
			return value[previousEntry()];
		}

		public ValueIterator() {
			super();
		}

		// forEachRemaining inherited from MapIterator superclass.
		// Despite the superclass declared with generics, the way Java inherits and generates bridge methods
		// avoids the boxing/unboxing
		@Override
		final void acceptOnIndex(final Consumer<? super V> action, final int index) {
			action.accept(value[index]);
		}

		@Override
		public V next() {
			return value[nextEntry()];
		}
	}

	@Override
	public ObjectCollection<V> values() {
		if (values == null) values = new AbstractObjectCollection<V>() {
			private static final int SPLITERATOR_CHARACTERISTICS = ObjectSpliterators.COLLECTION_SPLITERATOR_CHARACTERISTICS | java.util.Spliterator.ORDERED;

			@Override
			public ObjectIterator<V> iterator() {
				return new ValueIterator();
			}

			/**
			 * {@inheritDoc}
			 * 
			 * @see EntrySet#spliterator()
			 */
			@Override
			public ObjectSpliterator<V> spliterator() {
				return ObjectSpliterators.asSpliterator(iterator(), it.unimi.dsi.fastutil.Size64.sizeOf(Int2ObjectLinkedOpenHashMap.this), SPLITERATOR_CHARACTERISTICS);
			}

			/** {@inheritDoc} */
			@Override
			public void forEach(final Consumer<? super V> consumer) {
				final long[] link = Int2ObjectLinkedOpenHashMap.this.link;
				final V value[] = Int2ObjectLinkedOpenHashMap.this.value;
				for (int i = size, curr, next = first; i-- != 0;) {
					curr = next;
					next = (int)link[curr];
					consumer.accept(value[curr]);
				}
			}

			@Override
			public int size() {
				return size;
			}

			@Override
			public boolean contains(Object v) {
				return containsValue(v);
			}

			@Override
			public void clear() {
				Int2ObjectLinkedOpenHashMap.this.clear();
			}
		};
		return values;
	}

	/**
	 * Rehashes the map, making the table as small as possible.
	 *
	 * <p>
	 * This method rehashes the table to the smallest size satisfying the load factor. It can be used
	 * when the set will not be changed anymore, so to optimize access speed and size.
	 *
	 * <p>
	 * If the table size is already the minimum possible, this method does nothing.
	 *
	 * @return true if there was enough memory to trim the map.
	 * @see #trim(int)
	 */
	public boolean trim() {
		return trim(size);
	}

	/**
	 * Rehashes this map if the table is too large.
	 *
	 * <p>
	 * Let <var>N</var> be the smallest table size that can hold <code>max(n,{@link #size()})</code>
	 * entries, still satisfying the load factor. If the current table size is smaller than or equal to
	 * <var>N</var>, this method does nothing. Otherwise, it rehashes this map in a table of size
	 * <var>N</var>.
	 *
	 * <p>
	 * This method is useful when reusing maps. {@linkplain #clear() Clearing a map} leaves the table
	 * size untouched. If you are reusing a map many times, you can call this method with a typical size
	 * to avoid keeping around a very large table just because of a few large transient maps.
	 *
	 * @param n the threshold for the trimming.
	 * @return true if there was enough memory to trim the map.
	 * @see #trim()
	 */
	public boolean trim(final int n) {
		final int l = HashCommon.nextPowerOfTwo((int)Math.ceil(n / f));
		if (l >= this.n || size > maxFill(l, f)) return true;
		try {
			rehash(l);
		} catch (OutOfMemoryError cantDoIt) {
			return false;
		}
		return true;
	}

	/**
	 * Rehashes the map.
	 *
	 * <p>
	 * This method implements the basic rehashing strategy, and may be overridden by subclasses
	 * implementing different rehashing strategies (e.g., disk-based rehashing). However, you should not
	 * override this method unless you understand the internal workings of this class.
	 *
	 * @param newN the new size
	 */
	@SuppressWarnings("unchecked")
	protected void rehash(final int newN) {
		final int key[] = this.key;
		final V value[] = this.value;
		final int mask = newN - 1; // Note that this is used by the hashing macro
		final int newKey[] = new int[newN + 1];
		final V newValue[] = (V[])new Object[newN + 1];
		int i = first, prev = -1, newPrev = -1, t, pos;
		final long link[] = this.link;
		final long newLink[] = new long[newN + 1];
		first = -1;
		for (int j = size; j-- != 0;) {
			if (((key[i]) == (0))) pos = newN;
			else {
				pos = (it.unimi.dsi.fastutil.HashCommon.mix((key[i]))) & mask;
				while (!((newKey[pos]) == (0))) pos = (pos + 1) & mask;
			}
			newKey[pos] = key[i];
			newValue[pos] = value[i];
			if (prev != -1) {
				newLink[newPrev] ^= ((newLink[newPrev] ^ (pos & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
				newLink[pos] ^= ((newLink[pos] ^ ((newPrev & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
				newPrev = pos;
			} else {
				newPrev = first = pos;
				// Special case of SET(newLink[pos], -1, -1);
				newLink[pos] = -1L;
			}
			t = i;
			i = (int)link[i];
			prev = t;
		}
		this.link = newLink;
		this.last = newPrev;
		if (newPrev != -1)
			// Special case of SET_NEXT(newLink[newPrev], -1);
			newLink[newPrev] |= -1 & 0xFFFFFFFFL;
		n = newN;
		this.mask = mask;
		maxFill = maxFill(n, f);
		this.key = newKey;
		this.value = newValue;
	}

	/**
	 * Returns a deep copy of this map.
	 *
	 * <p>
	 * This method performs a deep copy of this hash map; the data stored in the map, however, is not
	 * cloned. Note that this makes a difference only for object keys.
	 *
	 * @return a deep copy of this map.
	 */
	@Override
	@SuppressWarnings("unchecked")
	public Int2ObjectLinkedOpenHashMap<V> clone() {
		Int2ObjectLinkedOpenHashMap<V> c;
		try {
			c = (Int2ObjectLinkedOpenHashMap<V>)super.clone();
		} catch (CloneNotSupportedException cantHappen) {
			throw new InternalError();
		}
		c.keys = null;
		c.values = null;
		c.entries = null;
		c.containsNullKey = containsNullKey;
		c.key = key.clone();
		c.value = value.clone();
		c.link = link.clone();
		return c;
	}

	/**
	 * Returns a hash code for this map.
	 *
	 * This method overrides the generic method provided by the superclass. Since {@code equals()} is
	 * not overriden, it is important that the value returned by this method is the same value as the
	 * one returned by the overriden method.
	 *
	 * @return a hash code for this map.
	 */
	@Override
	public int hashCode() {
		int h = 0;
		final int key[] = this.key;
		final V value[] = this.value;
		for (int j = realSize(), i = 0, t = 0; j-- != 0;) {
			while (((key[i]) == (0))) i++;
			t = (key[i]);
			if (this != value[i]) t ^= ((value[i]) == null ? 0 : (value[i]).hashCode());
			h += t;
			i++;
		}
		// Zero / null keys have hash zero.
		if (containsNullKey) h += ((value[n]) == null ? 0 : (value[n]).hashCode());
		return h;
	}

	private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException {
		final int key[] = this.key;
		final V value[] = this.value;
		final EntryIterator i = new EntryIterator();
		s.defaultWriteObject();
		for (int j = size, e; j-- != 0;) {
			e = i.nextEntry();
			s.writeInt(key[e]);
			s.writeObject(value[e]);
		}
	}

	@SuppressWarnings("unchecked")
	private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException {
		s.defaultReadObject();
		n = arraySize(size, f);
		maxFill = maxFill(n, f);
		mask = n - 1;
		final int key[] = this.key = new int[n + 1];
		final V value[] = this.value = (V[])new Object[n + 1];
		final long link[] = this.link = new long[n + 1];
		int prev = -1;
		first = last = -1;
		int k;
		V v;
		for (int i = size, pos; i-- != 0;) {
			k = s.readInt();
			v = (V)s.readObject();
			if (((k) == (0))) {
				pos = n;
				containsNullKey = true;
			} else {
				pos = (it.unimi.dsi.fastutil.HashCommon.mix((k))) & mask;
				while (!((key[pos]) == (0))) pos = (pos + 1) & mask;
			}
			key[pos] = k;
			value[pos] = v;
			if (first != -1) {
				link[prev] ^= ((link[prev] ^ (pos & 0xFFFFFFFFL)) & 0xFFFFFFFFL);
				link[pos] ^= ((link[pos] ^ ((prev & 0xFFFFFFFFL) << 32)) & 0xFFFFFFFF00000000L);
				prev = pos;
			} else {
				prev = first = pos;
				// Special case of SET_PREV(newLink[pos], -1);
				link[pos] |= (-1L & 0xFFFFFFFFL) << 32;
			}
		}
		last = prev;
		if (prev != -1)
			// Special case of SET_NEXT(link[prev], -1);
			link[prev] |= -1 & 0xFFFFFFFFL;
		if (ASSERTS) checkTable();
	}

	private void checkTable() {
	}
}
