/*
	* Copyright (C) 2007-2024 Sebastiano Vigna
	*
	* Licensed under the Apache License, Version 2.0 (the "License");
	* you may not use this file except in compliance with the License.
	* You may obtain a copy of the License at
	*
	*     http://www.apache.org/licenses/LICENSE-2.0
	*
	* Unless required by applicable law or agreed to in writing, software
	* distributed under the License is distributed on an "AS IS" BASIS,
	* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	* See the License for the specific language governing permissions and
	* limitations under the License.
	*/
package it.unimi.dsi.fastutil.floats;

import java.util.Map;
import java.util.NoSuchElementException;
import java.util.function.Consumer;
import it.unimi.dsi.fastutil.objects.AbstractObjectSet;
import it.unimi.dsi.fastutil.objects.ObjectIterator;
import it.unimi.dsi.fastutil.objects.ObjectSpliterator;
import it.unimi.dsi.fastutil.objects.ObjectSpliterators;
import it.unimi.dsi.fastutil.doubles.DoubleCollection;
import it.unimi.dsi.fastutil.doubles.AbstractDoubleCollection;
import it.unimi.dsi.fastutil.doubles.DoubleArrays;

/**
 * A simple, brute-force implementation of a map based on two parallel backing arrays.
 *
 * <p>
 * The main purpose of this implementation is that of wrapping cleanly the brute-force approach to
 * the storage of a very small number of pairs: just put them into two parallel arrays and scan
 * linearly to find an item.
 */
public class Float2DoubleArrayMap extends AbstractFloat2DoubleMap implements java.io.Serializable, Cloneable {
	private static final long serialVersionUID = 1L;
	/** The keys (valid up to {@link #size}, excluded). */
	protected transient float[] key;
	/** The values (parallel to {@link #key}). */
	protected transient double[] value;
	/** The number of valid entries in {@link #key} and {@link #value}. */
	protected int size;
	/** Cached set of entries. */
	protected transient FastEntrySet entries;
	/** Cached set of keys. */
	protected transient FloatSet keys;
	/** Cached collection of values. */
	protected transient DoubleCollection values;

	/**
	 * Creates a new empty array map with given key and value backing arrays. The resulting map will
	 * have as many entries as the given arrays.
	 *
	 * <p>
	 * It is responsibility of the caller that the elements of {@code key} are distinct.
	 *
	 * @param key the key array.
	 * @param value the value array (it <em>must</em> have the same length as {@code key}).
	 */
	public Float2DoubleArrayMap(final float[] key, final double[] value) {
		this.key = key;
		this.value = value;
		size = key.length;
		if (key.length != value.length) throw new IllegalArgumentException("Keys and values have different lengths (" + key.length + ", " + value.length + ")");
	}

	/**
	 * Creates a new empty array map.
	 */
	public Float2DoubleArrayMap() {
		this.key = FloatArrays.EMPTY_ARRAY;
		this.value = DoubleArrays.EMPTY_ARRAY;
	}

	/**
	 * Creates a new empty array map of given capacity.
	 *
	 * @param capacity the initial capacity.
	 */
	public Float2DoubleArrayMap(final int capacity) {
		this.key = new float[capacity];
		this.value = new double[capacity];
	}

	/**
	 * Creates a new empty array map copying the entries of a given map.
	 *
	 * @param m a map.
	 */
	public Float2DoubleArrayMap(final Float2DoubleMap m) {
		this(m.size());
		int i = 0;
		for (Float2DoubleMap.Entry e : m.float2DoubleEntrySet()) {
			key[i] = e.getFloatKey();
			value[i] = e.getDoubleValue();
			i++;
		}
		size = i;
	}

	/**
	 * Creates a new empty array map copying the entries of a given map.
	 *
	 * @param m a map.
	 */
	public Float2DoubleArrayMap(final Map<? extends Float, ? extends Double> m) {
		this(m.size());
		int i = 0;
		for (Map.Entry<? extends Float, ? extends Double> e : m.entrySet()) {
			key[i] = (e.getKey()).floatValue();
			value[i] = (e.getValue()).doubleValue();
			i++;
		}
		size = i;
	}

	/**
	 * Creates a new array map with given key and value backing arrays, using the given number of
	 * elements.
	 *
	 * <p>
	 * It is responsibility of the caller that the first {@code size} elements of {@code key} are
	 * distinct.
	 *
	 * @param key the key array.
	 * @param value the value array (it <em>must</em> have the same length as {@code key}).
	 * @param size the number of valid elements in {@code key} and {@code value}.
	 */
	public Float2DoubleArrayMap(final float[] key, final double[] value, final int size) {
		this.key = key;
		this.value = value;
		this.size = size;
		if (key.length != value.length) throw new IllegalArgumentException("Keys and values have different lengths (" + key.length + ", " + value.length + ")");
		if (size > key.length) throw new IllegalArgumentException("The provided size (" + size + ") is larger than or equal to the backing-arrays size (" + key.length + ")");
	}

	/**
	 * The entry class for an array map does not record key and value, but rather the position in the
	 * array of the corresponding entry. This is necessary so that calls to
	 * {@link java.util.Map.Entry#setValue(Object)} are reflected in the map
	 */
	private final class MapEntry implements Float2DoubleMap.Entry, Map.Entry<Float, Double>, FloatDoublePair {
		// The array index this entry refers to, or -1 if this entry has been deleted.
		int index;

		MapEntry() {
		}

		MapEntry(final int index) {
			this.index = index;
		}

		@Override

		public float getFloatKey() {
			return key[index];
		}

		@Override

		public float leftFloat() {
			return key[index];
		}

		@Override

		public double getDoubleValue() {
			return value[index];
		}

		@Override

		public double rightDouble() {
			return value[index];
		}

		@Override

		public double setValue(final double v) {
			final double oldValue = value[index];
			value[index] = v;
			return oldValue;
		}

		@Override
		public FloatDoublePair right(final double v) {
			value[index] = v;
			return this;
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method instead.
		 */
		@Deprecated
		@Override
		public Float getKey() {
			return Float.valueOf(key[index]);
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method instead.
		 */
		@Deprecated
		@Override
		public Double getValue() {
			return Double.valueOf(value[index]);
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method instead.
		 */
		@Deprecated
		@Override
		public Double setValue(final Double v) {
			return Double.valueOf(setValue((v).doubleValue()));
		}

		@SuppressWarnings("unchecked")
		@Override
		public boolean equals(final Object o) {
			if (!(o instanceof Map.Entry)) return false;
			Map.Entry<Float, Double> e = (Map.Entry<Float, Double>)o;
			return (Float.floatToIntBits(key[index]) == Float.floatToIntBits((e.getKey()).floatValue())) && (Double.doubleToLongBits(value[index]) == Double.doubleToLongBits((e.getValue()).doubleValue()));
		}

		@Override
		public int hashCode() {
			return it.unimi.dsi.fastutil.HashCommon.float2int(key[index]) ^ it.unimi.dsi.fastutil.HashCommon.double2int(value[index]);
		}

		@Override
		public String toString() {
			return key[index] + "=>" + value[index];
		}
	}

	private final class EntrySet extends AbstractObjectSet<Float2DoubleMap.Entry> implements FastEntrySet {
		@Override
		public ObjectIterator<Float2DoubleMap.Entry> iterator() {
			return new ObjectIterator<Float2DoubleMap.Entry>() {
				private MapEntry entry;
				int curr = -1, next = 0;

				@Override
				public boolean hasNext() {
					return next < size;
				}

				@Override

				public Entry next() {
					if (!hasNext()) throw new NoSuchElementException();
					return entry = new MapEntry(curr = next++);
				}

				@Override
				public void remove() {
					if (curr == -1) throw new IllegalStateException();
					curr = -1;
					final int tail = size-- - next--;
					System.arraycopy(key, next + 1, key, next, tail);
					System.arraycopy(value, next + 1, value, next, tail);
					entry.index = -1;
				}

				@Override
				public int skip(int n) {
					if (n < 0) throw new IllegalArgumentException("Argument must be nonnegative: " + n);
					n = Math.min(n, size - next);
					next += n;
					if (n != 0) curr = next - 1;
					return n;
				}

				@Override

				public void forEachRemaining(final Consumer<? super Float2DoubleMap.Entry> action) {
					final int max = size;
					while (next < max) {
						entry = new MapEntry(curr = next++);
						action.accept(entry);
					}
				}
			};
		}

		@Override
		public ObjectIterator<Float2DoubleMap.Entry> fastIterator() {
			return new ObjectIterator<Float2DoubleMap.Entry>() {
				private MapEntry entry = new MapEntry();
				int next = 0, curr = -1;

				@Override
				public boolean hasNext() {
					return next < size;
				}

				@Override

				public Entry next() {
					if (!hasNext()) throw new NoSuchElementException();
					entry.index = curr = next++;
					return entry;
				}

				@Override
				public void remove() {
					if (curr == -1) throw new IllegalStateException();
					curr = -1;
					final int tail = size-- - next--;
					System.arraycopy(key, next + 1, key, next, tail);
					System.arraycopy(value, next + 1, value, next, tail);
					entry.index = -1;
				}

				@Override
				public int skip(int n) {
					if (n < 0) throw new IllegalArgumentException("Argument must be nonnegative: " + n);
					n = Math.min(n, size - next);
					next += n;
					if (n != 0) curr = next - 1;
					return n;
				}

				@Override

				public void forEachRemaining(final Consumer<? super Float2DoubleMap.Entry> action) {
					final int max = size;
					while (next < max) {
						entry.index = curr = next++;
						action.accept(entry);
					}
				}
			};
		}

		// We already have to create an Entry object for each iteration, so the overhead from having
		// skeletal implementations isn't significant.
		final class EntrySetSpliterator extends ObjectSpliterators.EarlyBindingSizeIndexBasedSpliterator<Float2DoubleMap.Entry> implements ObjectSpliterator<Float2DoubleMap.Entry> {
			EntrySetSpliterator(int pos, int maxPos) {
				super(pos, maxPos);
			}

			@Override
			public int characteristics() {
				return ObjectSpliterators.SET_SPLITERATOR_CHARACTERISTICS | java.util.Spliterator.SUBSIZED | java.util.Spliterator.ORDERED;
			}

			@Override

			protected final Float2DoubleMap.Entry get(int location) {
				return new MapEntry(location);
			}

			@Override
			protected final EntrySetSpliterator makeForSplit(int pos, int maxPos) {
				return new EntrySetSpliterator(pos, maxPos);
			}
		}

		@Override
		public ObjectSpliterator<Float2DoubleMap.Entry> spliterator() {
			return new EntrySetSpliterator(0, size);
		}

		/** {@inheritDoc} */
		@Override

		public void forEach(final Consumer<? super Float2DoubleMap.Entry> action) {
			for (int i = 0, max = size; i < max; ++i) {
				action.accept(new MapEntry(i));
			}
		}

		/** {@inheritDoc} */
		@Override

		public void fastForEach(final Consumer<? super Float2DoubleMap.Entry> action) {
			final MapEntry entry = new MapEntry();
			for (int i = 0, max = size; i < max; ++i) {
				entry.index = i;
				action.accept(entry);
			}
		}

		@Override
		public int size() {
			return size;
		}

		@Override

		public boolean contains(Object o) {
			if (!(o instanceof Map.Entry)) return false;
			final Map.Entry<?, ?> e = (Map.Entry<?, ?>)o;
			if (e.getKey() == null || !(e.getKey() instanceof Float)) return false;
			if (e.getValue() == null || !(e.getValue() instanceof Double)) return false;
			final float k = ((Float)(e.getKey())).floatValue();
			return Float2DoubleArrayMap.this.containsKey(k) && (Double.doubleToLongBits(Float2DoubleArrayMap.this.get(k)) == Double.doubleToLongBits(((Double)(e.getValue())).doubleValue()));
		}

		@Override

		public boolean remove(final Object o) {
			if (!(o instanceof Map.Entry)) return false;
			final Map.Entry<?, ?> e = (Map.Entry<?, ?>)o;
			if (e.getKey() == null || !(e.getKey() instanceof Float)) return false;
			if (e.getValue() == null || !(e.getValue() instanceof Double)) return false;
			final float k = ((Float)(e.getKey())).floatValue();
			final double v = ((Double)(e.getValue())).doubleValue();
			final int oldPos = Float2DoubleArrayMap.this.findKey(k);
			if (oldPos == -1 || !(Double.doubleToLongBits(v) == Double.doubleToLongBits(Float2DoubleArrayMap.this.value[oldPos]))) return false;
			final int tail = size - oldPos - 1;
			System.arraycopy(Float2DoubleArrayMap.this.key, oldPos + 1, Float2DoubleArrayMap.this.key, oldPos, tail);
			System.arraycopy(Float2DoubleArrayMap.this.value, oldPos + 1, Float2DoubleArrayMap.this.value, oldPos, tail);
			Float2DoubleArrayMap.this.size--;
			return true;
		}
	}

	@Override
	public FastEntrySet float2DoubleEntrySet() {
		if (entries == null) entries = new EntrySet();
		return entries;
	}

	private int findKey(final float k) {
		final float[] key = this.key;
		for (int i = size; i-- != 0;) if ((Float.floatToIntBits(key[i]) == Float.floatToIntBits(k))) return i;
		return -1;
	}

	@Override

	public double get(final float k) {
		final float[] key = this.key;
		for (int i = size; i-- != 0;) if ((Float.floatToIntBits(key[i]) == Float.floatToIntBits(k))) return value[i];
		return defRetValue;
	}

	@Override
	public int size() {
		return size;
	}

	@Override
	public void clear() {
		size = 0;
	}

	@Override
	public boolean containsKey(final float k) {
		return findKey(k) != -1;
	}

	@Override
	public boolean containsValue(double v) {
		final double[] value = this.value;
		for (int i = size; i-- != 0;) if ((Double.doubleToLongBits(value[i]) == Double.doubleToLongBits(v))) return true;
		return false;
	}

	@Override
	public boolean isEmpty() {
		return size == 0;
	}

	@Override

	public double put(float k, double v) {
		final int oldKey = findKey(k);
		if (oldKey != -1) {
			final double oldValue = value[oldKey];
			value[oldKey] = v;
			return oldValue;
		}
		if (size == key.length) {
			final float[] newKey = new float[size == 0 ? 2 : size * 2];
			final double[] newValue = new double[size == 0 ? 2 : size * 2];
			for (int i = size; i-- != 0;) {
				newKey[i] = key[i];
				newValue[i] = value[i];
			}
			key = newKey;
			value = newValue;
		}
		key[size] = k;
		value[size] = v;
		size++;
		return defRetValue;
	}

	@Override

	public double remove(final float k) {
		final int oldPos = findKey(k);
		if (oldPos == -1) return defRetValue;
		final double oldValue = value[oldPos];
		final int tail = size - oldPos - 1;
		System.arraycopy(key, oldPos + 1, key, oldPos, tail);
		System.arraycopy(value, oldPos + 1, value, oldPos, tail);
		size--;
		return oldValue;
	}

	private final class KeySet extends AbstractFloatSet {
		@Override
		public boolean contains(final float k) {
			return findKey(k) != -1;
		}

		@Override
		public boolean remove(final float k) {
			final int oldPos = findKey(k);
			if (oldPos == -1) return false;
			final int tail = size - oldPos - 1;
			System.arraycopy(key, oldPos + 1, key, oldPos, tail);
			System.arraycopy(value, oldPos + 1, value, oldPos, tail);
			size--;
			return true;
		}

		@Override
		public FloatIterator iterator() {
			return new FloatIterator() {
				int pos = 0;

				@Override
				public boolean hasNext() {
					return pos < size;
				}

				@Override

				public float nextFloat() {
					if (!hasNext()) throw new NoSuchElementException();
					return key[pos++];
				}

				@Override
				public void remove() {
					if (pos == 0) throw new IllegalStateException();
					final int tail = size - pos;
					System.arraycopy(key, pos, key, pos - 1, tail);
					System.arraycopy(value, pos, value, pos - 1, tail);
					size--;
					pos--;
				}

				@Override

				public void forEachRemaining(final FloatConsumer action) {
					final float[] key = Float2DoubleArrayMap.this.key;
					final int max = size;
					while (pos < max) {
						action.accept(key[pos++]);
					}
				}
				// TODO either override skip or extend from AbstractIndexBasedIterator.
			};
		}

		final class KeySetSpliterator extends FloatSpliterators.EarlyBindingSizeIndexBasedSpliterator implements FloatSpliterator {
			KeySetSpliterator(int pos, int maxPos) {
				super(pos, maxPos);
			}

			@Override
			public int characteristics() {
				return FloatSpliterators.SET_SPLITERATOR_CHARACTERISTICS | java.util.Spliterator.SUBSIZED | java.util.Spliterator.ORDERED;
			}

			@Override

			protected final float get(int location) {
				return key[location];
			}

			@Override
			protected final KeySetSpliterator makeForSplit(int pos, int maxPos) {
				return new KeySetSpliterator(pos, maxPos);
			}

			@Override

			public void forEachRemaining(final FloatConsumer action) {
				final float[] key = Float2DoubleArrayMap.this.key;
				final int max = size;
				while (pos < max) {
					action.accept(key[pos++]);
				}
			}
		}

		@Override
		public FloatSpliterator spliterator() {
			return new KeySetSpliterator(0, size);
		}

		@Override

		public void forEach(FloatConsumer action) {
			final float[] key = Float2DoubleArrayMap.this.key;
			for (int i = 0, max = size; i < max; ++i) {
				action.accept(key[i]);
			}
		}

		@Override
		public int size() {
			return size;
		}

		@Override
		public void clear() {
			Float2DoubleArrayMap.this.clear();
		}
	}

	@Override
	public FloatSet keySet() {
		if (keys == null) keys = new KeySet();
		return keys;
	}

	private final class ValuesCollection extends AbstractDoubleCollection {
		@Override
		public boolean contains(final double v) {
			return containsValue(v);
		}

		@Override
		public it.unimi.dsi.fastutil.doubles.DoubleIterator iterator() {
			return new it.unimi.dsi.fastutil.doubles.DoubleIterator() {
				int pos = 0;

				@Override
				public boolean hasNext() {
					return pos < size;
				}

				@Override

				public double nextDouble() {
					if (!hasNext()) throw new NoSuchElementException();
					return value[pos++];
				}

				@Override
				public void remove() {
					if (pos == 0) throw new IllegalStateException();
					final int tail = size - pos;
					System.arraycopy(key, pos, key, pos - 1, tail);
					System.arraycopy(value, pos, value, pos - 1, tail);
					size--;
					pos--;
				}

				@Override

				public void forEachRemaining(final java.util.function.DoubleConsumer action) {
					final double[] value = Float2DoubleArrayMap.this.value;
					final int max = size;
					while (pos < max) {
						action.accept(value[pos++]);
					}
				}
				// TODO either override skip or extend from AbstractIndexBasedIterator.
			};
		}

		final class ValuesSpliterator extends it.unimi.dsi.fastutil.doubles.DoubleSpliterators.EarlyBindingSizeIndexBasedSpliterator implements it.unimi.dsi.fastutil.doubles.DoubleSpliterator {
			ValuesSpliterator(int pos, int maxPos) {
				super(pos, maxPos);
			}

			@Override
			public int characteristics() {
				return it.unimi.dsi.fastutil.doubles.DoubleSpliterators.COLLECTION_SPLITERATOR_CHARACTERISTICS | java.util.Spliterator.SUBSIZED | java.util.Spliterator.ORDERED;
			}

			@Override

			protected final double get(int location) {
				return value[location];
			}

			@Override
			protected final ValuesSpliterator makeForSplit(int pos, int maxPos) {
				return new ValuesSpliterator(pos, maxPos);
			}

			@Override

			public void forEachRemaining(final java.util.function.DoubleConsumer action) {
				final double[] value = Float2DoubleArrayMap.this.value;
				final int max = size;
				while (pos < max) {
					action.accept(value[pos++]);
				}
			}
		}

		@Override
		public it.unimi.dsi.fastutil.doubles.DoubleSpliterator spliterator() {
			return new ValuesSpliterator(0, size);
		}

		@Override

		public void forEach(java.util.function.DoubleConsumer action) {
			final double[] value = Float2DoubleArrayMap.this.value;
			for (int i = 0, max = size; i < max; ++i) {
				action.accept(value[i]);
			}
		}

		@Override
		public int size() {
			return size;
		}

		@Override
		public void clear() {
			Float2DoubleArrayMap.this.clear();
		}
	}

	@Override
	public DoubleCollection values() {
		if (values == null) values = new ValuesCollection();
		return values;
	}

	/**
	 * Returns a deep copy of this map.
	 *
	 * <p>
	 * This method performs a deep copy of this hash map; the data stored in the map, however, is not
	 * cloned. Note that this makes a difference only for object keys.
	 *
	 * @return a deep copy of this map.
	 */
	@Override

	public Float2DoubleArrayMap clone() {
		Float2DoubleArrayMap c;
		try {
			c = (Float2DoubleArrayMap)super.clone();
		} catch (CloneNotSupportedException cantHappen) {
			throw new InternalError();
		}
		c.key = key.clone();
		c.value = value.clone();
		c.entries = null;
		c.keys = null;
		c.values = null;
		return c;
	}

	private void writeObject(java.io.ObjectOutputStream s) throws java.io.IOException {
		s.defaultWriteObject();
		final float[] key = this.key;
		final double[] value = this.value;
		for (int i = 0, max = size; i < max; i++) {
			s.writeFloat(key[i]);
			s.writeDouble(value[i]);
		}
	}

	private void readObject(java.io.ObjectInputStream s) throws java.io.IOException, ClassNotFoundException {
		s.defaultReadObject();
		final float[] key = this.key = new float[size];
		final double[] value = this.value = new double[size];
		for (int i = 0; i < size; i++) {
			key[i] = s.readFloat();
			value[i] = s.readDouble();
		}
	}
}
