
/*
	* Copyright (C) 2002-2017 Sebastiano Vigna
	*
	* Licensed under the Apache License, Version 2.0 (the "License");
	* you may not use this file except in compliance with the License.
	* You may obtain a copy of the License at
	*
	*     http://www.apache.org/licenses/LICENSE-2.0
	*
	* Unless required by applicable law or agreed to in writing, software
	* distributed under the License is distributed on an "AS IS" BASIS,
	* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	* See the License for the specific language governing permissions and
	* limitations under the License.
	*/

package it.unimi.dsi.fastutil.objects;

import it.unimi.dsi.fastutil.chars.CharCollection;
import it.unimi.dsi.fastutil.chars.AbstractCharCollection;
import it.unimi.dsi.fastutil.chars.CharIterator;

import java.util.Iterator;
import java.util.Map;

/**
 * An abstract class providing basic methods for maps implementing a
 * type-specific interface.
 *
 * <p>
 * Optional operations just throw an {@link UnsupportedOperationException}.
 * Generic versions of accessors delegate to the corresponding type-specific
 * counterparts following the interface rules (they take care of returning
 * {@code null} on a missing key).
 *
 * <p>
 * As a further help, this class provides a {@link BasicEntry BasicEntry} inner
 * class that implements a type-specific version of {@link java.util.Map.Entry};
 * it is particularly useful for those classes that do not implement their own
 * entries (e.g., most immutable maps).
 */

public abstract class AbstractReference2CharMap<K> extends AbstractReference2CharFunction<K>
		implements
			Reference2CharMap<K>,
			java.io.Serializable {

	private static final long serialVersionUID = -4940583368468432370L;

	protected AbstractReference2CharMap() {
	}

	@Override
	public boolean containsValue(final char v) {
		return values().contains(v);
	}

	@Override
	public boolean containsKey(final Object k) {
		final ObjectIterator<Reference2CharMap.Entry<K>> i = reference2CharEntrySet().iterator();
		while (i.hasNext())
			if (i.next().getKey() == k)
				return true;

		return false;
	}

	@Override
	public boolean isEmpty() {
		return size() == 0;
	}

	/**
	 * This class provides a basic but complete type-specific entry class for
	 * all those maps implementations that do not have entries on their own
	 * (e.g., most immutable maps).
	 *
	 * <p>
	 * This class does not implement {@link java.util.Map.Entry#setValue(Object)
	 * setValue()}, as the modification would not be reflected in the base map.
	 */

	public static class BasicEntry<K> implements Reference2CharMap.Entry<K> {
		protected K key;
		protected char value;

		public BasicEntry() {
		}

		public BasicEntry(final K key, final Character value) {
			this.key = (key);
			this.value = (value).charValue();
		}

		public BasicEntry(final K key, final char value) {
			this.key = key;
			this.value = value;
		}
		@Override
		public K getKey() {
			return (key);
		}
		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method
		 *             instead.
		 */
		@Deprecated

		@Override
		public Character getValue() {
			return Character.valueOf(value);
		}

		@Override
		public char getCharValue() {
			return value;
		}

		@Override
		public char setValue(final char value) {
			throw new UnsupportedOperationException();
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method
		 *             instead.
		 */
		@Deprecated
		@Override
		public Character setValue(final Character value) {
			return Character.valueOf(setValue(value.charValue()));
		}

		@Override
		public boolean equals(final Object o) {
			if (!(o instanceof Map.Entry))
				return false;
			final Map.Entry<?, ?> e = (Map.Entry<?, ?>) o;

			if (e.getValue() == null || !(e.getValue() instanceof Character))
				return false;

			return ((key) == ((e.getKey()))) && ((value) == (((Character) (e.getValue())).charValue()));
		}

		@Override
		public int hashCode() {
			return (System.identityHashCode(key)) ^ (value);
		}

		@Override
		public String toString() {
			return key + "->" + value;
		}
	}

	/**
	 * Returns a type-specific-set view of the keys of this map.
	 *
	 * <p>
	 * The view is backed by the set returned by {@link Map#entrySet()}. Note
	 * that <em>no attempt is made at caching the result of this method</em>, as
	 * this would require adding some attributes that lightweight
	 * implementations would not need. Subclasses may easily override this
	 * policy by calling this method and caching the result, but implementors
	 * are encouraged to write more efficient ad-hoc implementations.
	 *
	 * @return a set view of the keys of this map; it may be safely cast to a
	 *         type-specific interface.
	 */
	@Override
	public ReferenceSet<K> keySet() {
		return new AbstractReferenceSet<K>() {
			@Override
			public boolean contains(final Object k) {
				return containsKey(k);
			}
			@Override
			public int size() {
				return AbstractReference2CharMap.this.size();
			}
			@Override
			public void clear() {
				AbstractReference2CharMap.this.clear();
			}
			@Override
			public ObjectIterator<K> iterator() {
				return new ObjectIterator<K>() {
					private final ObjectIterator<Reference2CharMap.Entry<K>> i = Reference2CharMaps
							.fastIterator(AbstractReference2CharMap.this);
					@Override
					public K next() {
						return i.next().getKey();
					};
					@Override
					public boolean hasNext() {
						return i.hasNext();
					}
					@Override
					public void remove() {
						i.remove();
					}
				};
			}
		};
	}

	/**
	 * Returns a type-specific-set view of the values of this map.
	 *
	 * <p>
	 * The view is backed by the set returned by {@link Map#entrySet()}. Note
	 * that <em>no attempt is made at caching the result of this method</em>, as
	 * this would require adding some attributes that lightweight
	 * implementations would not need. Subclasses may easily override this
	 * policy by calling this method and caching the result, but implementors
	 * are encouraged to write more efficient ad-hoc implementations.
	 *
	 * @return a set view of the values of this map; it may be safely cast to a
	 *         type-specific interface.
	 */
	@Override
	public CharCollection values() {
		return new AbstractCharCollection() {
			@Override
			public boolean contains(final char k) {
				return containsValue(k);
			}
			@Override
			public int size() {
				return AbstractReference2CharMap.this.size();
			}
			@Override
			public void clear() {
				AbstractReference2CharMap.this.clear();
			}

			@Override
			public CharIterator iterator() {
				return new CharIterator() {
					private final ObjectIterator<Reference2CharMap.Entry<K>> i = Reference2CharMaps
							.fastIterator(AbstractReference2CharMap.this);
					@Override
					public char nextChar() {
						return i.next().getCharValue();
					}
					@Override
					public boolean hasNext() {
						return i.hasNext();
					}
				};
			}
		};
	}

	/** {@inheritDoc} */
	@SuppressWarnings({"unchecked", "deprecation"})
	@Override
	public void putAll(final Map<? extends K, ? extends Character> m) {
		if (m instanceof Reference2CharMap) {
			ObjectIterator<Reference2CharMap.Entry<K>> i = Reference2CharMaps.fastIterator((Reference2CharMap<K>) m);

			while (i.hasNext()) {
				final Reference2CharMap.Entry<? extends K> e = i.next();
				put(e.getKey(), e.getCharValue());
			}
		} else {
			int n = m.size();
			final Iterator<? extends Map.Entry<? extends K, ? extends Character>> i = m.entrySet().iterator();
			Map.Entry<? extends K, ? extends Character> e;
			while (n-- != 0) {
				e = i.next();
				put(e.getKey(), e.getValue());
			}
		}
	}

	/**
	 * Returns a hash code for this map.
	 *
	 * The hash code of a map is computed by summing the hash codes of its
	 * entries.
	 *
	 * @return a hash code for this map.
	 */
	@Override
	public int hashCode() {
		int h = 0, n = size();
		final ObjectIterator<Reference2CharMap.Entry<K>> i = Reference2CharMaps.fastIterator(this);

		while (n-- != 0)
			h += i.next().hashCode();
		return h;
	}

	@Override
	public boolean equals(Object o) {
		if (o == this)
			return true;
		if (!(o instanceof Map))
			return false;

		final Map<?, ?> m = (Map<?, ?>) o;
		if (m.size() != size())
			return false;
		return reference2CharEntrySet().containsAll(m.entrySet());
	}

	@Override
	public String toString() {
		final StringBuilder s = new StringBuilder();
		final ObjectIterator<Reference2CharMap.Entry<K>> i = Reference2CharMaps.fastIterator(this);
		int n = size();
		Reference2CharMap.Entry<K> e;
		boolean first = true;

		s.append("{");

		while (n-- != 0) {
			if (first)
				first = false;
			else
				s.append(", ");

			e = i.next();

			if (this == e.getKey())
				s.append("(this map)");
			else

				s.append(String.valueOf(e.getKey()));
			s.append("=>");

			s.append(String.valueOf(e.getCharValue()));
		}

		s.append("}");
		return s.toString();
	}
}
