
/*
	* Copyright (C) 2002-2017 Sebastiano Vigna
	*
	* Licensed under the Apache License, Version 2.0 (the "License");
	* you may not use this file except in compliance with the License.
	* You may obtain a copy of the License at
	*
	*     http://www.apache.org/licenses/LICENSE-2.0
	*
	* Unless required by applicable law or agreed to in writing, software
	* distributed under the License is distributed on an "AS IS" BASIS,
	* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	* See the License for the specific language governing permissions and
	* limitations under the License.
	*/

package it.unimi.dsi.fastutil.longs;

import it.unimi.dsi.fastutil.longs.LongCollection;
import it.unimi.dsi.fastutil.longs.AbstractLongCollection;
import it.unimi.dsi.fastutil.longs.LongIterator;

import it.unimi.dsi.fastutil.objects.ObjectIterator;

import java.util.Iterator;
import java.util.Map;

/**
 * An abstract class providing basic methods for maps implementing a
 * type-specific interface.
 *
 * <p>
 * Optional operations just throw an {@link UnsupportedOperationException}.
 * Generic versions of accessors delegate to the corresponding type-specific
 * counterparts following the interface rules (they take care of returning
 * {@code null} on a missing key).
 *
 * <p>
 * As a further help, this class provides a {@link BasicEntry BasicEntry} inner
 * class that implements a type-specific version of {@link java.util.Map.Entry};
 * it is particularly useful for those classes that do not implement their own
 * entries (e.g., most immutable maps).
 */

public abstract class AbstractLong2LongMap extends AbstractLong2LongFunction
		implements
			Long2LongMap,
			java.io.Serializable {

	private static final long serialVersionUID = -4940583368468432370L;

	protected AbstractLong2LongMap() {
	}

	@Override
	public boolean containsValue(final long v) {
		return values().contains(v);
	}

	@Override
	public boolean containsKey(final long k) {
		final ObjectIterator<Long2LongMap.Entry> i = long2LongEntrySet().iterator();
		while (i.hasNext())
			if (i.next().getLongKey() == k)
				return true;

		return false;
	}

	@Override
	public boolean isEmpty() {
		return size() == 0;
	}

	/**
	 * This class provides a basic but complete type-specific entry class for
	 * all those maps implementations that do not have entries on their own
	 * (e.g., most immutable maps).
	 *
	 * <p>
	 * This class does not implement {@link java.util.Map.Entry#setValue(Object)
	 * setValue()}, as the modification would not be reflected in the base map.
	 */

	public static class BasicEntry implements Long2LongMap.Entry {
		protected long key;
		protected long value;

		public BasicEntry() {
		}

		public BasicEntry(final Long key, final Long value) {
			this.key = (key).longValue();
			this.value = (value).longValue();
		}

		public BasicEntry(final long key, final long value) {
			this.key = key;
			this.value = value;
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method
		 *             instead.
		 */
		@Deprecated

		@Override
		public Long getKey() {
			return Long.valueOf(key);
		}

		@Override
		public long getLongKey() {
			return key;
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method
		 *             instead.
		 */
		@Deprecated

		@Override
		public Long getValue() {
			return Long.valueOf(value);
		}

		@Override
		public long getLongValue() {
			return value;
		}

		@Override
		public long setValue(final long value) {
			throw new UnsupportedOperationException();
		}

		/**
		 * {@inheritDoc}
		 * 
		 * @deprecated Please use the corresponding type-specific method
		 *             instead.
		 */
		@Deprecated
		@Override
		public Long setValue(final Long value) {
			return Long.valueOf(setValue(value.longValue()));
		}

		@Override
		public boolean equals(final Object o) {
			if (!(o instanceof Map.Entry))
				return false;
			final Map.Entry<?, ?> e = (Map.Entry<?, ?>) o;

			if (e.getKey() == null || !(e.getKey() instanceof Long))
				return false;

			if (e.getValue() == null || !(e.getValue() instanceof Long))
				return false;

			return ((key) == (((Long) (e.getKey())).longValue())) && ((value) == (((Long) (e.getValue())).longValue()));
		}

		@Override
		public int hashCode() {
			return it.unimi.dsi.fastutil.HashCommon.long2int(key) ^ it.unimi.dsi.fastutil.HashCommon.long2int(value);
		}

		@Override
		public String toString() {
			return key + "->" + value;
		}
	}

	/**
	 * Returns a type-specific-set view of the keys of this map.
	 *
	 * <p>
	 * The view is backed by the set returned by {@link Map#entrySet()}. Note
	 * that <em>no attempt is made at caching the result of this method</em>, as
	 * this would require adding some attributes that lightweight
	 * implementations would not need. Subclasses may easily override this
	 * policy by calling this method and caching the result, but implementors
	 * are encouraged to write more efficient ad-hoc implementations.
	 *
	 * @return a set view of the keys of this map; it may be safely cast to a
	 *         type-specific interface.
	 */
	@Override
	public LongSet keySet() {
		return new AbstractLongSet() {
			@Override
			public boolean contains(final long k) {
				return containsKey(k);
			}
			@Override
			public int size() {
				return AbstractLong2LongMap.this.size();
			}
			@Override
			public void clear() {
				AbstractLong2LongMap.this.clear();
			}
			@Override
			public LongIterator iterator() {
				return new LongIterator() {
					private final ObjectIterator<Long2LongMap.Entry> i = Long2LongMaps
							.fastIterator(AbstractLong2LongMap.this);
					@Override
					public long nextLong() {
						return i.next().getLongKey();
					};
					@Override
					public boolean hasNext() {
						return i.hasNext();
					}
					@Override
					public void remove() {
						i.remove();
					}
				};
			}
		};
	}

	/**
	 * Returns a type-specific-set view of the values of this map.
	 *
	 * <p>
	 * The view is backed by the set returned by {@link Map#entrySet()}. Note
	 * that <em>no attempt is made at caching the result of this method</em>, as
	 * this would require adding some attributes that lightweight
	 * implementations would not need. Subclasses may easily override this
	 * policy by calling this method and caching the result, but implementors
	 * are encouraged to write more efficient ad-hoc implementations.
	 *
	 * @return a set view of the values of this map; it may be safely cast to a
	 *         type-specific interface.
	 */
	@Override
	public LongCollection values() {
		return new AbstractLongCollection() {
			@Override
			public boolean contains(final long k) {
				return containsValue(k);
			}
			@Override
			public int size() {
				return AbstractLong2LongMap.this.size();
			}
			@Override
			public void clear() {
				AbstractLong2LongMap.this.clear();
			}

			@Override
			public LongIterator iterator() {
				return new LongIterator() {
					private final ObjectIterator<Long2LongMap.Entry> i = Long2LongMaps
							.fastIterator(AbstractLong2LongMap.this);
					@Override
					public long nextLong() {
						return i.next().getLongValue();
					}
					@Override
					public boolean hasNext() {
						return i.hasNext();
					}
				};
			}
		};
	}

	/** {@inheritDoc} */
	@SuppressWarnings({"unchecked", "deprecation"})
	@Override
	public void putAll(final Map<? extends Long, ? extends Long> m) {
		if (m instanceof Long2LongMap) {
			ObjectIterator<Long2LongMap.Entry> i = Long2LongMaps.fastIterator((Long2LongMap) m);

			while (i.hasNext()) {
				final Long2LongMap.Entry e = i.next();
				put(e.getLongKey(), e.getLongValue());
			}
		} else {
			int n = m.size();
			final Iterator<? extends Map.Entry<? extends Long, ? extends Long>> i = m.entrySet().iterator();
			Map.Entry<? extends Long, ? extends Long> e;
			while (n-- != 0) {
				e = i.next();
				put(e.getKey(), e.getValue());
			}
		}
	}

	/**
	 * Returns a hash code for this map.
	 *
	 * The hash code of a map is computed by summing the hash codes of its
	 * entries.
	 *
	 * @return a hash code for this map.
	 */
	@Override
	public int hashCode() {
		int h = 0, n = size();
		final ObjectIterator<Long2LongMap.Entry> i = Long2LongMaps.fastIterator(this);

		while (n-- != 0)
			h += i.next().hashCode();
		return h;
	}

	@Override
	public boolean equals(Object o) {
		if (o == this)
			return true;
		if (!(o instanceof Map))
			return false;

		final Map<?, ?> m = (Map<?, ?>) o;
		if (m.size() != size())
			return false;
		return long2LongEntrySet().containsAll(m.entrySet());
	}

	@Override
	public String toString() {
		final StringBuilder s = new StringBuilder();
		final ObjectIterator<Long2LongMap.Entry> i = Long2LongMaps.fastIterator(this);
		int n = size();
		Long2LongMap.Entry e;
		boolean first = true;

		s.append("{");

		while (n-- != 0) {
			if (first)
				first = false;
			else
				s.append(", ");

			e = i.next();

			s.append(String.valueOf(e.getLongKey()));
			s.append("=>");

			s.append(String.valueOf(e.getLongValue()));
		}

		s.append("}");
		return s.toString();
	}
}
