/* Copyright 2004, 2005, 2006 Acegi Technology Pty Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package it.mice.voila.runtime.security;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.security.core.userdetails.UserDetailsService;


/**
 * <p>Retrieves user details (username, password, enabled flag, and authorities) from a JDBC location.</p>
 * <p>A default database structure is assumed, (see {#DEF_USERS_BY_USERNAME_QUERY} and 
 * {#DEF_AUTHORITIES_BY_USERNAME_QUERY}, which most users of this class will need to override, if using an existing
 * scheme. This may be done by setting the default query strings used. If this does not provide enough flexibility,
 * another strategy would be to subclass this class and override the {org.springframework.jdbc.object.MappingSqlQuery} instances used, via the
 * {#initMappingSqlQueries()} extension point.</p>
 * <p>In order to minimise backward compatibility issues, this DAO does not recognise the expiration of user
 * accounts or the expiration of user credentials. However, it does recognise and honour the user enabled/disabled
 * column.</p>
 *
 * @author Ben Alex
 * @author colin sampaleanu
 * @version $Id: UserDetailsJdbcDaoImpl.java,v 1.1 2007/02/16 11:34:06 zzy9v4 Exp $
 */
public abstract class UserDetailsJdbcDao implements UserDetailsService {
    private static Log logger = LogFactory.getLog(UserDetailsJdbcDao.class);
	/**
	 * BLANK_PASSWORD
	 */
	protected static final String BLANK_PASSWORD = "";
	/**
	 * TO_DATE
	 */
	protected static final String TO_DATE = "9999/12/31";
	/**
	 * FROM_DATE
	 */
	protected static final String FROM_DATE = "0001/01/01";
	/**
	 * DATE_PATTERN
	 */
	protected static final String DATE_PATTERN = "yyyy/MM/dd";


    //~ Instance fields ================================================================================================
    private String rolePrefix = OperationPerProfileConfigAttribute.ROLE_PREFIX;
    private ApplicationHolder applicationHolder;

    //~ Constructors ===================================================================================================
    public UserDetailsJdbcDao() {
    }

    //~ Methods ========================================================================================================
    public String getRolePrefix() {
        return rolePrefix;
    }

    /**
     * Allows a default role prefix to be specified. If this is set to a non-empty value, then it is
     * automatically prepended to any roles read in from the db. This may for example be used to add the
     * <code>ROLE_</code> prefix expected to exist in role names (by default) by some other Acegi Security framework
     * classes, in the case that the prefix is not already present in the db.
     *
     * @param rolePrefix the new prefix
     */
    public void setRolePrefix(String rolePrefix) {
        this.rolePrefix = rolePrefix;
    }

    public ApplicationHolder getApplicationHolder() {
        return applicationHolder;
    }

    public void setApplicationHolder(ApplicationHolder applicationHolder) {
        this.applicationHolder = applicationHolder;
    }
    
    public abstract boolean updateNewPassword(
    		String userId,
    		String password,
    		DefaultDaoAuthenticationProvider daoAuthenticationProvider);
}
