package ir.map.sdk_services;

import org.json.JSONArray;
import org.json.JSONException;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import ir.map.sdk_common.MapirLatLng;

public class MapUtils {

    public static double wrap(double n, double min, double max) {
        return n >= min && n < max ? n : mod(n - min, max - min) + min;
    }

    static double mod(double x, double m) {
        return (x % m + m) % m;
    }

    public static boolean containsLocation(MapirLatLng point, List<MapirLatLng> polygon, boolean geodesic) {
        return containsLocation(point.latitude, point.longitude, polygon, geodesic);
    }

    public static boolean containsLocation(double latitude, double longitude, List<MapirLatLng> polygon, boolean geodesic) {
        int size = polygon.size();
        if (size == 0) {
            return false;
        } else {
            double lat3 = Math.toRadians(latitude);
            double lng3 = Math.toRadians(longitude);
            MapirLatLng prev = (MapirLatLng) polygon.get(size - 1);
            double lat1 = Math.toRadians(prev.latitude);
            double lng1 = Math.toRadians(prev.longitude);
            int nIntersect = 0;

            double lng2;
            for (Iterator var17 = polygon.iterator(); var17.hasNext(); lng1 = lng2) {
                MapirLatLng point2 = (MapirLatLng) var17.next();
                double dLng3 = MapUtils.wrap(lng3 - lng1, -3.141592653589793D, 3.141592653589793D);
                if (lat3 == lat1 && dLng3 == 0.0D) {
                    return true;
                }

                double lat2 = Math.toRadians(point2.latitude);
                lng2 = Math.toRadians(point2.longitude);
                if (intersects(lat1, lat2, MapUtils.wrap(lng2 - lng1, -3.141592653589793D, 3.141592653589793D), lat3, dLng3, geodesic)) {
                    ++nIntersect;
                }

                lat1 = lat2;
            }

            return (nIntersect & 1) != 0;
        }
    }

    private static boolean intersects(double lat1, double lat2, double lng2, double lat3, double lng3, boolean geodesic) {
        if ((lng3 < 0.0D || lng3 < lng2) && (lng3 >= 0.0D || lng3 >= lng2)) {
            if (lat3 <= -1.5707963267948966D) {
                return false;
            } else if (lat1 > -1.5707963267948966D && lat2 > -1.5707963267948966D && lat1 < 1.5707963267948966D && lat2 < 1.5707963267948966D) {
                if (lng2 <= -3.141592653589793D) {
                    return false;
                } else {
                    double linearLat = (lat1 * (lng2 - lng3) + lat2 * lng3) / lng2;
                    return lat1 >= 0.0D && lat2 >= 0.0D && lat3 < linearLat ? false : (lat1 <= 0.0D
                            && lat2 <= 0.0D
                            && lat3 >= linearLat ? true : (lat3 >= 1.5707963267948966D ? true : (geodesic ? Math.tan(lat3) >=
                            tanLatGC(lat1, lat2, lng2, lng3) : mercator(lat3) >= mercatorLatRhumb(lat1, lat2, lng2, lng3))));
                }
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    private static double tanLatGC(double lat1, double lat2, double lng2, double lng3) {
        return (Math.tan(lat1) * Math.sin(lng2 - lng3) + Math.tan(lat2) * Math.sin(lng3)) / Math.sin(lng2);
    }

    static double mercator(double lat) {
        return Math.log(Math.tan(lat * 0.5D + 0.7853981633974483D));
    }

    private static double mercatorLatRhumb(double lat1, double lat2, double lng2, double lng3) {
        return (mercator(lat1) * (lng2 - lng3) + mercator(lat2) * lng3) / lng2;
    }

    public ArrayList<MapirLatLng> getEvenOddPoses() {
        return getMapPoses("even-odd");
    }

    public ArrayList<MapirLatLng> getTrafficPoses() {
        return getMapPoses("traffic");
    }

    private ArrayList<MapirLatLng> getMapPoses(String type) {
        ArrayList<MapirLatLng> latLngs = new ArrayList<>();
        String json = loadJSONFromAsset(type);
        if (json != null) {
            try {
                JSONArray jsonArray = new JSONArray(json);
                for (int i = 0; i < jsonArray.length(); i++) {
                    latLngs.add(new MapirLatLng(jsonArray.getJSONArray(i).getDouble(1), jsonArray.getJSONArray(i).getDouble(0)));
                }

            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return latLngs;
    }

    private String loadJSONFromAsset(String type) {
        String json = null;
        try {
            InputStream is = null;
            switch (type) {
                case "traffic":
                    is = ServiceSDK.getContext().getAssets().open("traffic-json.json");
                    break;
                case "even-odd":
                    is = ServiceSDK.getContext().getAssets().open("even-odd.json");
                    break;
                case "earthqueke":
                    is = ServiceSDK.getContext().getAssets().open("earthqueke.geojson");
                    break;
                case "metro_stations_isfahan":
                    is = ServiceSDK.getContext().getAssets().open("metro_stations_isfahan.geojson");
                    break;
                case "metro_lines_isfahan":
                    is = ServiceSDK.getContext().getAssets().open("metro_lines_isfahan.geojson");
                    break;
                case "metro_stations_mashhad":
                    is = ServiceSDK.getContext().getAssets().open("metro_stations_mashhad.geojson");
                    break;
                case "metro_lines_mashhad":
                    is = ServiceSDK.getContext().getAssets().open("metro_lines_mashhad.geojson");
                    break;
                case "metro_stations_shiraz":
                    is = ServiceSDK.getContext().getAssets().open("metro_stations_shiraz.geojson");
                    break;
                case "metro_lines_shiraz":
                    is = ServiceSDK.getContext().getAssets().open("metro_lines_shiraz.geojson");
                    break;
                case "metro_stations_tabriz":
                    is = ServiceSDK.getContext().getAssets().open("metro_stations_tabriz.geojson");
                    break;
                case "metro_lines_tabriz":
                    is = ServiceSDK.getContext().getAssets().open("metro_lines_tabriz.geojson");
                    break;
                case "metro_stations_tehran":
                    is = ServiceSDK.getContext().getAssets().open("metro_stations_tehran.geojson");
                    break;
                case "metro_lines_tehran":
                    is = ServiceSDK.getContext().getAssets().open("metro_lines_tehran.geojson");
                    break;
                case "bus_lines":
                    is = ServiceSDK.getContext().getAssets().open("bus_lines.geojson");
                    break;
                case "bus_stations":
                    is = ServiceSDK.getContext().getAssets().open("bus_stations.geojson");
                    break;
            }
            if (is != null) {
                int size = is.available();
                byte[] buffer = new byte[size];
                is.read(buffer);
                is.close();
                json = new String(buffer, "UTF-8");
            }
        } catch (IOException ex) {
            ex.printStackTrace();
            return null;
        }

        return json;
    }
}
