package ir.map.sdk_map.location;

import com.mapbox.geojson.Feature;

import ir.map.sdk_map.style.layers.CircleLayer;
import ir.map.sdk_map.style.layers.Layer;
import ir.map.sdk_map.style.layers.Property;
import ir.map.sdk_map.style.layers.SymbolLayer;
import ir.map.sdk_map.style.sources.GeoJsonOptions;
import ir.map.sdk_map.style.sources.GeoJsonSource;

import static ir.map.sdk_map.location.LocationComponentConstants.ACCURACY_LAYER;
import static ir.map.sdk_map.location.LocationComponentConstants.BACKGROUND_LAYER;
import static ir.map.sdk_map.location.LocationComponentConstants.BEARING_LAYER;
import static ir.map.sdk_map.location.LocationComponentConstants.FOREGROUND_LAYER;
import static ir.map.sdk_map.location.LocationComponentConstants.LOCATION_SOURCE;
import static ir.map.sdk_map.location.LocationComponentConstants.PROPERTY_ACCURACY_ALPHA;
import static ir.map.sdk_map.location.LocationComponentConstants.PROPERTY_ACCURACY_COLOR;
import static ir.map.sdk_map.location.LocationComponentConstants.PROPERTY_ACCURACY_RADIUS;
import static ir.map.sdk_map.location.LocationComponentConstants.PROPERTY_BACKGROUND_ICON;
import static ir.map.sdk_map.location.LocationComponentConstants.PROPERTY_BACKGROUND_STALE_ICON;
import static ir.map.sdk_map.location.LocationComponentConstants.PROPERTY_BEARING_ICON;
import static ir.map.sdk_map.location.LocationComponentConstants.PROPERTY_COMPASS_BEARING;
import static ir.map.sdk_map.location.LocationComponentConstants.PROPERTY_FOREGROUND_ICON;
import static ir.map.sdk_map.location.LocationComponentConstants.PROPERTY_FOREGROUND_ICON_OFFSET;
import static ir.map.sdk_map.location.LocationComponentConstants.PROPERTY_FOREGROUND_STALE_ICON;
import static ir.map.sdk_map.location.LocationComponentConstants.PROPERTY_GPS_BEARING;
import static ir.map.sdk_map.location.LocationComponentConstants.PROPERTY_LOCATION_STALE;
import static ir.map.sdk_map.location.LocationComponentConstants.PROPERTY_SHADOW_ICON_OFFSET;
import static ir.map.sdk_map.location.LocationComponentConstants.SHADOW_ICON;
import static ir.map.sdk_map.location.LocationComponentConstants.SHADOW_LAYER;
import static ir.map.sdk_map.style.expressions.Expression.get;
import static ir.map.sdk_map.style.expressions.Expression.literal;
import static ir.map.sdk_map.style.expressions.Expression.match;
import static ir.map.sdk_map.style.expressions.Expression.stop;
import static ir.map.sdk_map.style.expressions.Expression.switchCase;
import static ir.map.sdk_map.style.layers.Property.ICON_ROTATION_ALIGNMENT_MAP;
import static ir.map.sdk_map.style.layers.PropertyFactory.circleColor;
import static ir.map.sdk_map.style.layers.PropertyFactory.circleOpacity;
import static ir.map.sdk_map.style.layers.PropertyFactory.circlePitchAlignment;
import static ir.map.sdk_map.style.layers.PropertyFactory.circleRadius;
import static ir.map.sdk_map.style.layers.PropertyFactory.circleStrokeColor;
import static ir.map.sdk_map.style.layers.PropertyFactory.iconAllowOverlap;
import static ir.map.sdk_map.style.layers.PropertyFactory.iconIgnorePlacement;
import static ir.map.sdk_map.style.layers.PropertyFactory.iconImage;
import static ir.map.sdk_map.style.layers.PropertyFactory.iconOffset;
import static ir.map.sdk_map.style.layers.PropertyFactory.iconRotate;
import static ir.map.sdk_map.style.layers.PropertyFactory.iconRotationAlignment;

class LayerSourceProvider {

  private static final String EMPTY_STRING = "";

  GeoJsonSource generateSource(Feature locationFeature) {
    return new GeoJsonSource(
      LOCATION_SOURCE,
      locationFeature,
      new GeoJsonOptions().withMaxZoom(16)
    );
  }

  Layer generateLayer(String layerId) {
    SymbolLayer layer = new SymbolLayer(layerId, LOCATION_SOURCE);
    layer.setProperties(
      iconAllowOverlap(true),
      iconIgnorePlacement(true),
      iconRotationAlignment(ICON_ROTATION_ALIGNMENT_MAP),
      iconRotate(
        match(literal(layerId), literal(0f),
          stop(FOREGROUND_LAYER, get(PROPERTY_GPS_BEARING)),
          stop(BACKGROUND_LAYER, get(PROPERTY_GPS_BEARING)),
          stop(SHADOW_LAYER, get(PROPERTY_GPS_BEARING)),
          stop(BEARING_LAYER, get(PROPERTY_COMPASS_BEARING))
        )
      ),
      iconImage(
        match(literal(layerId), literal(EMPTY_STRING),
          stop(FOREGROUND_LAYER, switchCase(
            get(PROPERTY_LOCATION_STALE), get(PROPERTY_FOREGROUND_STALE_ICON),
            get(PROPERTY_FOREGROUND_ICON))),
          stop(BACKGROUND_LAYER, switchCase(
            get(PROPERTY_LOCATION_STALE), get(PROPERTY_BACKGROUND_STALE_ICON),
            get(PROPERTY_BACKGROUND_ICON))),
          stop(SHADOW_LAYER, literal(SHADOW_ICON)),
          stop(BEARING_LAYER, get(PROPERTY_BEARING_ICON))
        )
      ),
      iconOffset(
        match(literal(layerId), literal(new Float[] {0f, 0f}),
          stop(literal(FOREGROUND_LAYER), get(PROPERTY_FOREGROUND_ICON_OFFSET)),
          stop(literal(SHADOW_LAYER), get(PROPERTY_SHADOW_ICON_OFFSET))
        )
      )
    );
    return layer;
  }

  Layer generateAccuracyLayer() {
    return new CircleLayer(ACCURACY_LAYER, LOCATION_SOURCE)
      .withProperties(
        circleRadius(get(PROPERTY_ACCURACY_RADIUS)),
        circleColor(get(PROPERTY_ACCURACY_COLOR)),
        circleOpacity(get(PROPERTY_ACCURACY_ALPHA)),
        circleStrokeColor(get(PROPERTY_ACCURACY_COLOR)),
        circlePitchAlignment(Property.CIRCLE_PITCH_ALIGNMENT_MAP)
      );
  }
}
