package ir.map.sdk_map;

import android.annotation.SuppressLint;
import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.UiThread;

import ir.map.sdk_map.constants.MapirConstants;
import ir.map.sdk_map.exceptions.MapirConfigurationException;
import ir.map.sdk_map.log.Logger;
import ir.map.sdk_map.maps.TelemetryDefinition;
import ir.map.sdk_map.net.ConnectivityReceiver;
import ir.map.sdk_map.storage.FileSource;
import ir.map.sdk_map.utils.ThreadUtils;

/**
 * The entry point to initialize the Mapir Android SDK.
 * <p>
 * Obtain a reference by calling {@link #getInstance(Context, String)}. Usually this class is configured in
 * Application#onCreate() and is responsible for the active access token, application context, and
 * connectivity state.
 * </p>
 */
@UiThread
@SuppressLint("StaticFieldLeak")
public final class Mapir {

  private static final String TAG = "Mbgl-Mapir";

  private static ModuleProvider moduleProvider;
  private static Mapir INSTANCE;

  private Context context;
  private String accessToken;
  private Boolean connected;
  private TelemetryDefinition telemetry;

  Mapir(@NonNull Context context, @Nullable String accessToken) {
    this.context = context;
    this.accessToken = accessToken;
  }

  /**
   * Get an instance of Mapir.
   * <p>
   * This class manages the Mapir access token, application context, and connectivity state.
   * </p>
   *
   * @param context     Android context which holds or is an application context
   * @param accessToken Mapir access token
   * @return the single instance of Mapir
   */
  @UiThread
  public static synchronized Mapir getInstance(@NonNull Context context, @NonNull String accessToken) {
    ThreadUtils.checkThread("Mapir");
    if (INSTANCE == null) {
      Context appContext = context.getApplicationContext();
      FileSource.initializeFileDirsPaths(appContext);
      INSTANCE = new Mapir(appContext, accessToken);
      if (isAccessTokenValid(accessToken)) {
        initializeTelemetry();
      }
      ConnectivityReceiver.instance(appContext);
    }
    return INSTANCE;
  }

  /**
   * Get the current active access token for this application.
   *
   * @return Mapir access token
   */
  @Nullable
  public static String getAccessToken() {
    validateMapir();
    return INSTANCE.accessToken;
  }

  /**
   * Set the current active accessToken.
   */
  public static void setAccessToken(String accessToken) {
    validateMapir();
    INSTANCE.accessToken = accessToken;
    FileSource.getInstance(getApplicationContext()).setAccessToken(accessToken);
  }

  /**
   * Application context
   *
   * @return the application context
   */
  @NonNull
  public static Context getApplicationContext() {
    validateMapir();
    return INSTANCE.context;
  }

  /**
   * Manually sets the connectivity state of the app. This is useful for apps which control their
   * own connectivity state and want to bypass any checks to the ConnectivityManager.
   *
   * @param connected flag to determine the connectivity state, true for connected, false for
   *                  disconnected, and null for ConnectivityManager to determine.
   */
  public static synchronized void setConnected(Boolean connected) {
    validateMapir();
    // Connectivity state overridden by app
    INSTANCE.connected = connected;
  }

  /**
   * Determines whether we have an internet connection available. Please do not rely on this
   * method in your apps. This method is used internally by the SDK.
   *
   * @return true if there is an internet connection, false otherwise
   */
  public static synchronized Boolean isConnected() {
    validateMapir();
    if (INSTANCE.connected != null) {
      // Connectivity state overridden by app
      return INSTANCE.connected;
    }

    ConnectivityManager cm = (ConnectivityManager) INSTANCE.context.getSystemService(Context.CONNECTIVITY_SERVICE);
    NetworkInfo activeNetwork = cm.getActiveNetworkInfo();
    return (activeNetwork != null && activeNetwork.isConnected());
  }

  /**
   * Initializes telemetry
   */
  private static void initializeTelemetry() {
    try {
      INSTANCE.telemetry = getModuleProvider().obtainTelemetry();
    } catch (Exception exception) {
      String message = "Error occurred while initializing telemetry";
      Logger.e(TAG, message, exception);
      MapStrictMode.strictModeViolation(message, exception);
    }
  }

  /**
   * Get an instance of Telemetry if initialised
   *
   * @return instance of telemetry
   */
  @Nullable
  public static TelemetryDefinition getTelemetry() {
    return INSTANCE.telemetry;
  }

  /**
   * Get the module provider
   *
   * @return moduleProvider
   */
  @NonNull
  public static ModuleProvider getModuleProvider() {
    if (moduleProvider == null) {
      moduleProvider = new ModuleProviderImpl();
    }
    return moduleProvider;
  }

  /**
   * Runtime validation of Mapir creation.
   */
  private static void validateMapir() {
    if (INSTANCE == null) {
      throw new MapirConfigurationException();
    }
  }

  /**
   * Runtime validation of Mapir access token
   *
   * @param accessToken the access token to validate
   * @return true is valid, false otherwise
   */
  static boolean isAccessTokenValid(String accessToken) {
    if (accessToken == null) {
      return false;
    }

    accessToken = accessToken.trim().toLowerCase(MapirConstants.MAPIR_LOCALE);
    return accessToken.length() != 0 && (accessToken.startsWith("pk.") || accessToken.startsWith("sk."));
  }
}