package ir.map.sdk_map.wrapper;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.TimeInterpolator;
import android.animation.ValueAnimator;
import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.content.Context;
import android.graphics.Color;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.LayerDrawable;
import android.graphics.drawable.ShapeDrawable;
import android.graphics.drawable.shapes.OvalShape;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.os.MessageQueue;
import android.util.SparseArray;
import android.view.ViewGroup;
import android.view.animation.DecelerateInterpolator;

import com.google.android.gms.maps.model.LatLng;
import com.google.maps.android.geometry.Point;
import com.google.maps.android.projection.SphericalMercatorProjection;
import com.google.maps.android.ui.SquareTextView;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import ir.map.sdk_common.MaptexLatLng;
import ir.map.sdk_map.utils.IconGenerator;

/**
 * @author haniyeh
 */

public class MaptexDefaultClusterRenderer<T extends MaptexClusterItem> implements MaptexClusterRenderer<T> {
    private static final boolean SHOULD_ANIMATE;
    private static final int[] BUCKETS;
    private static final TimeInterpolator ANIMATION_INTERP;

    static {
        SHOULD_ANIMATE = Build.VERSION.SDK_INT >= 11;
        BUCKETS = new int[]{10, 20, 50, 100, 200, 500, 1000};
        ANIMATION_INTERP = new DecelerateInterpolator();
    }

    private final MaptexMap mMap;
    private final IconGenerator mIconGenerator;
    private final MaptexClusterManager<T> mClusterManager;
    private final float mDensity;
    private final ViewModifier mViewModifier = new ViewModifier();
    private boolean mAnimate;
    private ShapeDrawable mColoredCircleBackground;
    private Set<MarkerWithPosition> mMarkers = Collections.newSetFromMap(new ConcurrentHashMap());
    private SparseArray<MaptexBitmapDescriptor> mIcons = new SparseArray();
    private MaptexMarkerCache<T> mMarkerCache = new MaptexMarkerCache();
    private int mMinClusterSize = 4;
    private Set<? extends MaptexCluster<T>> mClusters;
    private Map<MaptexMarker, MaptexCluster<T>> mMarkerToCluster = new HashMap();
    private Map<MaptexCluster<T>, MaptexMarker> mClusterToMarker = new HashMap();
    private float mZoom;
    private MaptexClusterManager.OnClusterClickListener<T> mClickListener;
    private MaptexClusterManager.OnClusterInfoWindowClickListener<T> mInfoWindowClickListener;
    private MaptexClusterManager.OnClusterItemClickListener<T> mItemClickListener;
    private MaptexClusterManager.OnClusterItemInfoWindowClickListener<T> mItemInfoWindowClickListener;

    public MaptexDefaultClusterRenderer(Context context, MaptexMap map, MaptexClusterManager<T> clusterManager) {
        this.mMap = map;
        this.mAnimate = true;
        this.mDensity = context.getResources().getDisplayMetrics().density;
        this.mIconGenerator = new IconGenerator(context);
        this.mIconGenerator.setContentView(this.makeSquareTextView(context));
        this.mIconGenerator.setTextAppearance(com.google.maps.android.R.style.amu_ClusterIcon_TextAppearance);
        this.mIconGenerator.setBackground(this.makeClusterBackground());
        this.mClusterManager = clusterManager;
    }

    private static double distanceSquared(Point a, Point b) {
        return (a.x - b.x) * (a.x - b.x) + (a.y - b.y) * (a.y - b.y);
    }

    private static Point findClosestCluster(List<Point> markers, Point point) {
        if (markers != null && !markers.isEmpty()) {
            double minDistSquared = 10000.0D;
            Point closest = null;
            Iterator var5 = markers.iterator();

            while (var5.hasNext()) {
                Point candidate = (Point) var5.next();
                double dist = distanceSquared(candidate, point);
                if (dist < minDistSquared) {
                    closest = candidate;
                    minDistSquared = dist;
                }
            }

            return closest;
        } else {
            return null;
        }
    }

    public void onAdd() {
        this.mClusterManager.getMarkerCollection().setOnMarkerClickListener(new MaptexMap.OnMarkerClickListener() {
            public boolean onMarkerClick(MaptexMarker marker) {
                return mItemClickListener != null
                        && mItemClickListener
                        .onClusterItemClick(mMarkerCache.get(marker));
            }
        });
        this.mClusterManager.getMarkerCollection().setOnInfoWindowClickListener(new MaptexMap.OnInfoWindowClickListener() {
            public void onInfoWindowClick(MaptexMarker marker) {
                if (mItemInfoWindowClickListener != null) {
                    mItemInfoWindowClickListener
                            .onClusterItemInfoWindowClick(mMarkerCache.get(marker));
                }

            }
        });
        this.mClusterManager.getClusterMarkerCollection().setOnMarkerClickListener(new MaptexMap.OnMarkerClickListener() {
            public boolean onMarkerClick(MaptexMarker marker) {
                return mClickListener != null
                        && mClickListener
                        .onClusterClick(mMarkerToCluster.get(marker));
            }
        });
        this.mClusterManager.getClusterMarkerCollection().setOnInfoWindowClickListener(new MaptexMap.OnInfoWindowClickListener() {
            public void onInfoWindowClick(MaptexMarker marker) {
                if (mInfoWindowClickListener != null) {
                    mInfoWindowClickListener
                            .onClusterInfoWindowClick(mMarkerToCluster.get(marker));
                }

            }
        });
    }

    public void onRemove() {
        this.mClusterManager.getMarkerCollection().setOnMarkerClickListener(null);
        this.mClusterManager.getMarkerCollection().setOnInfoWindowClickListener(null);
        this.mClusterManager.getClusterMarkerCollection().setOnMarkerClickListener(null);
        this.mClusterManager.getClusterMarkerCollection().setOnInfoWindowClickListener(null);
    }

    private LayerDrawable makeClusterBackground() {
        this.mColoredCircleBackground = new ShapeDrawable(new OvalShape());
        ShapeDrawable outline = new ShapeDrawable(new OvalShape());
        outline.getPaint().setColor(-2130706433);
        LayerDrawable background = new LayerDrawable(new Drawable[]{outline, this.mColoredCircleBackground});
        int strokeWidth = (int) (this.mDensity * 3.0F);
        background.setLayerInset(1, strokeWidth, strokeWidth, strokeWidth, strokeWidth);
        return background;
    }

    private SquareTextView makeSquareTextView(Context context) {
        SquareTextView squareTextView = new SquareTextView(context);
        ViewGroup.LayoutParams layoutParams = new ViewGroup.LayoutParams(-2, -2);
        squareTextView.setLayoutParams(layoutParams);
        squareTextView.setId(com.google.maps.android.R.id.amu_text);
        int twelveDpi = (int) (12.0F * this.mDensity);
        squareTextView.setPadding(twelveDpi, twelveDpi, twelveDpi, twelveDpi);
        return squareTextView;
    }

    protected int getColor(int clusterSize) {
        float hueRange = 220.0F;
        float sizeRange = 300.0F;
        float size = Math.min((float) clusterSize, 300.0F);
        float hue = (300.0F - size) * (300.0F - size) / 90000.0F * 220.0F;
        return Color.HSVToColor(new float[]{hue, 1.0F, 0.6F});
    }

    protected String getClusterText(int bucket) {
        return bucket < BUCKETS[0] ? String.valueOf(bucket) : bucket + "+";
    }

    protected int getBucket(MaptexCluster<T> cluster) {
        int size = cluster.getSize();
        if (size <= BUCKETS[0]) {
            return size;
        } else {
            for (int i = 0; i < BUCKETS.length - 1; ++i) {
                if (size < BUCKETS[i + 1]) {
                    return BUCKETS[i];
                }
            }

            return BUCKETS[BUCKETS.length - 1];
        }
    }

    public int getMinClusterSize() {
        return this.mMinClusterSize;
    }

    public void setMinClusterSize(int minClusterSize) {
        this.mMinClusterSize = minClusterSize;
    }

    protected boolean shouldRenderAsCluster(MaptexCluster<T> cluster) {
        return cluster.getSize() > this.mMinClusterSize;
    }

    public void onClustersChanged(Set<? extends MaptexCluster<T>> clusters) {
        this.mViewModifier.queue(clusters);
    }

    public void setOnClusterClickListener(MaptexClusterManager.OnClusterClickListener<T> listener) {
        this.mClickListener = listener;
    }

    public void setOnClusterInfoWindowClickListener(MaptexClusterManager.OnClusterInfoWindowClickListener<T> listener) {
        this.mInfoWindowClickListener = listener;
    }

    public void setOnClusterItemClickListener(MaptexClusterManager.OnClusterItemClickListener<T> listener) {
        this.mItemClickListener = listener;
    }

    public void setOnClusterItemInfoWindowClickListener(MaptexClusterManager.OnClusterItemInfoWindowClickListener<T> listener) {
        this.mItemInfoWindowClickListener = listener;
    }

    public void setAnimation(boolean animate) {
        this.mAnimate = animate;
    }

    protected void onBeforeClusterItemRendered(T item, MaptexMarkerOptions markerOptions) {
    }

    protected void onBeforeClusterRendered(MaptexCluster<T> cluster, MaptexMarkerOptions markerOptions) {
        int bucket = this.getBucket(cluster);
        MaptexBitmapDescriptor descriptor = this.mIcons.get(bucket);
        if (descriptor == null) {
            this.mColoredCircleBackground.getPaint().setColor(this.getColor(bucket));
            descriptor = MaptexBitmapDescriptorFactory.fromBitmap(this.mIconGenerator.makeIcon(this.getClusterText(bucket)));
            this.mIcons.put(bucket, descriptor);
        }

        markerOptions.icon(descriptor);
    }

    protected void onClusterRendered(MaptexCluster<T> cluster, MaptexMarker marker) {
    }

    protected void onClusterItemRendered(T clusterItem, MaptexMarker marker) {
    }

    public MaptexMarker getMarker(T clusterItem) {
        return mMarkerCache.get((T) clusterItem);
    }

    public T getClusterItem(MaptexMarker marker) {
        return mMarkerCache.get(marker);
    }

    public MaptexMarker getMarker(MaptexCluster<T> cluster) {
        return mClusterToMarker.get(cluster);
    }

    public MaptexCluster<T> getCluster(MaptexMarker marker) {
        return mMarkerToCluster.get(marker);
    }

    private static class MarkerWithPosition {
        private final MaptexMarker marker;
        private MaptexLatLng position;

        private MarkerWithPosition(MaptexMarker marker) {
            this.marker = marker;
            this.position = marker.getPosition();
        }

        public boolean equals(Object other) {
            return other instanceof MarkerWithPosition ? this.marker
                    .equals(((MarkerWithPosition) other).marker) : false;
        }

        public int hashCode() {
            return this.marker.hashCode();
        }
    }

    private static class MaptexMarkerCache<T> {
        private Map<T, MaptexMarker> mCache;
        private Map<MaptexMarker, T> mCacheReverse;

        private MaptexMarkerCache() {
            this.mCache = new HashMap();
            this.mCacheReverse = new HashMap();
        }

        public MaptexMarker get(T item) {
            return this.mCache.get(item);
        }

        public T get(MaptexMarker m) {
            return this.mCacheReverse.get(m);
        }

        public void put(T item, MaptexMarker m) {
            this.mCache.put(item, m);
            this.mCacheReverse.put(m, item);
        }

        public void remove(MaptexMarker m) {
            T item = this.mCacheReverse.get(m);
            this.mCacheReverse.remove(m);
            this.mCache.remove(item);
        }
    }

    @TargetApi(12)
    private class AnimationTask extends AnimatorListenerAdapter implements ValueAnimator.AnimatorUpdateListener {
        private final MarkerWithPosition markerWithPosition;
        private final MaptexMarker marker;
        private final MaptexLatLng from;
        private final MaptexLatLng to;
        private boolean mRemoveOnComplete;
        private MaptexMarkerManager mMarkerManager;

        private AnimationTask(MarkerWithPosition markerWithPosition, MaptexLatLng from, MaptexLatLng to) {
            this.markerWithPosition = markerWithPosition;
            this.marker = markerWithPosition.marker;
            this.from = from;
            this.to = to;
        }

        public void perform() {
            ValueAnimator valueAnimator = ValueAnimator.ofFloat(new float[]{0.0F, 1.0F});
            valueAnimator.setInterpolator(ANIMATION_INTERP);
            valueAnimator.addUpdateListener(this);
            valueAnimator.addListener(this);
            valueAnimator.start();
        }

        public void onAnimationEnd(Animator animation) {
            if (this.mRemoveOnComplete) {
                MaptexCluster<T> cluster = mMarkerToCluster.get(this.marker);
                mClusterToMarker.remove(cluster);
                mMarkerCache.remove(this.marker);
                mMarkerToCluster.remove(this.marker);
                mMarkerManager.remove(this.marker);
            }

            this.markerWithPosition.position = this.to;
        }

        public void removeOnAnimationComplete(MaptexMarkerManager markerManager) {
            this.mMarkerManager = markerManager;
            this.mRemoveOnComplete = true;
        }

        public void onAnimationUpdate(ValueAnimator valueAnimator) {
            float fraction = valueAnimator.getAnimatedFraction();
            double lat = (this.to.latitude - this.from.latitude) * (double) fraction + this.from.latitude;
            double lngDelta = this.to.longitude - this.from.longitude;
            if (Math.abs(lngDelta) > 180.0D) {
                lngDelta -= Math.signum(lngDelta) * 360.0D;
            }

            double lng = lngDelta * (double) fraction + this.from.longitude;
            MaptexLatLng position = new MaptexLatLng(lat, lng);
            this.marker.setPosition(position);
        }
    }

    private class CreateMarkerTask {
        private final MaptexCluster<T> cluster;
        private final Set<MarkerWithPosition> newMarkers;
        private final MaptexLatLng animateFrom;

        public CreateMarkerTask(MaptexCluster<T> var1, Set<MarkerWithPosition> c, MaptexLatLng markersAdded) {
            this.cluster = var1;
            this.newMarkers = c;
            this.animateFrom = markersAdded;
        }

        private void perform(MaptexMarkerModifier markerModifier) {
            if (shouldRenderAsCluster(this.cluster)) {
                MaptexMarker marker = mClusterToMarker.get(this.cluster);
                MarkerWithPosition markerWithPosition;
                if (marker == null) {
                    MaptexMarkerOptions markerOptionsx = (new MaptexMarkerOptions())
                            .position(this.animateFrom == null ? this.cluster.getPosition() : this.animateFrom);
                    onBeforeClusterRendered(this.cluster, markerOptionsx);
                    marker = mClusterManager.getClusterMarkerCollection().addMarker(markerOptionsx);
                    mMarkerToCluster.put(marker, this.cluster);
                    mClusterToMarker.put(this.cluster, marker);
                    markerWithPosition = new MarkerWithPosition(marker);
                    if (this.animateFrom != null) {
                        markerModifier.animate(markerWithPosition, this.animateFrom, this.cluster.getPosition());
                    }
                } else {
                    markerWithPosition = new MarkerWithPosition(marker);
                }

                onClusterRendered(this.cluster, marker);
                this.newMarkers.add(markerWithPosition);
            } else {
                Iterator var2 = this.cluster.getItems().iterator();

                while (var2.hasNext()) {
                    T item = (T) var2.next();
                    MaptexMarker markerx = mMarkerCache.get((T) item);
                    MarkerWithPosition markerWithPositionx;
                    if (markerx == null) {
                        MaptexMarkerOptions markerOptions = new MaptexMarkerOptions();
                        if (this.animateFrom != null) {
                            markerOptions.position(this.animateFrom);
                        } else {
                            markerOptions.position(item.getPosition());
                        }

                        if (item.getTitle() != null && item.getSnippet() != null) {
                            markerOptions.title(item.getTitle());
                            markerOptions.snippet(item.getSnippet());
                        } else if (item.getSnippet() != null) {
                            markerOptions.title(item.getSnippet());
                        } else if (item.getTitle() != null) {
                            markerOptions.title(item.getTitle());
                        }

                        onBeforeClusterItemRendered(item, markerOptions);
                        markerx = mClusterManager.getMarkerCollection().addMarker(markerOptions);
                        markerWithPositionx = new MarkerWithPosition(markerx);
                        mMarkerCache.put(item, markerx);
                        if (this.animateFrom != null) {
                            markerModifier.animate(markerWithPositionx, this.animateFrom, item.getPosition());
                        }
                    } else {
                        markerWithPositionx = new MarkerWithPosition(markerx);
                    }

                    onClusterItemRendered(item, markerx);
                    this.newMarkers.add(markerWithPositionx);
                }

            }
        }
    }

    @SuppressLint({"HandlerLeak"})
    private class MaptexMarkerModifier extends Handler implements MessageQueue.IdleHandler {
        private static final int BLANK = 0;
        private final Lock lock;
        private final Condition busyCondition;
        private Queue<CreateMarkerTask> mCreateMarkerTasks;
        private Queue<CreateMarkerTask> mOnScreenCreateMarkerTasks;
        private Queue<MaptexMarker> mRemoveMarkerTasks;
        private Queue<MaptexMarker> mOnScreenRemoveMarkerTasks;
        private Queue<AnimationTask> mAnimationTasks;
        private boolean mListenerAdded;

        private MaptexMarkerModifier() {
            super(Looper.getMainLooper());
            this.lock = new ReentrantLock();
            this.busyCondition = this.lock.newCondition();
            this.mCreateMarkerTasks = new LinkedList();
            this.mOnScreenCreateMarkerTasks = new LinkedList();
            this.mRemoveMarkerTasks = new LinkedList();
            this.mOnScreenRemoveMarkerTasks = new LinkedList();
            this.mAnimationTasks = new LinkedList();
        }

        public void add(boolean priority, CreateMarkerTask c) {
            this.lock.lock();
            this.sendEmptyMessage(0);
            if (priority) {
                this.mOnScreenCreateMarkerTasks.add(c);
            } else {
                this.mCreateMarkerTasks.add(c);
            }

            this.lock.unlock();
        }

        public void remove(boolean priority, MaptexMarker m) {
            this.lock.lock();
            this.sendEmptyMessage(0);
            if (priority) {
                this.mOnScreenRemoveMarkerTasks.add(m);
            } else {
                this.mRemoveMarkerTasks.add(m);
            }

            this.lock.unlock();
        }

        public void animate(MarkerWithPosition marker, MaptexLatLng from, MaptexLatLng to) {
            this.lock.lock();
            this.mAnimationTasks.add(new AnimationTask(marker, from, to));
            this.lock.unlock();
        }

        @TargetApi(11)
        public void animateThenRemove(MarkerWithPosition marker, MaptexLatLng from, MaptexLatLng to) {
            this.lock.lock();
            AnimationTask animationTask =
                    new AnimationTask(marker, from, to);
            animationTask.removeOnAnimationComplete(mClusterManager.getMarkerManager());
            this.mAnimationTasks.add(animationTask);
            this.lock.unlock();
        }

        public void handleMessage(Message msg) {
            if (!this.mListenerAdded) {
                Looper.myQueue().addIdleHandler(this);
                this.mListenerAdded = true;
            }

            this.removeMessages(0);
            this.lock.lock();

            try {
                int i = 0;

                while (true) {
                    if (i >= 10) {
                        if (!this.isBusy()) {
                            this.mListenerAdded = false;
                            Looper.myQueue().removeIdleHandler(this);
                            this.busyCondition.signalAll();
                        } else {
                            this.sendEmptyMessageDelayed(0, 10L);
                        }
                        break;
                    }

                    this.performNextTask();
                    ++i;
                }
            } finally {
                this.lock.unlock();
            }

        }

        @TargetApi(11)
        private void performNextTask() {
            if (!this.mOnScreenRemoveMarkerTasks.isEmpty()) {
                this.removeMarker(this.mOnScreenRemoveMarkerTasks.poll());
            } else if (!this.mAnimationTasks.isEmpty()) {
                (this.mAnimationTasks.poll()).perform();
            } else if (!this.mOnScreenCreateMarkerTasks.isEmpty()) {
                (this.mOnScreenCreateMarkerTasks.poll()).perform(this);
            } else if (!this.mCreateMarkerTasks.isEmpty()) {
                (this.mCreateMarkerTasks.poll()).perform(this);
            } else if (!this.mRemoveMarkerTasks.isEmpty()) {
                this.removeMarker(this.mRemoveMarkerTasks.poll());
            }

        }

        private void removeMarker(MaptexMarker m) {
            MaptexCluster<T> cluster = mMarkerToCluster.get(m);
            mClusterToMarker.remove(cluster);
            mMarkerCache.remove(m);
            mMarkerToCluster.remove(m);
            mClusterManager.getMarkerManager().remove(m);
        }

        public boolean isBusy() {
            boolean var1;
            try {
                this.lock.lock();
                var1 = !this.mCreateMarkerTasks.isEmpty() || !this.mOnScreenCreateMarkerTasks.isEmpty()
                        || !this.mOnScreenRemoveMarkerTasks.isEmpty() || !this.mRemoveMarkerTasks.isEmpty() || !this.mAnimationTasks.isEmpty();
            } finally {
                this.lock.unlock();
            }

            return var1;
        }

        public void waitUntilFree() {
            while (this.isBusy()) {
                this.sendEmptyMessage(0);
                this.lock.lock();

                try {
                    if (this.isBusy()) {
                        this.busyCondition.await();
                    }
                } catch (InterruptedException var5) {
                    throw new RuntimeException(var5);
                } finally {
                    this.lock.unlock();
                }
            }

        }

        public boolean queueIdle() {
            this.sendEmptyMessage(0);
            return true;
        }
    }

    private class RenderTask implements Runnable {
        final Set<? extends MaptexCluster<T>> clusters;
        private Runnable mCallback;
        private MaptexProjection mProjection;
        private SphericalMercatorProjection mSphericalMercatorProjection;
        private float mMapZoom;

        private RenderTask(Set<? extends MaptexCluster<T>> var1) {
            this.clusters = var1;
        }

        public void setCallback(Runnable callback) {
            this.mCallback = callback;
        }

        public void setProjection(MaptexProjection projection) {
            this.mProjection = projection;
        }

        public void setMapZoom(float zoom) {
            this.mMapZoom = zoom;
            this.mSphericalMercatorProjection = new SphericalMercatorProjection(
                    256.0D * Math.pow(2.0D, (double) Math.min(zoom, mZoom)));
        }

        @SuppressLint({"NewApi"})
        public void run() {
            if (this.clusters.equals(mClusters)) {
                this.mCallback.run();
            } else {
                MaptexMarkerModifier markerModifier = new MaptexMarkerModifier();
                float zoom = this.mMapZoom;
                boolean zoomingIn = zoom > mZoom;
                float zoomDelta = zoom - mZoom;
                Set<MarkerWithPosition> markersToRemove = mMarkers;
                MaptexLatLngBounds visibleBounds = this.mProjection.getVisibleRegion().maptexLatLngBounds;
                List<Point> existingClustersOnScreen = null;
                if (mClusters != null && SHOULD_ANIMATE) {
                    existingClustersOnScreen = new ArrayList();
                    Iterator var8 = mClusters.iterator();

                    while (var8.hasNext()) {
                        MaptexCluster<T> c = (MaptexCluster) var8.next();
                        if (shouldRenderAsCluster(c) && visibleBounds.contains(c.getPosition())) {
                            Point pointx = this.mSphericalMercatorProjection.toPoint(new LatLng(c.getPosition().latitude, c.getPosition().longitude));
                            existingClustersOnScreen.add(pointx);
                        }
                    }
                }

                Set<MarkerWithPosition> newMarkers = Collections.newSetFromMap(new ConcurrentHashMap());
                Iterator var17 = this.clusters.iterator();

                while (true) {
                    while (true) {
                        com.google.maps.android.projection.Point p;
                        while (var17.hasNext()) {
                            MaptexCluster<T> cx = (MaptexCluster) var17.next();
                            boolean onScreenx = visibleBounds.contains(cx.getPosition());
                            if (zoomingIn && onScreenx && SHOULD_ANIMATE) {
                                p = this.mSphericalMercatorProjection.toPoint(new LatLng(cx.getPosition().latitude, cx.getPosition().longitude));
                                Point closestx = findClosestCluster(existingClustersOnScreen, p);
                                if (closestx != null && mAnimate) {
                                    LatLng latLng = this.mSphericalMercatorProjection.toLatLng(closestx);
                                    MaptexLatLng animateTo = new MaptexLatLng(latLng.latitude, latLng.longitude);
                                    markerModifier.add(
                                            true, new CreateMarkerTask(cx, newMarkers, animateTo));
                                } else {
                                    markerModifier.add(true, new CreateMarkerTask(cx, newMarkers, (MaptexLatLng) null));
                                }
                            } else {
                                markerModifier.add(onScreenx, new CreateMarkerTask(cx, newMarkers, (MaptexLatLng) null));
                            }
                        }

                        markerModifier.waitUntilFree();
                        markersToRemove.removeAll(newMarkers);
                        List<Point> newClustersOnScreen = null;
                        Iterator var20;
                        if (SHOULD_ANIMATE) {
                            newClustersOnScreen = new ArrayList();
                            var20 = this.clusters.iterator();

                            while (var20.hasNext()) {
                                MaptexCluster<T> cxx = (MaptexCluster) var20.next();
                                if (shouldRenderAsCluster(cxx) && visibleBounds.contains(cxx.getPosition())) {
                                    p = this.mSphericalMercatorProjection.toPoint(
                                            new LatLng(cxx.getPosition().latitude, cxx.getPosition().longitude));
                                    newClustersOnScreen.add(p);
                                }
                            }
                        }

                        var20 = markersToRemove.iterator();

                        while (true) {
                            while (true) {
                                while (var20.hasNext()) {
                                    MarkerWithPosition marker = (MarkerWithPosition) var20.next();
                                    boolean onScreen = visibleBounds.contains(marker.position);
                                    if (!zoomingIn && zoomDelta > -3.0F && onScreen && SHOULD_ANIMATE) {
                                        Point point = this.mSphericalMercatorProjection.toPoint(
                                                new LatLng(marker.position.latitude, marker.position.longitude));
                                        Point closest = findClosestCluster(newClustersOnScreen, point);
                                        if (closest != null && mAnimate) {
                                            LatLng latLng = this.mSphericalMercatorProjection.toLatLng(closest);
                                            MaptexLatLng animateTox = new MaptexLatLng(latLng.latitude, latLng.longitude);
                                            markerModifier.animateThenRemove(marker, marker.position, animateTox);
                                        } else {
                                            markerModifier.remove(true, marker.marker);
                                        }
                                    } else {
                                        markerModifier.remove(onScreen, marker.marker);
                                    }
                                }

                                markerModifier.waitUntilFree();
                                mMarkers = newMarkers;
                                mClusters = this.clusters;
                                mZoom = zoom;
                                this.mCallback.run();
                                return;
                            }
                        }
                    }
                }
            }
        }
    }

    @SuppressLint({"HandlerLeak"})
    private class ViewModifier extends Handler {
        private static final int RUN_TASK = 0;
        private static final int TASK_FINISHED = 1;
        private boolean mViewModificationInProgress;
        private RenderTask mNextClusters;

        private ViewModifier() {
            this.mViewModificationInProgress = false;
            this.mNextClusters = null;
        }

        public void handleMessage(Message msg) {
            if (msg.what == 1) {
                this.mViewModificationInProgress = false;
                if (this.mNextClusters != null) {
                    this.sendEmptyMessage(0);
                }

            } else {
                this.removeMessages(0);
                if (!this.mViewModificationInProgress) {
                    if (this.mNextClusters != null) {
                        MaptexProjection projection = mMap.getProjection();
                        RenderTask renderTask;
                        synchronized (this) {
                            renderTask = this.mNextClusters;
                            this.mNextClusters = null;
                            this.mViewModificationInProgress = true;
                        }

                        renderTask.setCallback(new Runnable() {
                            public void run() {
                                ViewModifier.this.sendEmptyMessage(1);
                            }
                        });
                        renderTask.setProjection(projection);
                        renderTask.setMapZoom(mMap.getCameraPosition().zoom);
                        (new Thread(renderTask)).start();
                    }
                }
            }
        }

        public void queue(Set<? extends MaptexCluster<T>> clusters) {
            synchronized (this) {
                this.mNextClusters = new RenderTask(clusters);
            }

            this.sendEmptyMessage(0);
        }
    }
}
