package ir.map.sdk_map.wrapper;

import android.content.Context;
import android.os.AsyncTask;
import android.os.Build;

import java.util.Set;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;


/**
 * @author haniyeh
 */

public class MaptexClusterManager<T extends MaptexClusterItem> implements MaptexMap.OnCameraIdleListener,
        MaptexMap.OnMarkerClickListener, MaptexMap.OnInfoWindowClickListener {
    private final MaptexMarkerManager mMarkerManager;
    private final MaptexMarkerManager.Collection mMarkers;
    private final MaptexMarkerManager.Collection mClusterMarkers;
    private final ReadWriteLock mAlgorithmLock;
    private final ReadWriteLock mClusterTaskLock;
    private MaptexAlgorithm<T> mAlgorithm;
    private MaptexClusterRenderer<T> mRenderer;
    private MaptexMap mMap;
    private MaptexCameraPosition mPreviousCameraPosition;
    private ClusterTask mClusterTask;
    private OnClusterItemClickListener<T> mOnClusterItemClickListener;
    private OnClusterInfoWindowClickListener<T> mOnClusterInfoWindowClickListener;
    private OnClusterItemInfoWindowClickListener<T> mOnClusterItemInfoWindowClickListener;
    private OnClusterClickListener<T> mOnClusterClickListener;

    public MaptexClusterManager(Context context, MaptexMap map) {
        this(context, map, new MaptexMarkerManager(map));
    }

    public MaptexClusterManager(Context context, MaptexMap map, MaptexMarkerManager markerManager) {
        this.mAlgorithmLock = new ReentrantReadWriteLock();
        this.mClusterTaskLock = new ReentrantReadWriteLock();
        this.mMap = map;
        this.mMarkerManager = markerManager;
        this.mClusterMarkers = markerManager.newCollection();
        this.mMarkers = markerManager.newCollection();
        this.mRenderer = new MaptexDefaultClusterRenderer(context, map, this);
        this.mAlgorithm = new MaptexPreCachingAlgorithmDecorator(new MaptexNonHierarchicalDistanceBasedAlgorithm());
        this.mClusterTask = new ClusterTask();
        this.mRenderer.onAdd();
    }

    public MaptexMarkerManager.Collection getMarkerCollection() {
        return this.mMarkers;
    }

    public MaptexMarkerManager.Collection getClusterMarkerCollection() {
        return this.mClusterMarkers;
    }

    public MaptexMarkerManager getMarkerManager() {
        return this.mMarkerManager;
    }

    public void setAnimation(boolean animate) {
        this.mRenderer.setAnimation(animate);
    }

    public MaptexClusterRenderer<T> getRenderer() {
        return this.mRenderer;
    }

    public void setRenderer(MaptexClusterRenderer<T> view) {
        this.mRenderer.setOnClusterClickListener((OnClusterClickListener) null);
        this.mRenderer.setOnClusterItemClickListener((OnClusterItemClickListener) null);
        this.mClusterMarkers.clear();
        this.mMarkers.clear();
        this.mRenderer.onRemove();
        this.mRenderer = view;
        this.mRenderer.onAdd();
        this.mRenderer.setOnClusterClickListener(this.mOnClusterClickListener);
        this.mRenderer.setOnClusterInfoWindowClickListener(this.mOnClusterInfoWindowClickListener);
        this.mRenderer.setOnClusterItemClickListener(this.mOnClusterItemClickListener);
        this.mRenderer.setOnClusterItemInfoWindowClickListener(this.mOnClusterItemInfoWindowClickListener);
        this.cluster();
    }

    public MaptexAlgorithm<T> getAlgorithm() {
        return this.mAlgorithm;
    }

    public void setAlgorithm(MaptexAlgorithm<T> algorithm) {
        this.mAlgorithmLock.writeLock().lock();

        try {
            if (this.mAlgorithm != null) {
                algorithm.addItems(this.mAlgorithm.getItems());
            }

            this.mAlgorithm = new MaptexPreCachingAlgorithmDecorator(algorithm);
        } finally {
            this.mAlgorithmLock.writeLock().unlock();
        }

        this.cluster();
    }

    public void clearItems() {
        this.mAlgorithmLock.writeLock().lock();

        try {
            this.mAlgorithm.clearItems();
        } finally {
            this.mAlgorithmLock.writeLock().unlock();
        }

    }

    public void addItems(java.util.Collection<T> items) {
        this.mAlgorithmLock.writeLock().lock();

        try {
            this.mAlgorithm.addItems(items);
        } finally {
            this.mAlgorithmLock.writeLock().unlock();
        }

    }

    public void addItem(T myItem) {
        this.mAlgorithmLock.writeLock().lock();

        try {
            this.mAlgorithm.addItem(myItem);
        } finally {
            this.mAlgorithmLock.writeLock().unlock();
        }

    }

    public void removeItem(T item) {
        this.mAlgorithmLock.writeLock().lock();

        try {
            this.mAlgorithm.removeItem(item);
        } finally {
            this.mAlgorithmLock.writeLock().unlock();
        }

    }

    public void cluster() {
        this.mClusterTaskLock.writeLock().lock();

        try {
            this.mClusterTask.cancel(true);
            this.mClusterTask = new ClusterTask();
            if (Build.VERSION.SDK_INT < 11) {
                this.mClusterTask.execute(new Float[]{Float.valueOf(this.mMap.getCameraPosition().zoom)});
            } else {
                this.mClusterTask.executeOnExecutor(AsyncTask.THREAD_POOL_EXECUTOR, new Float[]{Float.valueOf(this.mMap.getCameraPosition().zoom)});
            }
        } finally {
            this.mClusterTaskLock.writeLock().unlock();
        }

    }

    public void onCameraIdle() {
        if (this.mRenderer instanceof MaptexMap.OnCameraIdleListener) {
            ((MaptexMap.OnCameraIdleListener) this.mRenderer).onCameraIdle();
        }

        MaptexCameraPosition position = this.mMap.getCameraPosition();
        if (this.mPreviousCameraPosition == null || this.mPreviousCameraPosition.zoom != position.zoom) {
            this.mPreviousCameraPosition = this.mMap.getCameraPosition();
            this.cluster();
        }
    }

    public boolean onMarkerClick(MaptexMarker marker) {
        return this.getMarkerManager().onMarkerClick(marker);
    }

    public void onInfoWindowClick(MaptexMarker marker) {
        this.getMarkerManager().onInfoWindowClick(marker);
    }

    public void setOnClusterClickListener(OnClusterClickListener<T> listener) {
        this.mOnClusterClickListener = listener;
        this.mRenderer.setOnClusterClickListener(listener);
    }

    public void setOnClusterInfoWindowClickListener(OnClusterInfoWindowClickListener<T> listener) {
        this.mOnClusterInfoWindowClickListener = listener;
        this.mRenderer.setOnClusterInfoWindowClickListener(listener);
    }

    public void setOnClusterItemClickListener(OnClusterItemClickListener<T> listener) {
        this.mOnClusterItemClickListener = listener;
        this.mRenderer.setOnClusterItemClickListener(listener);
    }

    public void setOnClusterItemInfoWindowClickListener(OnClusterItemInfoWindowClickListener<T> listener) {
        this.mOnClusterItemInfoWindowClickListener = listener;
        this.mRenderer.setOnClusterItemInfoWindowClickListener(listener);
    }

    public interface OnClusterItemInfoWindowClickListener<T extends MaptexClusterItem> {
        void onClusterItemInfoWindowClick(T var1);
    }

    public interface OnClusterItemClickListener<T extends MaptexClusterItem> {
        boolean onClusterItemClick(T var1);
    }

    public interface OnClusterInfoWindowClickListener<T extends MaptexClusterItem> {
        void onClusterInfoWindowClick(MaptexCluster<T> var1);
    }

    public interface OnClusterClickListener<T extends MaptexClusterItem> {
        boolean onClusterClick(MaptexCluster<T> var1);
    }

    private class ClusterTask extends AsyncTask<Float, Void, Set<? extends MaptexCluster<T>>> {
        private ClusterTask() {
        }

        protected Set<? extends MaptexCluster<T>> doInBackground(Float... zoom) {
            MaptexClusterManager.this.mAlgorithmLock.readLock().lock();

            Set var2;
            try {
                var2 = MaptexClusterManager.this.mAlgorithm.getClusters((double) zoom[0].floatValue());
            } finally {
                MaptexClusterManager.this.mAlgorithmLock.readLock().unlock();
            }

            return var2;
        }

        protected void onPostExecute(Set<? extends MaptexCluster<T>> clusters) {
            MaptexClusterManager.this.mRenderer.onClustersChanged(clusters);
        }
    }
}