package ir.map.sdk_map;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;

import ir.map.sdk_common.MaptexLatLng;
import ir.map.sdk_map.wrapper.MaptexBusStation;
import ir.map.sdk_map.wrapper.MaptexMetroLine;
import ir.map.sdk_map.wrapper.MaptexMetroStation;


public class MaptexPlanUtils {

    private static MaptexLatLng ISFAHAN_CENTER = new MaptexLatLng(32.666330, 51.677370);
    private static MaptexLatLng MASHHAD_CENTER = new MaptexLatLng(36.316266, 59.572624);
    private static MaptexLatLng SHIRAZ_CENTER = new MaptexLatLng(29.601036, 52.524347);
    private static MaptexLatLng TABRIZ_CENTER = new MaptexLatLng(38.069565, 46.313403);
    private static MaptexLatLng TEHRAN_CENTER = new MaptexLatLng(35.695086, 51.394086);
    private static int nearestCityId = 0;
    private ArrayList<MaptexLatLng> centersOfCitiesList = new ArrayList<>();

    public MaptexPlanUtils() {
        initCentersOfCitiesList();
    }

    public ArrayList<MaptexLatLng> getEvenOddPoses() {
        return getMapPoses("even-odd");
    }

    public ArrayList<MaptexLatLng> getTrafficPoses() {
        return getMapPoses("traffic");
    }

    private ArrayList<MaptexLatLng> getMapPoses(String type) {
        ArrayList<MaptexLatLng> latLngs = new ArrayList<>();
        String json = loadJSONFromAsset(type);
        if (json != null) {
            try {
                JSONArray jsonArray = new JSONArray(json);
                for (int i = 0; i < jsonArray.length(); i++) {
                    latLngs.add(new MaptexLatLng(jsonArray.getJSONArray(i).getDouble(1), jsonArray.getJSONArray(i).getDouble(0)));
                }

            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
        return latLngs;
    }

    public ArrayList<MaptexMetroLine> getBusLines() {
        ArrayList<MaptexMetroLine> busLines = new ArrayList<>();
        String json = loadJSONFromAsset("bus_lines");
        try {
            JSONObject jsonObject = new JSONObject(json);
            JSONArray features = jsonObject.getJSONArray("features");
            for (int i = 0; i < features.length(); i++) {
                MaptexMetroLine line = new MaptexMetroLine();
                JSONObject properties = features.getJSONObject(i).getJSONObject("properties");
                line.name = properties.getString("name");
                line.number = properties.getInt("linenumber");
                JSONObject geometry = features.getJSONObject(i).getJSONObject("geometry");
                if ((geometry.getString("type")).equals("LineString")) {
                    JSONArray coordinates = geometry.getJSONArray("coordinates");
                    ArrayList<MaptexLatLng> latLngs = new ArrayList<>();
                    for (int j = 0; j < coordinates.length(); j++) {
                        MaptexLatLng latLng = new MaptexLatLng(coordinates.getJSONArray(j).getDouble(1),
                                coordinates.getJSONArray(j).getDouble(0));
                        latLngs.add(latLng);
                    }
                    line.coordinates = latLngs;
                }
                busLines.add(line);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return busLines;
    }

    public ArrayList<MaptexBusStation> getBusStations() {
        ArrayList<MaptexBusStation> stations = new ArrayList<>();
        String json = loadJSONFromAsset("bus_stations");
        try {
            JSONObject jsonObject = new JSONObject(json);
            JSONArray features = jsonObject.getJSONArray("features");
            for (int i = 0; i < features.length(); i++) {
                JSONObject properties = features.getJSONObject(i).getJSONObject("properties");
                String name = properties.getString("name");
                MaptexBusStation station = new MaptexBusStation();
                station.name = name;
                JSONObject geometry = features.getJSONObject(i).getJSONObject("geometry");
                if ((geometry.getString("type")).equals("Point")) {
                    station.latLng = new MaptexLatLng(
                            geometry.getJSONArray("coordinates").getDouble(1),
                            geometry.getJSONArray("coordinates").getDouble(0));
                }
                stations.add(station);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return stations;
    }

    public ArrayList<MaptexLatLng> getBusStationsPoints() {
        ArrayList<MaptexLatLng> stations = new ArrayList<>();
        String json = loadJSONFromAsset("bus_stations");
        try {
            JSONObject jsonObject = new JSONObject(json);
            JSONArray features = jsonObject.getJSONArray("features");
            for (int i = 0; i < features.length(); i++) {
                JSONObject geometry = features.getJSONObject(i).getJSONObject("geometry");
                if ((geometry.getString("type")).equals("Point")) {
                    stations.add(new MaptexLatLng(geometry.getJSONArray("coordinates").getDouble(1), geometry.getJSONArray("coordinates").getDouble(0)));
                }
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return stations;
    }

    public ArrayList<MaptexMetroLine> getMetroLines(MaptexLatLng centerOfMaptexLatLng) {
        ArrayList<MaptexMetroLine> metroLines = new ArrayList<>();

        try {
            String json = getMetroLinesOfNearestCity(centerOfMaptexLatLng);
            JSONObject jsonObject = new JSONObject(json);
            JSONArray features = jsonObject.getJSONArray("features");
            for (int i = 0; i < features.length(); i++) {
                MaptexMetroLine line = new MaptexMetroLine();
                JSONObject properties = features.getJSONObject(i).getJSONObject("properties");
                line.color = properties.getString("linecolor");
                line.name = properties.getString("name");
                line.number = properties.getInt("linenumber");
                JSONObject geometry = features.getJSONObject(i).getJSONObject("geometry");
                if ((geometry.getString("type")).equals("LineString")) {
                    JSONArray coordinates = geometry.getJSONArray("coordinates");
                    ArrayList<MaptexLatLng> latLngs = new ArrayList<>();
                    for (int j = 0; j < coordinates.length(); j++) {
                        MaptexLatLng latLng = new MaptexLatLng(coordinates.getJSONArray(j).getDouble(1),
                                coordinates.getJSONArray(j).getDouble(0));
                        latLngs.add(latLng);
                    }
                    line.coordinates = latLngs;
                }
                metroLines.add(line);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return metroLines;
    }

    private void initCentersOfCitiesList() {
        centersOfCitiesList.add(ISFAHAN_CENTER);
        centersOfCitiesList.add(MASHHAD_CENTER);
        centersOfCitiesList.add(SHIRAZ_CENTER);
        centersOfCitiesList.add(TABRIZ_CENTER);
        centersOfCitiesList.add(TEHRAN_CENTER);
    }

    public boolean checkChangeNearestCity(MaptexLatLng latLng) {
        int lastNearestCityId = nearestCityId;
        int newNearestCityId = -1;
        Double minDistance = Double.MAX_VALUE;
        for (int i = 0; i < 5; i++) {
            if (MaptexSphericalUtil.computeDistanceBetween(latLng, centersOfCitiesList.get(i)) < minDistance) {
                minDistance = MaptexSphericalUtil.computeDistanceBetween(latLng, centersOfCitiesList.get(i));
                newNearestCityId = i;
            }
        }

        if (newNearestCityId != lastNearestCityId)
            return true;
        else
            return false;
    }

    private int getNearestCity(MaptexLatLng latLng) {
        int nearestCity = -1;
        Double minDistance = Double.MAX_VALUE;
        for (int i = 0; i < 5; i++) {
            if (MaptexSphericalUtil.computeDistanceBetween(latLng, centersOfCitiesList.get(i)) < minDistance) {
                minDistance = MaptexSphericalUtil.computeDistanceBetween(latLng, centersOfCitiesList.get(i));
                nearestCity = i;
            }
        }
        nearestCityId = nearestCity;
        return nearestCity;
    }

    public ArrayList<MaptexMetroStation> getMetroStations(MaptexLatLng centerOfMaptexLatLng) {
        ArrayList<MaptexMetroStation> stations = new ArrayList<>();

        try {
            String json = getMetroStationsOfNearestCity(centerOfMaptexLatLng);
            JSONObject jsonObject = new JSONObject(json);
            JSONArray features = jsonObject.getJSONArray("features");
            for (int i = 0; i < features.length(); i++) {
                JSONObject properties = features.getJSONObject(i).getJSONObject("properties");
                String name = properties.getString("name");
                MaptexMetroStation station = new MaptexMetroStation();
                station.name = name;
                JSONObject geometry = features.getJSONObject(i).getJSONObject("geometry");
                if ((geometry.getString("type")).equals("Point")) {
                    station.latLng = new MaptexLatLng(
                            geometry.getJSONArray("coordinates").getDouble(1),
                            geometry.getJSONArray("coordinates").getDouble(0));
                }
                stations.add(station);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return stations;
    }

    private String getMetroStationsOfNearestCity(MaptexLatLng centerOfMaptexLatLng) {
        switch (getNearestCity(centerOfMaptexLatLng)) {
            case 0: //MASHHAD
                return loadJSONFromAsset("metro_stations_isfahan");
            case 1: //ISFAHAN
                return loadJSONFromAsset("metro_stations_mashhad");
            case 2: //SHIRAZ
                return loadJSONFromAsset("metro_stations_shiraz");
            case 3: //TABRIZ
                return loadJSONFromAsset("metro_stations_tabriz");
            case 4: //TEHRAN
                return loadJSONFromAsset("metro_stations_tehran");
            default:
                return null;
        }
    }

    private String getMetroLinesOfNearestCity(MaptexLatLng centerOfMaptexLatLng) {
        switch (getNearestCity(centerOfMaptexLatLng)) {
            case 0: //MASHHAD
                return loadJSONFromAsset("metro_lines_isfahan");
            case 1: //ISFAHAN
                return loadJSONFromAsset("metro_lines_mashhad");
            case 2: //SHIRAZ
                return loadJSONFromAsset("metro_lines_shiraz");
            case 3: //TABRIZ
                return loadJSONFromAsset("metro_lines_tabriz");
            case 4: //TEHRAN
                return loadJSONFromAsset("metro_lines_tehran");
            default:
                return null;
        }
    }

    public ArrayList<MaptexLatLng> getMetroStationsPoints(MaptexLatLng centerOfMaptexLatLng) {
        ArrayList<MaptexLatLng> stations = new ArrayList<>();

        try {
            String json = getMetroStationsOfNearestCity(centerOfMaptexLatLng);
            JSONObject jsonObject = new JSONObject(json);
            JSONArray features = jsonObject.getJSONArray("features");
            for (int i = 0; i < features.length(); i++) {
                JSONObject geometry = features.getJSONObject(i).getJSONObject("geometry");
                if ((geometry.getString("type")).equals("Point")) {
                    stations.add(new MaptexLatLng(geometry.getJSONArray("coordinates").getDouble(1), geometry.getJSONArray("coordinates").getDouble(0)));
                }
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return stations;
    }

    public ArrayList<MaptexLatLng> getMapPosSafePlace() {
        ArrayList<MaptexLatLng> latLngs = new ArrayList<>();
        String json = loadJSONFromAsset("earthqueke");
        try {
            JSONObject jsonObject = new JSONObject(json);
            JSONArray features = jsonObject.getJSONArray("features");
            for (int i = 0; i < features.length(); i++) {
                JSONObject geometry = features.getJSONObject(i).getJSONObject("geometry");
                if ((geometry.getString("type")).equals("Point")) {
                    latLngs.add(new MaptexLatLng(geometry.getJSONArray("coordinates").getDouble(1), geometry.getJSONArray("coordinates").getDouble(0)));
                }
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return latLngs;
    }

    private String loadJSONFromAsset(String type) {
        String json = null;
        try {
            InputStream is = null;
            switch (type) {
                case "traffic":
                    is = MapSDK.getContext().getAssets().open("traffic-json.json");
                    break;
                case "even-odd":
                    is = MapSDK.getContext().getAssets().open("even-odd.json");
                    break;
                case "earthqueke":
                    is = MapSDK.getContext().getAssets().open("earthqueke.geojson");
                    break;
                case "metro_stations_isfahan":
                    is = MapSDK.getContext().getAssets().open("metro_stations_isfahan.geojson");
                    break;
                case "metro_lines_isfahan":
                    is = MapSDK.getContext().getAssets().open("metro_lines_isfahan.geojson");
                    break;
                case "metro_stations_mashhad":
                    is = MapSDK.getContext().getAssets().open("metro_stations_mashhad.geojson");
                    break;
                case "metro_lines_mashhad":
                    is = MapSDK.getContext().getAssets().open("metro_lines_mashhad.geojson");
                    break;
                case "metro_stations_shiraz":
                    is = MapSDK.getContext().getAssets().open("metro_stations_shiraz.geojson");
                    break;
                case "metro_lines_shiraz":
                    is = MapSDK.getContext().getAssets().open("metro_lines_shiraz.geojson");
                    break;
                case "metro_stations_tabriz":
                    is = MapSDK.getContext().getAssets().open("metro_stations_tabriz.geojson");
                    break;
                case "metro_lines_tabriz":
                    is = MapSDK.getContext().getAssets().open("metro_lines_tabriz.geojson");
                    break;
                case "metro_stations_tehran":
                    is = MapSDK.getContext().getAssets().open("metro_stations_tehran.geojson");
                    break;
                case "metro_lines_tehran":
                    is = MapSDK.getContext().getAssets().open("metro_lines_tehran.geojson");
                    break;
                case "bus_lines":
                    is = MapSDK.getContext().getAssets().open("bus_lines.geojson");
                    break;
                case "bus_stations":
                    is = MapSDK.getContext().getAssets().open("bus_stations.geojson");
                    break;
            }
            if (is != null) {
                int size = is.available();
                byte[] buffer = new byte[size];
                is.read(buffer);
                is.close();
                json = new String(buffer, "UTF-8");
            }
        } catch (IOException ex) {
            ex.printStackTrace();
            return null;
        }

        return json;
    }

}
