/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

/** @module vertx-web-api-contract-js/router_factory */
var utils = require('vertx-js/util/utils');
var Router = require('vertx-web-js/router');
var RoutingContext = require('vertx-web-js/routing_context');

var io = Packages.io;
var JsonObject = io.vertx.core.json.JsonObject;
var JRouterFactory = Java.type('io.vertx.ext.web.api.contract.RouterFactory');
var RouterFactoryOptions = Java.type('io.vertx.ext.web.api.contract.RouterFactoryOptions');

/**
 Main interface for Design Driven Router factory
 @class
*/
var RouterFactory = function(j_val, j_arg_0) {

  var j_routerFactory = j_val;
  var that = this;
  var j_Specification = typeof j_arg_0 !== 'undefined' ? j_arg_0 : utils.unknown_jtype;

  /**
   Mount to paths that have to follow a security schema a security handler

   @public
   @param securitySchemaName {string} 
   @param handler {function} 
   @return {RouterFactory} 
   */
  this.addSecurityHandler = function(securitySchemaName, handler) {
    var __args = arguments;
    if (__args.length === 2 && typeof __args[0] === 'string' && typeof __args[1] === 'function') {
      j_routerFactory["addSecurityHandler(java.lang.String,io.vertx.core.Handler)"](securitySchemaName, function(jVal) {
      handler(utils.convReturnVertxGen(RoutingContext, jVal));
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add an handler to a path with a method. If combination path/method is not available in
   specification, it will throw a {@link RouterFactoryException}. Deprecated in favour of
   operation id

   @public
   @param method {Object} 
   @param path {string} 
   @param handler {function} 
   @return {RouterFactory} 
   */
  this.addHandler = function(method, path, handler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_routerFactory["addHandler(io.vertx.core.http.HttpMethod,java.lang.String,io.vertx.core.Handler)"](io.vertx.core.http.HttpMethod.valueOf(method), path, function(jVal) {
      handler(utils.convReturnVertxGen(RoutingContext, jVal));
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Add a failure handler to a path with a method. If combination path/method is not available in
   specification, it will throw a {@link RouterFactoryException}. Deprecated in favour of
   operation id

   @public
   @param method {Object} 
   @param path {string} 
   @param failureHandler {function} 
   @return {RouterFactory} 
   */
  this.addFailureHandler = function(method, path, failureHandler) {
    var __args = arguments;
    if (__args.length === 3 && typeof __args[0] === 'string' && typeof __args[1] === 'string' && typeof __args[2] === 'function') {
      j_routerFactory["addFailureHandler(io.vertx.core.http.HttpMethod,java.lang.String,io.vertx.core.Handler)"](io.vertx.core.http.HttpMethod.valueOf(method), path, function(jVal) {
      failureHandler(utils.convReturnVertxGen(RoutingContext, jVal));
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Override options

   @public
   @param options {Object} 
   @return {RouterFactory} 
   */
  this.setOptions = function(options) {
    var __args = arguments;
    if (__args.length === 1 && (typeof __args[0] === 'object' && __args[0] != null)) {
      j_routerFactory["setOptions(io.vertx.ext.web.api.contract.RouterFactoryOptions)"](options != null ? new RouterFactoryOptions(new JsonObject(Java.asJSONCompatible(options))) : null);
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Get options of router factory. For more info <a href="../../dataobjects.html#RouterFactoryOptions">RouterFactoryOptions</a>

   @public

   @return {Object} 
   */
  this.getOptions = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnDataObject(j_routerFactory["getOptions()"]());
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Deprecated. Instantiate <a href="../../dataobjects.html#RouterFactoryOptions">RouterFactoryOptions</a>
   and load it using {@link RouterFactory#setOptions}

   @public
   @param handler {function} 
   @return {RouterFactory} 
   */
  this.setValidationFailureHandler = function(handler) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] === 'function') {
      j_routerFactory["setValidationFailureHandler(io.vertx.core.Handler)"](function(jVal) {
      handler(utils.convReturnVertxGen(RoutingContext, jVal));
    });
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Deprecated. Instantiate <a href="../../dataobjects.html#RouterFactoryOptions">RouterFactoryOptions</a>
   and load it using {@link RouterFactory#setOptions}

   @public
   @param enable {boolean} 
   @return {RouterFactory} 
   */
  this.enableValidationFailureHandler = function(enable) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='boolean') {
      j_routerFactory["enableValidationFailureHandler(boolean)"](enable);
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Deprecated. Instantiate <a href="../../dataobjects.html#RouterFactoryOptions">RouterFactoryOptions</a>
   and load it using {@link RouterFactory#setOptions}

   @public
   @param enable {boolean} 
   @return {RouterFactory} 
   */
  this.mountOperationsWithoutHandlers = function(enable) {
    var __args = arguments;
    if (__args.length === 1 && typeof __args[0] ==='boolean') {
      j_routerFactory["mountOperationsWithoutHandlers(boolean)"](enable);
      return that;
    } else throw new TypeError('function invoked with invalid arguments');
  };

  /**
   Construct a new router based on spec. It will fail if you are trying to mount a spec with security schemes
   without assigned handlers<br/>
   <b>Note:</b> Router is constructed in this function, so it will be respected the path definition ordering.

   @public

   @return {Router} 
   */
  this.getRouter = function() {
    var __args = arguments;
    if (__args.length === 0) {
      return utils.convReturnVertxGen(Router, j_routerFactory["getRouter()"]());
    } else throw new TypeError('function invoked with invalid arguments');
  };

  // A reference to the underlying Java delegate
  // NOTE! This is an internal API and must not be used in user code.
  // If you rely on this property your code is likely to break if we change it / remove it without warning.
  this._jdel = j_routerFactory;
};

RouterFactory._jclass = utils.getJavaClass("io.vertx.ext.web.api.contract.RouterFactory");
RouterFactory._jtype = {
  accept: function(obj) {
    return RouterFactory._jclass.isInstance(obj._jdel);
  },
  wrap: function(jdel) {
    var obj = Object.create(RouterFactory.prototype, {});
    RouterFactory.apply(obj, arguments);
    return obj;
  },
  unwrap: function(obj) {
    return obj._jdel;
  }
};
RouterFactory._create = function(jdel) {
  var obj = Object.create(RouterFactory.prototype, {});
  RouterFactory.apply(obj, arguments);
  return obj;
}
module.exports = RouterFactory;