/*
 * Decompiled with CFR 0.152.
 */
package io.trino.plugin.password.file;

import at.favre.lib.crypto.bcrypt.BCrypt;
import at.favre.lib.crypto.bcrypt.IllegalBCryptFormatException;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.io.BaseEncoding;
import io.trino.plugin.password.file.HashedPasswordException;
import io.trino.plugin.password.file.HashingAlgorithm;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.List;
import java.util.Objects;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;

public final class EncryptionUtil {
    private static final int BCRYPT_MIN_COST = 8;
    private static final int PBKDF2_MIN_ITERATIONS = 1000;

    private EncryptionUtil() {
    }

    public static int getBCryptCost(String password) {
        try {
            return BCrypt.Version.VERSION_2A.parser.parse((byte[])password.getBytes((Charset)StandardCharsets.UTF_8)).cost;
        }
        catch (IllegalBCryptFormatException e) {
            throw new HashedPasswordException("Invalid BCrypt password", e);
        }
    }

    public static int getPBKDF2Iterations(String password) {
        return PBKDF2Password.fromString(password).iterations();
    }

    public static boolean doesBCryptPasswordMatch(String inputPassword, String hashedPassword) {
        return BCrypt.verifyer().verify((char[])inputPassword.toCharArray(), (CharSequence)hashedPassword).verified;
    }

    public static boolean doesPBKDF2PasswordMatch(String inputPassword, String hashedPassword) {
        PBKDF2Password password = PBKDF2Password.fromString(hashedPassword);
        try {
            PBEKeySpec spec = new PBEKeySpec(inputPassword.toCharArray(), password.salt(), password.iterations(), password.hash().length * 8);
            SecretKeyFactory key = SecretKeyFactory.getInstance("PBKDF2WithHmacSHA1");
            byte[] inputHash = key.generateSecret(spec).getEncoded();
            if (password.hash().length != inputHash.length) {
                throw new HashedPasswordException("PBKDF2 password input is malformed");
            }
            return MessageDigest.isEqual(password.hash(), inputHash);
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new HashedPasswordException("Invalid PBKDF2 password", e);
        }
    }

    public static HashingAlgorithm getHashingAlgorithm(String password) {
        if (password.startsWith("$2y")) {
            if (EncryptionUtil.getBCryptCost(password) < 8) {
                throw new HashedPasswordException("Minimum cost of BCrypt password must be 8");
            }
            return HashingAlgorithm.BCRYPT;
        }
        if (password.contains(":")) {
            if (EncryptionUtil.getPBKDF2Iterations(password) < 1000) {
                throw new HashedPasswordException("Minimum iterations of PBKDF2 password must be 1000");
            }
            return HashingAlgorithm.PBKDF2;
        }
        throw new HashedPasswordException("Password hashing algorithm cannot be determined");
    }

    private static class PBKDF2Password {
        private final int iterations;
        private final byte[] salt;
        private final byte[] hash;

        private PBKDF2Password(int iterations, byte[] salt, byte[] hash) {
            this.iterations = iterations;
            this.salt = Objects.requireNonNull(salt, "salt is null");
            this.hash = Objects.requireNonNull(hash, "hash is null");
        }

        public int iterations() {
            return this.iterations;
        }

        public byte[] salt() {
            return this.salt;
        }

        public byte[] hash() {
            return this.hash;
        }

        public static PBKDF2Password fromString(String password) {
            try {
                List parts = Splitter.on((String)":").splitToList((CharSequence)password);
                Preconditions.checkArgument((parts.size() == 3 ? 1 : 0) != 0, (Object)"wrong part count");
                int iterations = Integer.parseInt((String)parts.get(0));
                byte[] salt = BaseEncoding.base16().lowerCase().decode((CharSequence)parts.get(1));
                byte[] hash = BaseEncoding.base16().lowerCase().decode((CharSequence)parts.get(2));
                return new PBKDF2Password(iterations, salt, hash);
            }
            catch (IllegalArgumentException e) {
                throw new HashedPasswordException("Invalid PBKDF2 password");
            }
        }
    }
}

